package com.atlassian.adf.model.mark;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.ex.AdfException;
import com.atlassian.adf.model.mark.type.FormattedTextMark;
import com.atlassian.adf.model.node.Paragraph;
import com.atlassian.adf.model.node.Text;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.util.EnumParser;
import com.atlassian.adf.util.Factory;

import javax.annotation.concurrent.Immutable;
import java.util.Map;
import java.util.function.Supplier;

import static com.atlassian.adf.model.node.Text.text;
import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.checkType;
import static com.atlassian.adf.util.ParserSupport.getAttrOrThrow;

/**
 * The {@code subsup} mark sets <sup>superscript</sup> or <sub>subscript</sub> styling.
 * This mark applies to {@link Text} nodes and can only be combined with other
 * {@link FormattedTextMark formatted-text} marks.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link Paragraph#p(InlineContent[]) p}(
 *         {@link Text#text(String) text}("Hello "),
 *         {@link Text#text(String) text}("world").{@link Text#sup() sup}(),
 *         {@link Text#text(String) text}("!")
 * );
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "paragraph",
 *     "content": [
 *       {
 *         "type": "text",
 *         "text": "Hello "
 *       },
 *       {
 *         "type": "text",
 *         "text": "world",
 *         "marks": [
 *           {
 *             "type": "subsup",
 *             "attrs": {
 *               "type": "sup"
 *             }
 *           }
 *         ]
 *       },
 *       {
 *         "type": "text",
 *         "text": "!"
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <p>Hello <sup>world</sup>!</p>
 * </div>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/marks/subsup/">Mark - subsup</a>
 */
@Documentation(state = Documentation.State.REVIEWED, date = "2023-07-26")
@Immutable
public class SubSup
        extends AbstractMark
        implements FormattedTextMark {

    private static final SubSup SUB = new SubSup(Type.SUB);
    private static final SubSup SUP = new SubSup(Type.SUP);

    static final Factory<SubSup> FACTORY = new Factory<>(Mark.Type.SUBSUP, SubSup.class, SubSup::parse);

    private final Type type;

    private SubSup(Type type) {
        this.type = type;
    }

    /**
     * @return a {@code subsup} mark for <sub>subscript</sub> text
     */
    public static SubSup sub() {
        return SUB;
    }

    /**
     * Convenience method for constructing a {@code text} node with a {@code sub} mark.
     *
     * @param text the text to mark
     * @return the resulting marked text node
     */
    public static Text sub(String text) {
        return text(text).sub();
    }

    /**
     * Convenience method for applying a {@code sub} mark to an existing {@code text} node.
     *
     * @param text the text to mark
     * @return the resulting marked text node
     * @throws AdfException if it isn't possible to apply that mark to the existing {@code text} node
     */
    public static Text sub(Text text) {
        return text.sub();
    }

    /**
     * @return a {@code subsup} mark for <sup>superscript</sup> text
     */
    public static SubSup sup() {
        return SUP;
    }

    /**
     * Convenience method for constructing a {@code text} node with a {@code sup} mark.
     *
     * @param text the text to mark
     * @return the resulting marked text node
     */
    public static Text sup(String text) {
        return text(text).sup();
    }

    /**
     * Convenience method for applying a {@code sup} mark to an existing {@code text} node.
     *
     * @param text the text to mark
     * @return the resulting marked text node
     * @throws AdfException if it isn't possible to apply that mark to the existing {@code text} node
     */
    public static Text sup(Text text) {
        return text.sup();
    }

    @Override
    public SubSup copy() {
        return this;
    }

    public boolean isSub() {
        return this == SUB;
    }

    public boolean isSup() {
        return this == SUP;
    }

    public Type type() {
        return type;
    }

    @Override
    public String elementType() {
        return Mark.Type.SUBSUP;
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .add(Key.ATTRS, map(Attr.TYPE, type.type()));
    }

    private static SubSup parse(Map<String, ?> map) {
        checkType(map, Mark.Type.SUBSUP);
        String type = getAttrOrThrow(map, Attr.TYPE);
        return Type.PARSER.parse(type).toSubSup();
    }

    @Override
    public String toString() {
        return elementType() + '[' + type.type() + ']';
    }


    public enum Type {
        SUB("sub", SubSup::sub),
        SUP("sup", SubSup::sup);

        static final EnumParser<Type> PARSER = new EnumParser<>(Type.class, Type::type);

        private final String type;
        private final Supplier<SubSup> toSubSup;

        Type(String type, Supplier<SubSup> toSubSup) {
            this.type = type;
            this.toSubSup = toSubSup;
        }

        public String type() {
            return type;
        }

        SubSup toSubSup() {
            return toSubSup.get();
        }
    }
}
