package com.atlassian.adf.model.mark;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.ex.AdfException;
import com.atlassian.adf.model.mark.type.FormattedTextMark;
import com.atlassian.adf.model.node.Paragraph;
import com.atlassian.adf.model.node.Text;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.util.Factory;

import javax.annotation.concurrent.Immutable;
import java.util.Map;

import static com.atlassian.adf.model.node.Text.text;
import static com.atlassian.adf.util.ParserSupport.checkType;

/**
 * The {@code em} mark sets <strong>strong</strong> styling.
 * This mark applies to {@link Text} nodes and can only be combined with other
 * {@link FormattedTextMark formatted-text} marks.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link Paragraph#p(InlineContent[]) p}(
 *         {@link Text#text(String) text}("Hello "),
 *         {@link Text#text(String) text}("world").{@link Text#strong() strong}(),
 *         {@link Text#text(String) text}("!")
 * );
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "paragraph",
 *     "content": [
 *       {
 *         "type": "text",
 *         "text": "Hello "
 *       },
 *       {
 *         "type": "text",
 *         "text": "world",
 *         "marks": [
 *           {
 *             "type": "strong"
 *           }
 *         ]
 *       },
 *       {
 *         "type": "text",
 *         "text": "!"
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <p>Hello <strong>world</strong>!</p>
 * </div>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/marks/strong/">Mark - strong</a>
 */
@Documentation(state = Documentation.State.REVIEWED, date = "2023-07-26")
@Immutable
public class Strong
        extends AbstractMark
        implements FormattedTextMark {

    private static final Strong STRONG = new Strong();
    static final Factory<Strong> FACTORY = new Factory<>(Type.STRONG, Strong.class, Strong::parse);

    private Strong() {
        // singleton
    }

    /**
     * @return a {@code strong} mark
     */
    public static Strong strong() {
        return STRONG;
    }

    /**
     * Convenience method for constructing a {@code text} node with a {@code strong} mark.
     *
     * @param text the text to mark
     * @return the resulting marked text node
     */
    public static Text strong(String text) {
        return text(text).strong();
    }

    /**
     * Convenience method for applying a {@code strong} mark to an existing {@code text} node.
     *
     * @param text the text to mark
     * @return the resulting marked text node
     * @throws AdfException if it isn't possible to apply that mark to the existing {@code text} node
     */
    public static Text strong(Text text) {
        return text.strong();
    }

    @Override
    public Strong copy() {
        return this;
    }

    @Override
    public String elementType() {
        return Type.STRONG;
    }

    private static Strong parse(Map<String, ?> map) {
        checkType(map, Type.STRONG);
        return strong();
    }
}
