package com.atlassian.adf.model.mark;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.ex.AdfException;
import com.atlassian.adf.model.mark.type.FormattedTextMark;
import com.atlassian.adf.model.node.Paragraph;
import com.atlassian.adf.model.node.Text;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.util.Factory;

import javax.annotation.concurrent.Immutable;
import java.util.Map;

import static com.atlassian.adf.model.node.Text.text;
import static com.atlassian.adf.util.ParserSupport.checkType;

/**
 * The {@code em} mark sets <em>italic</em> (<strong>em</strong>phasis) styling.
 * This mark applies to {@link Text} nodes and can only be combined with other
 * {@link FormattedTextMark formatted-text} marks.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link Paragraph#p(InlineContent[]) p}(
 *         {@link Text#text(String) text}("Hello "),
 *         {@link Text#text(String) text}("world").{@link Text#em() em}(),
 *         {@link Text#text(String) text}("!")
 * );
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "paragraph",
 *     "content": [
 *       {
 *         "type": "text",
 *         "text": "Hello "
 *       },
 *       {
 *         "type": "text",
 *         "text": "world",
 *         "marks": [
 *           {
 *             "type": "em"
 *           }
 *         ]
 *       },
 *       {
 *         "type": "text",
 *         "text": "!"
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <p>Hello <em>world</em>!</p>
 * </div>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/marks/em/">Mark - em</a>
 */
@Documentation(state = Documentation.State.REVIEWED, date = "2023-07-26")
@Immutable
public class Em
        extends AbstractMark
        implements FormattedTextMark {

    private static final Em EM = new Em();
    static final Factory<Em> FACTORY = new Factory<>(Type.EM, Em.class, Em::parse);

    private Em() {
        // singleton
    }

    /**
     * @return an {@code em} mark
     */
    public static Em em() {
        return EM;
    }

    /**
     * Convenience method for constructing a {@code text} node with an {@code em} mark.
     *
     * @param text the text to mark
     * @return the resulting marked text node
     */
    public static Text em(String text) {
        return text(text).em();
    }

    /**
     * Convenience method for applying an {@code em} mark to an existing {@code text} node.
     *
     * @param text the text to mark
     * @return the resulting marked text node
     * @throws AdfException if it isn't possible to apply that mark to the existing {@code text} node
     */
    public static Text em(Text text) {
        return text.em();
    }

    @Override
    public Em copy() {
        return this;
    }

    @Override
    public String elementType() {
        return Type.EM;
    }

    private static Em parse(Map<String, ?> map) {
        checkType(map, Type.EM);
        return em();
    }
}
