package com.atlassian.adf.model.mark;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.ex.AdfException;
import com.atlassian.adf.model.mark.type.CodeTextMark;
import com.atlassian.adf.model.node.Paragraph;
import com.atlassian.adf.model.node.Text;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.util.Factory;

import javax.annotation.concurrent.Immutable;
import java.util.Map;

import static com.atlassian.adf.model.node.Text.text;
import static com.atlassian.adf.util.ParserSupport.checkType;

/**
 * The {@code code} mark sets code styling.
 * <p>
 * This mark applies to {@link Text} nodes and indicates that they should be displayed in a monospace font
 * and may have other style elements, like a different background color, to help them stand out.
 * <p>
 * The {@code code} mark can only be combined with other {@link CodeTextMark code marks}; that is,
 * it cannot be used with pure formatted-text marks like {@link Em}.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link Paragraph#p(InlineContent[]) p}(
 *         {@link Text#text(String) text}("Hello "),
 *         {@link Text#text(String) text}("world").{@link Text#code() code}(),
 *         {@link Text#text(String) text}("!")
 * );
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "paragraph",
 *     "content": [
 *       {
 *         "type": "text",
 *         "text": "Hello "
 *       },
 *       {
 *         "type": "text",
 *         "text": "world",
 *         "marks": [
 *           {
 *             "type": "code"
 *           }
 *         ]
 *       },
 *       {
 *         "type": "text",
 *         "text": "!"
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <p>Hello
 * <code style="background-color: rgba(9, 30, 66, 0.08)">world</code>!
 * </p>
 * </div>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/marks/code/">Mark - code</a>
 */
@Immutable
@Documentation(
        state = Documentation.State.INCOMPLETE,
        date = "2023-07-26",
        comment = "should mention 'annotation' marks"
)
public class Code
        extends AbstractMark
        implements CodeTextMark {

    private static final Code CODE = new Code();
    static Factory<Code> FACTORY = new Factory<>(Type.CODE, Code.class, Code::parse);

    private Code() {
        // singleton
    }

    @Override
    public Code copy() {
        return this;
    }

    /**
     * @return a {@code code} mark
     */
    public static Code code() {
        return CODE;
    }

    /**
     * Convenience method for constructing a {@code text} node with a {@code code} mark.
     *
     * @param text the text to mark
     * @return the resulting marked text node
     */
    public static Text code(String text) {
        return text(text).code();
    }

    /**
     * Convenience method for applying a {@code code} mark to an existing {@code text} node.
     *
     * @param text the text to mark
     * @return the resulting marked text node
     * @throws AdfException if it isn't possible to apply that mark to the existing {@code text} node
     */
    public static Text code(Text text) {
        return text.code();
    }

    @Override
    public String elementType() {
        return Type.CODE;
    }

    private static Code parse(Map<String, ?> map) {
        checkType(map, Type.CODE);
        return code();
    }
}