package com.atlassian.adf.model.mark;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.mark.type.CodeBlockMark;
import com.atlassian.adf.model.mark.type.ExpandMark;
import com.atlassian.adf.model.mark.type.LayoutSectionMark;
import com.atlassian.adf.model.node.CodeBlock;
import com.atlassian.adf.model.node.Doc;
import com.atlassian.adf.model.node.Paragraph;
import com.atlassian.adf.model.node.type.DocContent;
import com.atlassian.adf.util.EnumParser;
import com.atlassian.adf.util.Factory;

import javax.annotation.concurrent.Immutable;
import java.util.Map;
import java.util.function.Supplier;

import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.checkType;
import static com.atlassian.adf.util.ParserSupport.getAttrOrThrow;

/**
 * The {@code breakout} mark sets block nodes to extend wider than the normal content area.
 * When not present, elements fit within the narrower margins dictated by ADF's normal layout,
 * as inspired by many modern websites like {@code medium.com}. For some content, particularly
 * technical content like code blocks, this can be too restrictive for comfortable viewing.
 * A {@code breakout} mark sets wider margins for specific block node types where this is
 * anticipated to be a concern.
 * <p>
 * All this sounds very Confluence-specific, so this feature is probably not used or
 * supported by other products.
 * <p>
 * The JSON schema currently gives the following reference points:
 * <ul>
 * <li>{@link CodeBlock codeBlock}</li>
 * <li>{@link com.atlassian.adf.model.node.Expand expand}</li>
 * <li>{@link com.atlassian.adf.model.node.LayoutSection layoutSection}</li>
 * </ul>
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre><code>
 * {@link Doc#doc(DocContent[]) doc}(
 *     {@link Paragraph#p(String) p}("Normal:"),
 *     {@link CodeBlock#codeBlock(String) codeBlock}(LOREM_IPSUM),
 *     {@link Paragraph#p(String) p}("Full-width:"),
 *     {@link CodeBlock#codeBlock(String) codeBlock}(LOREM_IPSUM)
 *             .{@link CodeBlock#fullWidth() fullWidth}()
 * )
 * </code></pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *  {
 *    "type": "doc",
 *    "version": 1,
 *    "content": [
 *      {
 *        "type": "paragraph",
 *        "content": [
 *          {
 *            "type": "text",
 *            "text": "Normal:"
 *          }
 *        ]
 *      },
 *      {
 *        "type": "codeBlock",
 *        "content": [
 *          {
 *            "type": "text",
 *            "text": "Lorem ipsum ..."
 *          }
 *        ]
 *      }
 *      {
 *        "type": "paragraph",
 *        "content": [
 *          {
 *            "type": "text",
 *            "text": "Full-width:"
 *          },
 *        ]
 *      },
 *      {
 *        "type": "codeBlock",
 *        "content": [
 *          {
 *            "type": "text",
 *            "text": "Lorem ipsum ..."
 *          }
 *        ],
 *        "marks": [
 *          {
 *            "type": "breakout",
 *            "attrs": {
 *              "mode": "full-width"
 *            }
 *          }
 *        ]
 *      }
 *    ]
 *  }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 *     <p>Normal:</p>
 *     <pre style="background-color: rgb(244, 245, 247); width: 760px;">
 *     Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut
 *     labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco
 *     laboris nisi ut aliquip ex ea commodo consequat. Duis aute irure dolor in reprehenderit in
 *     voluptate velit esse cillum dolore eu fugiat nulla pariatur. Excepteur sint occaecat cupidatat
 *     non proident, sunt in culpa qui officia deserunt mollit anim id est laborum.
 *     </pre>
 *     <p>Full-width:</p>
 *     <pre style="background-color: rgb(244, 245, 247); width: 100%;">
 *     Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut
 *     labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco
 *     laboris nisi ut aliquip ex ea commodo consequat. Duis aute irure dolor in reprehenderit in
 *     voluptate velit esse cillum dolore eu fugiat nulla pariatur. Excepteur sint occaecat cupidatat
 *     non proident, sunt in culpa qui officia deserunt mollit anim id est laborum.
 *     </pre>
 * </div>
 */
@Immutable
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class Breakout
        extends AbstractMark
        implements CodeBlockMark, ExpandMark, LayoutSectionMark {

    private static final Breakout WIDE = new Breakout(Mode.WIDE);
    private static final Breakout FULL_WIDTH = new Breakout(Mode.FULL_WIDTH);

    static final Factory<Breakout> FACTORY = new Factory<>(Type.BREAKOUT, Breakout.class, Breakout::parse);

    private final Mode mode;

    private Breakout(Mode mode) {
        this.mode = mode;
    }

    @Override
    public Breakout copy() {
        return this;
    }

    /**
     * @return a breakout mark indicating that a {@link Mode#WIDE wide} layout should be used
     */
    public static Breakout wide() {
        return WIDE;
    }

    /**
     * @return a breakout mark indicating that a {@link Mode#FULL_WIDTH full-width} layout should be used
     */
    public static Breakout fullWidth() {
        return FULL_WIDTH;
    }

    @Override
    public String elementType() {
        return Type.BREAKOUT;
    }

    /**
     * Returns the {@link Mode mode} setting for this breakout mark.
     */
    public Mode mode() {
        return mode;
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .add(Key.ATTRS, map(Attr.MODE, mode.mode()));
    }

    @Override
    public String toString() {
        return elementType() + '[' + mode + ']';
    }

    private static Breakout parse(Map<String, ?> map) {
        checkType(map, Type.BREAKOUT);
        return Mode.PARSER
                .parse(getAttrOrThrow(map, Attr.MODE, String.class))
                .toBreakout();
    }

    /**
     * Specifies the desired width of content that breaks out of the usual content margins.
     */
    public enum Mode {
        /**
         * Breaks out of the normal margins to permit a wider display of this content, but still leaving
         * a moderate margin on either side.
         */
        WIDE("wide", Breakout::wide),

        /**
         * Breaks out of the normal margins to use the widest display of the content with the smallest margins
         * permitted in Atlassian's design standards.
         */
        FULL_WIDTH("full-width", Breakout::fullWidth);

        private final String mode;
        private final Supplier<Breakout> toBreakout;

        Mode(String mode, Supplier<Breakout> toBreakout) {
            this.mode = mode;
            this.toBreakout = toBreakout;
        }

        static final EnumParser<Mode> PARSER = new EnumParser<>(Mode.class, Mode::mode);

        /**
         * @return the string representation of this breakout mode
         */
        public String mode() {
            return mode;
        }

        /**
         * @return a full {@link Breakout breakout} mark with this value set as its {@code mode}.
         */
        Breakout toBreakout() {
            return toBreakout.get();
        }
    }
}
