package com.atlassian.adf.model.mark;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.mark.type.HeadingMark;
import com.atlassian.adf.model.mark.type.ParagraphMark;
import com.atlassian.adf.model.mark.type.PositionMark;
import com.atlassian.adf.model.node.Doc;
import com.atlassian.adf.model.node.Heading;
import com.atlassian.adf.model.node.Paragraph;
import com.atlassian.adf.model.node.type.DocContent;
import com.atlassian.adf.util.EnumParser;
import com.atlassian.adf.util.Factory;

import javax.annotation.concurrent.Immutable;
import java.util.Map;
import java.util.function.Supplier;

import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.checkType;
import static com.atlassian.adf.util.ParserSupport.getAttrOrThrow;
import static java.util.Objects.requireNonNull;

/**
 * The {@code alignment} mark sets right (end) or center alignment for a node. By default, nodes are left-aligned.
 * This mark applies to:
 * <ul>
 *     <li>{@link Paragraph paragraph}</li>
 *     <li>{@link Heading heading}</li>
 * </ul>
 * <p>
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link Doc#doc(DocContent...) doc}(
 *         {@link Paragraph#p(String) p}("Default"),
 *         {@link Paragraph#p(String) p}("Center").{@link Paragraph#alignment(Alignment) alignment}({@link Alignment#center() center()}),
 *         {@link Paragraph#p(String) p}("End").{@link Paragraph#end() end}()
 * );
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 * {
 *   "type": "doc",
 *   "version": 1,
 *   "content": [
 *     {
 *       "type": "paragraph",
 *       "content": [
 *         {
 *           "type": "text",
 *           "text": "Default"
 *         }
 *       ]
 *     },
 *     {
 *       "type": "paragraph",
 *       "content": [
 *         {
 *           "type": "text",
 *           "text": "Center"
 *         }
 *       ],
 *       "marks": [
 *         {
 *           "type": "alignment",
 *           "attrs": {
 *             "align": "center"
 *           }
 *         }
 *       ]
 *     },
 *     {
 *       "type": "paragraph",
 *       "content": [
 *         {
 *           "type": "text",
 *           "text": "End"
 *         }
 *       ],
 *       "marks": [
 *         {
 *           "type": "alignment",
 *           "attrs": {
 *             "align": "end"
 *           }
 *         }
 *       ]
 *     }
 *   ]
 * }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <p>Default</p>
 * <p style="text-align: center;">Center</p>
 * <p style="text-align: right;">End</p>
 * </div>
 */
@Immutable
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class Alignment
        extends AbstractMark
        implements HeadingMark, ParagraphMark, PositionMark {

    private static final Alignment CENTER = new Alignment(Align.CENTER);
    private static final Alignment END = new Alignment(Align.END);

    static final Factory<Alignment> FACTORY = new Factory<>(Type.ALIGNMENT, Alignment.class, Alignment::parse);

    private final Align align;

    private Alignment(Align align) {
        this.align = requireNonNull(align, "align");
    }

    @Override
    public Alignment copy() {
        return this;
    }

    /**
     * Returns an {@code alignment} mark with {@code align} set to {@link Align#CENTER center}.
     */
    public static Alignment center() {
        return CENTER;
    }

    /**
     * Returns an {@code alignment} mark with {@code align} set to {@link Align#END end}.
     */
    public static Alignment end() {
        return END;
    }

    /**
     * Returns an {@code alignment} mark with the given {@code align} value.
     */
    public static Alignment alignment(String align) {
        return alignment(Align.PARSER.parse(align));
    }

    /**
     * Returns an {@code alignment} mark with the given {@code align} value.
     */
    public static Alignment alignment(Align align) {
        return align.toAlignment();
    }

    /**
     * Returns this {@code alignment} mark's {@code align} value.
     */
    public Align align() {
        return align;
    }

    @Override
    public String elementType() {
        return Type.ALIGNMENT;
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .add(Key.ATTRS, map(Attr.ALIGN, align.align()));
    }

    @Override
    public String toString() {
        return elementType() + '[' + align + ']';
    }

    private static Alignment parse(Map<String, ?> map) {
        checkType(map, Type.ALIGNMENT);
        return alignment(getAttrOrThrow(map, Attr.ALIGN, String.class));
    }

    /**
     * Specifies how the content should be aligned relative to the overall page layout.
     */
    public enum Align {
        /**
         * The content should be centered.
         */
        CENTER("center", Alignment::center),

        /**
         * The content should be aligned on the right (end).
         */
        END("end", Alignment::end);

        static final EnumParser<Align> PARSER = new EnumParser<>(Align.class, Align::align);

        private final String align;
        private final Supplier<Alignment> toAlignment;

        Align(String align, Supplier<Alignment> toAlignment) {
            this.align = align;
            this.toAlignment = toAlignment;
        }

        public String align() {
            return align;
        }

        Alignment toAlignment() {
            return toAlignment.get();
        }
    }
}
