package com.atlassian.adf.model.ex.node;

import static java.util.Objects.requireNonNull;

/**
 * Reports a problem with a {@code media} node or one of the containers that can hold one: {@code mediaGroup},
 * {@code mediaSingle}, or {@code mediaInline}.
 */
public abstract class MediaException extends NodeException {
    private static final long serialVersionUID = 1L;

    MediaException(String message) {
        super(message);
    }

    /**
     * Reports an attempt to use an invalid {@code width} or {@code height} value on a {@code media} node or
     * one of its containers.
     */
    public static abstract class MediaSizeException extends MediaException {
        private static final long serialVersionUID = 1L;

        MediaSizeException(String message) {
            super(message);
        }
    }

    /**
     * Reports an attempt to use an invalid {@code width} value on a {@code media} node or one of its
     * containers, except that {@code mediaSingle} uses {@link WidthMustBeValidPercentage} instead.
     */
    public static class WidthMustBePositive extends MediaSizeException {
        private static final long serialVersionUID = 1L;

        private final Number width;

        public WidthMustBePositive(Number width) {
            super("The 'width' value must be positive: " + width);
            this.width = requireNonNull(width, "width");
        }

        public Number width() {
            return width;
        }
    }

    /**
     * Reports an attempt to use an invalid {@code height} value on a {@code media} node or
     * one of its containers.
     */
    public static class HeightMustBePositive extends MediaSizeException {
        private static final long serialVersionUID = 1L;

        private final Number height;

        public HeightMustBePositive(Number height) {
            super("The 'height' value must be positive: " + height);
            this.height = requireNonNull(height, "height");
        }

        public Number height() {
            return height;
        }
    }

    /**
     * Reports an attempt to use an invalid {@code width} value on a {@code mediaSingle} node
     * when it is expected to be a {@code percentage}.
     */
    public static class WidthMustBeValidPercentage extends MediaSizeException {
        private static final long serialVersionUID = 1L;

        private final Number width;

        public WidthMustBeValidPercentage(Number width) {
            super("The 'width' value must be a valid percentage from 0 to 100: " + width);
            this.width = width;
        }

        public Number width() {
            return width;
        }
    }

    /**
     * Reports an attempt to use an invalid {@code width} value on a {@code mediaSingle} node
     * when it is expected to be a {@code pixel} count.
     */
    public static class WidthMustBeValidPixelCount extends MediaSizeException {
        private static final long serialVersionUID = 1L;

        private final int width;

        public WidthMustBeValidPixelCount(int width) {
            super("The 'width' value must be a non-negative integer: " + width);
            this.width = width;
        }

        public int width() {
            return width;
        }
    }

    /**
     * Reports that too many content items were provided for a {@code mediaSingle} node.
     * That node type can only contain exactly one or two content items: a {@code media} node,
     * optionally followed by a {@code caption} node.
     */
    public static class TooManyContentItems extends MediaException {
        private static final long serialVersionUID = 1L;

        private final int count;

        public TooManyContentItems(int count) {
            super("At most two content items are allowed: a 'media' and its optional 'caption': " + count);
            this.count = count;
        }

        public int count() {
            return count;
        }
    }
}
