package com.atlassian.adf.model.ex.mark;

import com.atlassian.adf.model.ex.AdfException;
import com.atlassian.adf.model.mark.Mark;

import javax.annotation.Nullable;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import static java.util.Collections.emptySet;
import static java.util.Objects.requireNonNull;

/**
 * Reports a problem with an ADF mark.
 */
public abstract class MarkException extends AdfException {
    private static final long serialVersionUID = 1L;

    MarkException(String message) {
        super(message);
    }

    /**
     * Indicates that the given mark type does not satisfy the type system's contraints on what
     * mark types are permitted here. For example, if the ADF specified a {@code strong} mark on a
     * {@code paragraph} node. While suitable for placing on {@code text} nodes within the paragraph,
     * the paragraph itself only supports position marks, like {@code alignment} or {@code indentation}.
     */
    public static class TypeMismatch extends MarkException {
        private static final long serialVersionUID = 1L;

        private final Class<? extends Mark> expectedMarkClass;
        private final String type;

        public TypeMismatch(Class<? extends Mark> expectedMarkClass, String type) {
            super("Cannot accept mark '" + type + "' as " + expectedMarkClass.getName());
            this.expectedMarkClass = requireNonNull(expectedMarkClass, "expectedMarkClass");
            this.type = requireNonNull(type, "type");
        }

        public Class<? extends Mark> expectedMarkClass() {
            return expectedMarkClass;
        }

        public String type() {
            return type;
        }
    }

    /**
     * Indicates that the given mark type does not satisfy the type system's contraints on what
     * mark types are permitted here. For example, if the ADF specified a {@code strong} mark on a
     * {@code paragraph} node. While suitable for placing on {@code text} nodes within the paragraph,
     * the paragraph itself only supports position marks, like {@code alignment} or {@code indentation}.
     */
    public static class TypeUnsupported extends MarkException {
        private static final long serialVersionUID = 1L;

        private final Class<? extends Mark> expectedMarkClass;
        private final String type;

        public TypeUnsupported(Class<? extends Mark> expectedMarkClass, String type) {
            super("Cannot accept unsupported mark '" + type + "' as " + expectedMarkClass.getName());
            this.expectedMarkClass = requireNonNull(expectedMarkClass, "expectedMarkClass");
            this.type = requireNonNull(type, "type");
        }

        public Class<? extends Mark> expectedMarkClass() {
            return expectedMarkClass;
        }

        public String type() {
            return type;
        }
    }

    /**
     * Indicates that a mark which could not be excluded by the type system is nevertheless invalid in the
     * given circumstances due to other constraints related to ADF or how the marked node has been used.
     * For example, subtypes of this exception class are thrown for an attempt to use more than one mark
     * with the same {@code type} on a single node, to use both formatted-text and code marks on a single
     * {@code text} node, or to use any marks at all on a {@code text} node inside of a {@code codeBlock}.
     */
    public static abstract class ConstraintViolation extends MarkException {
        private static final long serialVersionUID = 1L;

        public ConstraintViolation(String message) {
            super(message);
        }
    }

    /**
     * Indicates that a request to disable marks on a given node cannot proceed because the target node is
     * already marked. For example, within a {@code codeBlock} node, the {@code text} nodes it contains are
     * not permitted to have any marks. This exception is thrown if the caller attempts to add a {@code text}
     * node with marks on it to a {@code codeBlock}, because the {@code codeBlock} will request the {@code text}
     * node to disable marking, which it cannot do with marks already present on it.
     */
    public static class RestrictedMarkAlreadyPresent extends ConstraintViolation {
        private static final long serialVersionUID = 1L;

        private final String reason;
        private final List<String> markTypes;

        public RestrictedMarkAlreadyPresent(String reason, Map<String, ?> marks) {
            this(reason, marks.keySet());
        }

        public RestrictedMarkAlreadyPresent(String reason, Collection<String> markTypes) {
            super(reason + ": marks already present: " + markTypes);
            this.reason = requireNonNull(reason, "reason");
            this.markTypes = List.copyOf(markTypes);
            if (this.markTypes.isEmpty()) {
                throw new IllegalArgumentException("markTypes should not have been empty");
            }
        }

        /**
         * @return the reason that marks should be disabled
         */
        public String reason() {
            return reason;
        }

        /**
         * @return the marks that were already present when the request to disable them was made
         */
        public List<String> markTypes() {
            return markTypes;
        }
    }

    /**
     * Indicates that the new mark cannot be added because its use has been restricted for some reason.
     * As an example, a text node that has a {@code code} mark cannot also use a formatted-text mark
     * like {@code strong}.
     */
    public static class MarkDisallowed extends ConstraintViolation {
        private static final long serialVersionUID = 1L;

        private final String reason;
        private final String markType;

        public MarkDisallowed(String reason, Mark mark) {
            this(reason, mark.elementType());
        }

        public MarkDisallowed(String reason, String markType) {
            super(reason + ": " + markType);
            this.reason = requireNonNull(reason, "reason");
            this.markType = requireNonNull(markType, "markType");
        }

        /**
         * @return the reason why the use of the mark has been restricted
         */
        public String reason() {
            return reason;
        }

        /**
         * @return the type of mark that could not be added
         */
        public String markType() {
            return markType;
        }
    }

    /**
     * Indicates that the caller has attempted to add two marks with the same value for the {@code type} field
     * to the same node. It is a common constraint across all node types in ADF that at most one mark of a given
     * type is all that is ever permitted.
     */
    public static class DuplicateMarkType extends ConstraintViolation {
        private static final long serialVersionUID = 1L;

        private final String markType;

        public DuplicateMarkType(Mark mark) {
            this(mark.elementType());
        }

        public DuplicateMarkType(String markType) {
            super("duplicate mark type: " + markType);
            this.markType = requireNonNull(markType, "markType");
        }

        /**
         * @return the mark type that would have been duplicated
         */
        public String markType() {
            return markType;
        }
    }

    /**
     * Indicates that the new mark cannot be added because there is a limit to how many marks the
     * node is permitted to have and that limit has already been reached.
     */
    public static class LimitReached extends ConstraintViolation {
        private static final long serialVersionUID = 1L;

        private final int limit;
        private final List<String> markTypes;

        public LimitReached(int limit, @Nullable Map<String, ?> marks) {
            this(limit, (marks != null) ? marks.keySet() : emptySet());
        }

        public LimitReached(int limit, Collection<String> markTypes) {
            super("mark limit reached: " + limit + ": " + markTypes);
            this.limit = limit;
            this.markTypes = List.copyOf(markTypes);
        }

        /**
         * @return the maximum number of marks that are permitted
         */
        public int limit() {
            return limit;
        }

        /**
         * @return the mark types that are already present
         */
        public List<String> markTypes() {
            return markTypes;
        }
    }
}
