/*
 * Decompiled with CFR 0.152.
 */
package com.android.utils;

import com.android.SdkConstants;
import com.google.common.base.Charsets;
import com.google.common.base.Joiner;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.Iterables;
import com.google.common.hash.HashCode;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.List;
import java.util.regex.Pattern;

public final class FileUtils {
    private static final Joiner PATH_JOINER = Joiner.on((char)File.separatorChar);
    private static final Joiner COMMA_SEPARATED_JOINER = Joiner.on((String)", ");
    private static final Joiner UNIX_NEW_LINE_JOINER = Joiner.on((char)'\n');

    private FileUtils() {
    }

    public static void deletePath(File path) throws IOException {
        if (!path.exists()) {
            return;
        }
        if (path.isDirectory()) {
            FileUtils.deleteDirectoryContents(path);
        }
        if (!path.delete()) {
            throw new IOException(String.format("Could not delete path '%s'.", path));
        }
    }

    public static void deleteDirectoryContents(File directory) throws IOException {
        Preconditions.checkArgument((boolean)directory.isDirectory(), (Object)"!directory.isDirectory");
        File[] files = directory.listFiles();
        Preconditions.checkNotNull((Object)files);
        for (File file : files) {
            FileUtils.deletePath(file);
        }
    }

    public static void cleanOutputDir(File path) throws IOException {
        if (!path.isDirectory()) {
            if (path.exists()) {
                FileUtils.deletePath(path);
            }
            if (!path.mkdirs()) {
                throw new IOException(String.format("Could not create empty folder %s", path));
            }
            return;
        }
        FileUtils.deleteDirectoryContents(path);
    }

    public static void copyFile(File from, File to) throws IOException {
        Files.copy(from.toPath(), to.toPath(), StandardCopyOption.COPY_ATTRIBUTES, StandardCopyOption.REPLACE_EXISTING);
    }

    public static void copyDirectory(File from, File to) throws IOException {
        Preconditions.checkArgument((boolean)from.isDirectory(), (Object)"Source path is not a directory.");
        Preconditions.checkArgument((!to.exists() || to.isDirectory() ? 1 : 0) != 0, (Object)"Destination path exists and is not a directory.");
        FileUtils.mkdirs(to);
        File[] children = from.listFiles();
        if (children != null) {
            for (File child : children) {
                if (child.isFile()) {
                    FileUtils.copyFileToDirectory(child, to);
                    continue;
                }
                if (child.isDirectory()) {
                    FileUtils.copyDirectoryToDirectory(child, to);
                    continue;
                }
                throw new IllegalArgumentException("Don't know how to copy file " + child.getAbsolutePath());
            }
        }
    }

    public static void copyFileToDirectory(File from, File to) throws IOException {
        FileUtils.copyFile(from, new File(to, from.getName()));
    }

    public static void copyDirectoryToDirectory(File from, File to) throws IOException {
        FileUtils.copyDirectory(from, new File(to, from.getName()));
    }

    public static void copyDirectoryContentToDirectory(File from, File to) throws IOException {
        Preconditions.checkArgument((boolean)from.isDirectory(), (Object)"Source path is not a directory.");
        File[] children = from.listFiles();
        if (children != null) {
            for (File f : children) {
                File destination;
                if (f.isDirectory()) {
                    destination = new File(to, FileUtils.relativePath(f, from));
                    com.google.common.io.Files.createParentDirs((File)destination);
                    FileUtils.mkdirs(destination);
                    FileUtils.copyDirectoryContentToDirectory(f, destination);
                    continue;
                }
                if (!f.isFile()) continue;
                destination = new File(to, FileUtils.relativePath(f.getParentFile(), from));
                com.google.common.io.Files.createParentDirs((File)destination);
                FileUtils.mkdirs(destination);
                FileUtils.copyFileToDirectory(f, destination);
            }
        }
    }

    public static File mkdirs(File folder) {
        if (!folder.mkdirs() && !folder.exists()) {
            throw new RuntimeException("Cannot create directory " + folder);
        }
        return folder;
    }

    public static void delete(File file) throws IOException {
        boolean result = file.delete();
        if (!result) {
            throw new IOException("Failed to delete " + file.getAbsolutePath());
        }
    }

    public static void deleteIfExists(File file) throws IOException {
        boolean result = file.delete();
        if (!result && file.exists()) {
            throw new IOException("Failed to delete " + file.getAbsolutePath());
        }
    }

    public static void renameTo(File file, File to) throws IOException {
        boolean result = file.renameTo(to);
        if (!result) {
            throw new IOException("Failed to rename " + file.getAbsolutePath() + " to " + to);
        }
    }

    public static File join(File dir, String ... paths) {
        if (paths.length == 0) {
            return dir;
        }
        return new File(dir, PATH_JOINER.join((Object[])paths));
    }

    public static File join(File dir, Iterable<String> paths) {
        return new File(dir, PATH_JOINER.join(paths));
    }

    public static String join(String ... paths) {
        return PATH_JOINER.join((Object[])paths);
    }

    public static String join(Iterable<String> paths) {
        return PATH_JOINER.join(paths);
    }

    public static String loadFileWithUnixLineSeparators(File file) throws IOException {
        return UNIX_NEW_LINE_JOINER.join((Iterable)com.google.common.io.Files.readLines((File)file, (Charset)Charsets.UTF_8));
    }

    public static String relativePath(File file, File dir) {
        Preconditions.checkArgument((file.isFile() || file.isDirectory() ? 1 : 0) != 0, (String)"%s is not a file nor a directory.", (Object[])new Object[]{file.getPath()});
        Preconditions.checkArgument((boolean)dir.isDirectory(), (String)"%s is not a directory.", (Object[])new Object[]{dir.getPath()});
        return FileUtils.relativePossiblyNonExistingPath(file, dir);
    }

    public static String relativePossiblyNonExistingPath(File file, File dir) {
        String path = dir.toURI().relativize(file.toURI()).getPath();
        return FileUtils.toSystemDependentPath(path);
    }

    public static String toSystemDependentPath(String path) {
        if (File.separatorChar != '/') {
            path = path.replace('/', File.separatorChar);
        }
        return path;
    }

    public static String toSystemIndependentPath(String path) {
        if (File.separatorChar != '/') {
            path = path.replace(File.separatorChar, '/');
        }
        return path;
    }

    public static String sha1(File file) throws IOException {
        return Hashing.sha1().hashBytes(com.google.common.io.Files.toByteArray((File)file)).toString();
    }

    public static FluentIterable<File> getAllFiles(File dir) {
        return com.google.common.io.Files.fileTreeTraverser().preOrderTraversal((Object)dir).filter(com.google.common.io.Files.isFile());
    }

    public static String getNamesAsCommaSeparatedList(Iterable<File> files) {
        return COMMA_SEPARATED_JOINER.join(Iterables.transform(files, File::getName));
    }

    public static String getDirectoryNameForJar(File inputFile) {
        HashFunction hashFunction = Hashing.sha1();
        HashCode hashCode = hashFunction.hashString((CharSequence)inputFile.getAbsolutePath(), Charsets.UTF_16LE);
        String name = com.google.common.io.Files.getNameWithoutExtension((String)inputFile.getName());
        if (name.equals("classes") && inputFile.getAbsolutePath().contains("exploded-aar")) {
            File versionDir = inputFile.getParentFile().getParentFile();
            File artifactDir = versionDir.getParentFile();
            File groupDir = artifactDir.getParentFile();
            name = Joiner.on((char)'-').join((Object)groupDir.getName(), (Object)artifactDir.getName(), new Object[]{versionDir.getName()});
        }
        name = name + "_" + hashCode.toString();
        return name;
    }

    public static void createFile(File file, String content) throws IOException {
        Preconditions.checkArgument((!file.exists() ? 1 : 0) != 0, (String)"%s exists already.", (Object[])new Object[]{file});
        com.google.common.io.Files.createParentDirs((File)file);
        com.google.common.io.Files.write((CharSequence)content, (File)file, (Charset)Charsets.UTF_8);
    }

    public static List<File> find(File base, Pattern pattern) {
        Preconditions.checkArgument((boolean)base.isDirectory(), (String)"'%s' must be a directory.", (Object[])new Object[]{base.getAbsolutePath()});
        return com.google.common.io.Files.fileTreeTraverser().preOrderTraversal((Object)base).filter(Predicates.compose((Predicate)Predicates.contains((Pattern)pattern), File::getPath)).toList();
    }

    public static Optional<File> find(File base, String name) {
        Preconditions.checkArgument((boolean)base.isDirectory(), (String)"'%s' must be a directory.", (Object[])new Object[]{base.getAbsolutePath()});
        return com.google.common.io.Files.fileTreeTraverser().preOrderTraversal((Object)base).filter(Predicates.compose((Predicate)Predicates.equalTo((Object)name), File::getName)).last();
    }

    public static byte[] readSegment(File file, long start, int length) throws Exception {
        byte[] data;
        Preconditions.checkArgument((start >= 0L ? 1 : 0) != 0, (Object)"start < 0");
        Preconditions.checkArgument((length >= 0 ? 1 : 0) != 0, (Object)"length < 0");
        try (RandomAccessFile raf = new RandomAccessFile(file, "r");){
            int r;
            raf.seek(start);
            data = new byte[length];
            for (int tot = 0; tot < length; tot += r) {
                r = raf.read(data, tot, length - tot);
                if (r >= 0) continue;
                throw new EOFException();
            }
        }
        return data;
    }

    public static String joinFilePaths(Iterable<File> files) {
        return Joiner.on((char)File.pathSeparatorChar).join(Iterables.transform(files, File::getAbsolutePath));
    }

    public static String getValidFileName(String baseName, String extension, File directory) {
        String fileName = extension.isEmpty() ? baseName : baseName + "." + extension;
        String validBaseName = baseName.replaceAll("[^a-zA-Z0-9]", "_");
        String validExtension = extension.replaceAll("[^a-zA-Z0-9]", "_");
        String validExtensionWithDot = validExtension.isEmpty() ? "" : "." + validExtension;
        String validFileName = validBaseName + validExtensionWithDot;
        String fileHash = Hashing.sha1().hashString((CharSequence)fileName, Charsets.UTF_16LE).toString();
        if (!validFileName.equals(fileName)) {
            validFileName = validBaseName + "_" + fileHash + validExtensionWithDot;
        }
        if (FileUtils.isFilePathTooLong(validFileName, directory) && FileUtils.isFilePathTooLong(validFileName = fileHash + validExtensionWithDot, directory)) {
            throw new RuntimeException("File name or file path is too long: " + new File(directory, validFileName).getAbsolutePath());
        }
        return validFileName;
    }

    public static boolean isFilePathTooLong(String fileName, File directory) {
        return fileName.length() > 255 || new File(directory, fileName).getAbsolutePath().length() > 240 && SdkConstants.currentPlatform() == 2;
    }
}

