package com.android.billingclient.api;

import android.text.TextUtils;
import com.android.billingclient.api.BillingClient.BillingResponse;
import com.android.billingclient.api.BillingClient.SkuType;
import java.util.List;
import org.json.JSONException;
import org.json.JSONObject;

/** Represents an in-app product's or subscription's listing details. */
public class SkuDetails {
  private final String mOriginalJson;
  private final JSONObject mParsedJson;

  public SkuDetails(String jsonSkuDetails) throws JSONException {
    mOriginalJson = jsonSkuDetails;
    mParsedJson = new JSONObject(mOriginalJson);
  }

  /** Returns the product Id. */
  public String getSku() {
    return mParsedJson.optString("productId");
  }

  /** Returns SKU type. */
  @SuppressWarnings("WrongConstant")
  @SkuType
  public String getType() {
    return mParsedJson.optString("type");
  }

  /**
   * Returns formatted price of the item, including its currency sign. The price does not include
   * tax.
   */
  public String getPrice() {
    return mParsedJson.optString("price");
  }

  /**
   * Returns price in micro-units, where 1,000,000 micro-units equal one unit of the currency.
   *
   * <p>For example, if price is "€7.99", price_amount_micros is "7990000". This value represents
   * the localized, rounded price for a particular currency.
   */
  public long getPriceAmountMicros() {
    return mParsedJson.optLong("price_amount_micros");
  }

  /**
   * Returns ISO 4217 currency code for price.
   *
   * <p>For example, if price is specified in British pounds sterling, price_currency_code is "GBP".
   */
  public String getPriceCurrencyCode() {
    return mParsedJson.optString("price_currency_code");
  }

  /** Returns the title of the product. */
  public String getTitle() {
    return mParsedJson.optString("title");
  }

  /** Returns the description of the product. */
  public String getDescription() {
    return mParsedJson.optString("description");
  }

  @Override
  public String toString() {
    return "SkuDetails: " + mOriginalJson;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    SkuDetails details = (SkuDetails) o;

    return TextUtils.equals(mOriginalJson, details.mOriginalJson);
  }

  @Override
  public int hashCode() {
    return mOriginalJson.hashCode();
  }

  /** Result list and code for querySkuDetailsAsync method */
  public static class SkuDetailsResult {
    private List<SkuDetails> mSkuDetailsList;
    @BillingResponse private int mResponseCode;

    public SkuDetailsResult(List<SkuDetails> skuDetailsList, @BillingResponse int resultCode) {
      this.mSkuDetailsList = skuDetailsList;
      this.mResponseCode = resultCode;
    }

    public List<SkuDetails> getSkuDetailsList() {
      return mSkuDetailsList;
    }

    @BillingResponse
    public int getResponseCode() {
      return mResponseCode;
    }
  }
}
