package com.android.billingclient.api;

import static com.android.billingclient.api.BillingClient.SkuType.SUBS;
import static com.android.billingclient.util.BillingHelper.INAPP_CONTINUATION_TOKEN;
import static com.android.billingclient.util.BillingHelper.RESPONSE_BUY_INTENT;
import static com.android.billingclient.util.ProxyBillingActivity.RECEIVER_EXTRA;

import android.app.Activity;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.content.pm.ResolveInfo;
import android.os.Bundle;
import android.os.Handler;
import android.os.IBinder;
import android.os.RemoteException;
import android.os.ResultReceiver;
import android.support.annotation.NonNull;
import android.support.annotation.UiThread;
import android.support.annotation.VisibleForTesting;
import android.support.annotation.WorkerThread;
import android.text.TextUtils;
import com.android.billingclient.api.Purchase.PurchasesResult;
import com.android.billingclient.api.SkuDetails.SkuDetailsResult;
import com.android.billingclient.util.BillingHelper;
import com.android.billingclient.util.ProxyBillingActivity;
import com.android.vending.billing.IInAppBillingService;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.json.JSONException;

/**
 * Implementation of {@link BillingClient} for communication between the in-app billing library and
 * client's application code.
 */
public class BillingClientImpl extends BillingClient {
  private static final String TAG = "BillingClient";

  /**
   * The maximum number of items than can be requested by a call to Billing service's
   * getSkuDetails() method
   */
  private static final int MAX_SKU_DETAILS_ITEMS_PER_REQUEST = 20;

  /** A list of SKUs inside getSkuDetails request bundle. */
  private static final String GET_SKU_DETAILS_ITEM_LIST = "ITEM_ID_LIST";

  /** Field's key to hold VR related constant. */
  private static final String KEY_VR = "vr";

  /** Field's key to hold library version key constant. */
  private static final String LIBRARY_VERSION_KEY = "libraryVersion";

  /** Field's key value to hold current library version. */
  private static final String LIBRARY_VERSION = "dp-1";

  /** Main thread handler to post results from Executor. */
  private final Handler mUiThreadHandler = new Handler();

  /**
   * Wrapper on top of PURCHASES_UPDATED broadcast receiver to return all purchases receipts to the
   * developer in one place for both app initiated and Play Store initated purhases.
   */
  private final BillingBroadcastManager mBroadcastManager;

  /** Context we were passed during initialization. */
  private Context mContext;

  /** Service binder */
  private IInAppBillingService mService;

  /** Connection to the service. */
  private ServiceConnection mServiceConnection;

  /** If subscriptions are is supported (for billing v3 and higher) or not. */
  private boolean mSubscriptionsSupported;

  /** If subscription update is supported (for billing v5 and higher) or not. */
  private boolean mSubscriptionUpdateSupported;

  /**
   * Service that helps us to keep a pool of background threads suitable for current device specs.
   */
  private ExecutorService mExecutorService;

  /** True if we were successfully connected to the service. */
  private boolean mSetupDone;

  @UiThread
  BillingClientImpl(@NonNull Context context, @NonNull PurchasesUpdatedListener listener) {
    mContext = context.getApplicationContext();
    mBroadcastManager = new BillingBroadcastManager(mContext, listener);
  }

  @Override
  public @BillingResponse int isFeatureSupported(@FeatureType String feature) {
    if (!isReady()) {
      return BillingResponse.SERVICE_DISCONNECTED;
    }

    switch (feature) {
      case FeatureType.SUBSCRIPTIONS:
        return mSubscriptionsSupported ? BillingResponse.OK : BillingResponse.FEATURE_NOT_SUPPORTED;

      case FeatureType.SUBSCRIPTIONS_UPDATE:
        return mSubscriptionUpdateSupported
            ? BillingResponse.OK
            : BillingResponse.FEATURE_NOT_SUPPORTED;

      case FeatureType.IN_APP_ITEMS_ON_VR:
        return isBillingSupportedOnVr(SkuType.INAPP);

      case FeatureType.SUBSCRIPTIONS_ON_VR:
        return isBillingSupportedOnVr(SUBS);

      default:
        BillingHelper.logWarn(TAG, "Unsupported feature: " + feature);
        return BillingResponse.DEVELOPER_ERROR;
    }
  }

  @Override
  public boolean isReady() {
    return mSetupDone && mService != null && mServiceConnection != null;
  }

  @Override
  public void startConnection(@NonNull BillingClientStateListener listener) {
    if (isReady()) {
      BillingHelper.logVerbose(TAG, "Service connection is valid. No need to re-initialize.");
      listener.onBillingSetupFinished(BillingResponse.OK);
      return;
    }

    // Start listening to PURCHASES_UPDATED broadcasts
    mBroadcastManager.registerReceiver();

    // Connection to billing service
    BillingHelper.logVerbose(TAG, "Starting in-app billing setup.");
    mServiceConnection = new BillingServiceConnection(listener);

    Intent serviceIntent = new Intent("com.android.vending.billing.InAppBillingService.BIND");
    serviceIntent.setPackage("com.android.vending");
    List<ResolveInfo> intentServices =
        mContext.getPackageManager().queryIntentServices(serviceIntent, 0);

    if (intentServices != null && !intentServices.isEmpty()) {
      // service available to handle that Intent
      serviceIntent.putExtra(LIBRARY_VERSION_KEY, LIBRARY_VERSION);
      mContext.bindService(serviceIntent, mServiceConnection, Context.BIND_AUTO_CREATE);
    } else {
      // no service available to handle that Intent
      BillingHelper.logVerbose(TAG, "Billing service unavailable on device.");
      listener.onBillingSetupFinished(BillingResponse.BILLING_UNAVAILABLE);
    }
  }

  @Override
  public void endConnection() {
    mBroadcastManager.destroy();
    mSetupDone = false;

    if (mServiceConnection != null) {
      BillingHelper.logVerbose(TAG, "Unbinding from service.");
      if (mContext != null) {
        mContext.unbindService(mServiceConnection);
        mContext = null;
      }
      mServiceConnection = null;
    }

    mService = null;

    if (mExecutorService != null) {
      mExecutorService.shutdownNow();
      mExecutorService = null;
    }
  }

  @Override
  public int launchBillingFlow(Activity activity, BillingFlowParams params) {
    if (!isReady()) {
      return BillingResponse.SERVICE_DISCONNECTED;
    }

    @SkuType String skuType = params.getSkuType();
    String newSku = params.getSku();

    // Checking for mandatory params fields
    if (newSku == null) {
      BillingHelper.logWarn(TAG, "Please fix the input params. SKU can't be null.");
      return BillingResponse.DEVELOPER_ERROR;
    }

    if (skuType == null) {
      BillingHelper.logWarn(TAG, "Please fix the input params. SkuType can't be null.");
      return BillingResponse.DEVELOPER_ERROR;
    }

    if (params.getOldSkus() != null && params.getOldSkus().size() < 1) {
      BillingHelper.logWarn(TAG, "Please fix the input params. OldSkus size can't be 0.");
      return BillingResponse.DEVELOPER_ERROR;
    }

    // Checking for requested features support
    if (skuType.equals(SUBS) && !mSubscriptionsSupported) {
      return BillingResponse.FEATURE_NOT_SUPPORTED;
    }

    boolean isSubscriptionUpdate = (params.getOldSkus() != null);

    if (isSubscriptionUpdate && !mSubscriptionUpdateSupported) {
      return BillingResponse.FEATURE_NOT_SUPPORTED;
    }

    try {
      BillingHelper.logVerbose(
          TAG, "Constructing buy intent for " + newSku + ", " + "item type: " + skuType);

      Bundle buyIntentBundle;

      if (params.hasExtraParams()) {
        Bundle extraParams = constructExtraParams(params);
        int apiVersion = (params.getVrPurchaseFlow()) ? 7 : 6;
        buyIntentBundle =
            mService.getBuyIntentExtraParams(
                apiVersion, mContext.getPackageName(), newSku, skuType, null, extraParams);
      } else if (isSubscriptionUpdate) {
        // For subscriptions update we are calling corresponding service method
        buyIntentBundle =
            mService.getBuyIntentToReplaceSkus(
                5, mContext.getPackageName(), params.getOldSkus(), newSku, SUBS, null);
      } else {
        buyIntentBundle =
            mService.getBuyIntent(3, mContext.getPackageName(), newSku, skuType, null);
      }

      int responseCode = BillingHelper.getResponseCodeFromBundle(buyIntentBundle, TAG);
      if (responseCode != BillingResponse.OK) {
        BillingHelper.logWarn(TAG, "Unable to buy item, Error response code: " + responseCode);
        return responseCode;
      }

      // Create a result receiver that will propagate the result from InvisibleActivity
      // into PurchasesUpdatedListener specified in the constructor.
      ResultReceiver purchaseResultReceiver =
          new ResultReceiver(mUiThreadHandler) {
            @Override
            protected void onReceiveResult(int responseCode, Bundle resultData) {

              List<Purchase> purchases =
                  (resultData == null) ? null : BillingHelper.extractPurchases(resultData);
              mBroadcastManager.getListener().onPurchasesUpdated(responseCode, purchases);
            }
          };
      // Launching an invisible activity that will handle the purchase result
      Intent intent = new Intent(activity, ProxyBillingActivity.class);
      intent.putExtra(RECEIVER_EXTRA, purchaseResultReceiver);
      intent.putExtra(RESPONSE_BUY_INTENT, buyIntentBundle.getParcelable(RESPONSE_BUY_INTENT));
      // We need an activity reference here to avoid using FLAG_ACTIVITY_NEW_TASK.
      // But we don't want to keep a reference to it inside the field to avoid memory leaks.
      // Plus all the other methods need just a Context reference, so could be used from the
      // Service or Application.
      activity.startActivity(intent);
    } catch (RemoteException e) {
      String msg =
          "RemoteException while launching launching replace subscriptions flow: "
              + "; for sku: "
              + newSku
              + "; try to reconnect";
      BillingHelper.logWarn(TAG, msg);
      return BillingResponse.SERVICE_DISCONNECTED;
    }

    return BillingResponse.OK;
  }

  @Override
  public PurchasesResult queryPurchases(@SkuType String skuType) {
    if (!isReady()) {
      return new PurchasesResult(null, BillingResponse.SERVICE_DISCONNECTED);
    }

    return queryPurchasesInternal(skuType, false /* queryHistory */);
  }

  @Override
  public void querySkuDetailsAsync(
      @SkuType final String skuType,
      final List<String> skuList,
      final SkuDetailsResponseListener listener) {
    if (!isReady()) {
      listener.onSkuDetailsResponse(
          new SkuDetailsResult(null, BillingResponse.SERVICE_DISCONNECTED));
    }

    if (skuList == null) {
      throw new IllegalArgumentException(
          "Please provide a non-null list of SKUs to " + "querySkuDetailsAsync()");
    } else if (skuList.size() == 0) {
      throw new IllegalArgumentException(
          "Please provide a non-zero length list of SKUs to " + "querySkuDetailsAsync()");
    }

    executeAsync(
        new Runnable() {
          @Override
          public void run() {
            final SkuDetailsResult result = querySkuDetailsInternal(skuType, skuList);
            // Post the result to main thread
            postToUiThread(
                new Runnable() {
                  @Override
                  public void run() {
                    listener.onSkuDetailsResponse(result);
                  }
                });
          }
        });
  }

  @Override
  public void consumeAsync(final String purchaseToken, final ConsumeResponseListener listener) {
    if (!isReady()) {
      listener.onConsumeResponse(null, BillingResponse.SERVICE_DISCONNECTED);
    }

    executeAsync(
        new Runnable() {
          @Override
          public void run() {
            consumeInternal(purchaseToken, listener);
          }
        });
  }

  @Override
  public void queryPurchaseHistoryAsync(
      final @SkuType String skuType, final PurchaseHistoryResponseListener listener) {
    if (!isReady()) {
      listener.onPurchaseHistoryResponse(
          new PurchasesResult(null, BillingResponse.SERVICE_DISCONNECTED));
    }

    executeAsync(
        new Runnable() {
          @Override
          public void run() {
            final PurchasesResult result = queryPurchasesInternal(skuType, true /*queryHistory */);

            // Post the result to main thread
            postToUiThread(
                new Runnable() {
                  @Override
                  public void run() {
                    listener.onPurchaseHistoryResponse(result);
                  }
                });
          }
        });
  }

  private Bundle constructExtraParams(BillingFlowParams params) {
    Bundle extraParams = new Bundle();

    if (params.getReplaceSkusProration()) {
      extraParams.putBoolean("replaceSkusProration", true);
    }
    if (params.getAccountId() != null) {
      extraParams.putString("accountId", params.getAccountId());
    }
    if (params.getVrPurchaseFlow()) {
      extraParams.putBoolean(KEY_VR, true);
    }
    if (params.getOldSkus() != null) {
      extraParams.putStringArrayList("skusToReplace", params.getOldSkus());
    }

    return extraParams;
  }

  private void executeAsync(Runnable runnable) {
    if (mExecutorService == null) {
      mExecutorService = Executors.newFixedThreadPool(BillingHelper.NUMBER_OF_CORES);
    }

    mExecutorService.submit(runnable);
  }

  /** Checks if billing on VR is supported for corresponding billing type. */
  private int isBillingSupportedOnVr(@SkuType String skuType) {
    try {
      int supportedResult =
          mService.isBillingSupportedExtraParams(
              7 /* apiVersion */, mContext.getPackageName(), skuType, generateVrBundle());
      return (supportedResult == BillingResponse.OK)
          ? BillingResponse.OK
          : BillingResponse.FEATURE_NOT_SUPPORTED;
    } catch (RemoteException e) {
      BillingHelper.logWarn(
          TAG, "RemoteException while checking if billing is supported; " + "try to reconnect");
      return BillingResponse.SERVICE_DISCONNECTED;
    }
  }

  /**
   * Generates a Bundle to indicate that we are request a method for VR experience within
   * extraParams
   */
  private Bundle generateVrBundle() {
    Bundle result = new Bundle();
    result.putBoolean(KEY_VR, true);
    return result;
  }

  @VisibleForTesting
  SkuDetailsResult querySkuDetailsInternal(@SkuType String skuType, List<String> skuList) {
    List<SkuDetails> resultList = new ArrayList<>();

    // Split the sku list into packs of no more than MAX_SKU_DETAILS_ITEMS_PER_REQUEST elements
    int startIndex = 0, listSize = skuList.size();
    while (startIndex < listSize) {
      // Prepare a network request up to a maximum amount of supported elements
      int endIndex = startIndex + MAX_SKU_DETAILS_ITEMS_PER_REQUEST;
      if (endIndex > listSize) {
        endIndex = listSize;
      }
      ArrayList<String> curSkuList = new ArrayList<>(skuList.subList(startIndex, endIndex));
      Bundle querySkus = new Bundle();
      querySkus.putStringArrayList(GET_SKU_DETAILS_ITEM_LIST, curSkuList);
      Bundle skuDetails;
      try {
        skuDetails = mService.getSkuDetails(3, mContext.getPackageName(), skuType, querySkus);
      } catch (RemoteException e) {
        String msg = "querySkuDetailsAsync got a remote exception (try to reconnect): " + e;
        BillingHelper.logWarn(TAG, msg);
        return new SkuDetailsResult(null, BillingResponse.SERVICE_DISCONNECTED);
      }

      if (skuDetails == null) {
        BillingHelper.logWarn(TAG, "querySkuDetailsAsync got null sku details list");
        return new SkuDetailsResult(null, BillingResponse.ITEM_UNAVAILABLE);
      }

      if (!skuDetails.containsKey(BillingHelper.RESPONSE_GET_SKU_DETAILS_LIST)) {
        @BillingResponse
        int responseCode = BillingHelper.getResponseCodeFromBundle(skuDetails, TAG);

        if (responseCode != BillingResponse.OK) {
          BillingHelper.logWarn(TAG, "getSkuDetails() failed. Response code: " + responseCode);
          return new SkuDetailsResult(resultList, responseCode);
        } else {
          BillingHelper.logWarn(
              TAG,
              "getSkuDetails() returned a bundle with neither" + " an error nor a detail list.");
          return new SkuDetailsResult(resultList, BillingResponse.ERROR);
        }
      }

      ArrayList<String> skuDetailsJsonList =
          skuDetails.getStringArrayList(BillingHelper.RESPONSE_GET_SKU_DETAILS_LIST);

      if (skuDetailsJsonList == null) {
        BillingHelper.logWarn(TAG, "querySkuDetailsAsync got null response list");
        return new SkuDetailsResult(null, BillingResponse.ITEM_UNAVAILABLE);
      }

      for (int i = 0; i < skuDetailsJsonList.size(); ++i) {
        String thisResponse = skuDetailsJsonList.get(i);
        SkuDetails currentSkuDetails;
        try {
          currentSkuDetails = new SkuDetails(thisResponse);
        } catch (JSONException e) {
          BillingHelper.logWarn(TAG, "Got a JSON exception trying to decode SkuDetails");
          return new SkuDetailsResult(null, BillingResponse.ERROR);
        }
        BillingHelper.logVerbose(TAG, "Got sku details: " + currentSkuDetails);
        resultList.add(currentSkuDetails);
      }

      // Switching start index to the end of just received pack
      startIndex += MAX_SKU_DETAILS_ITEMS_PER_REQUEST;
    }

    return new SkuDetailsResult(resultList, BillingResponse.OK);
  }

  /**
   * Queries purchases or purchases history and combines all the multi-page results into one list
   */
  private PurchasesResult queryPurchasesInternal(@SkuType String skuType, boolean queryHistory) {
    BillingHelper.logVerbose(
        TAG, "Querying owned items, item type: " + skuType + "; " + "history: " + queryHistory);

    String continueToken = null;
    List<Purchase> resultList = new ArrayList<>();

    do {
      Bundle ownedItems;
      try {
        if (queryHistory) {
          ownedItems =
              mService.getPurchaseHistory(
                  6 /* apiVersion */,
                  mContext.getPackageName(),
                  skuType,
                  continueToken,
                  null /* extraParams */);
        } else {
          ownedItems =
              mService.getPurchases(
                  3 /* apiVersion */, mContext.getPackageName(), skuType, continueToken);
        }
      } catch (RemoteException e) {
        BillingHelper.logWarn(
            TAG, "Got exception trying to get purchases: " + e + "; try to reconnect");
        return new PurchasesResult(null, BillingResponse.SERVICE_DISCONNECTED);
      }

      if (ownedItems == null) {
        BillingHelper.logWarn(TAG, "queryPurchases got null owned items list");
        return new PurchasesResult(null, BillingResponse.ERROR);
      }

      @BillingResponse int responseCode = BillingHelper.getResponseCodeFromBundle(ownedItems, TAG);

      if (responseCode != BillingResponse.OK) {
        BillingHelper.logWarn(TAG, "getPurchases() failed. Response code: " + responseCode);
        return new PurchasesResult(null, responseCode);
      }

      if (!ownedItems.containsKey(BillingHelper.RESPONSE_INAPP_ITEM_LIST)
          || !ownedItems.containsKey(BillingHelper.RESPONSE_INAPP_PURCHASE_DATA_LIST)
          || !ownedItems.containsKey(BillingHelper.RESPONSE_INAPP_SIGNATURE_LIST)) {
        BillingHelper.logWarn(
            TAG, "Bundle returned from getPurchases() doesn't contain required fields.");
        return new PurchasesResult(null, BillingResponse.ERROR);
      }

      ArrayList<String> ownedSkus =
          ownedItems.getStringArrayList(BillingHelper.RESPONSE_INAPP_ITEM_LIST);
      ArrayList<String> purchaseDataList =
          ownedItems.getStringArrayList(BillingHelper.RESPONSE_INAPP_PURCHASE_DATA_LIST);
      ArrayList<String> signatureList =
          ownedItems.getStringArrayList(BillingHelper.RESPONSE_INAPP_SIGNATURE_LIST);

      if (ownedSkus == null) {
        BillingHelper.logWarn(TAG, "Bundle returned from getPurchases() contains null SKUs list.");
        return new PurchasesResult(null, BillingResponse.ERROR);
      }

      if (purchaseDataList == null) {
        BillingHelper.logWarn(
            TAG, "Bundle returned from getPurchases() contains null purchases list.");
        return new PurchasesResult(null, BillingResponse.ERROR);
      }

      if (signatureList == null) {
        BillingHelper.logWarn(
            TAG, "Bundle returned from getPurchases() contains null signatures list.");
        return new PurchasesResult(null, BillingResponse.ERROR);
      }

      for (int i = 0; i < purchaseDataList.size(); ++i) {
        String purchaseData = purchaseDataList.get(i);
        String signature = signatureList.get(i);
        String sku = ownedSkus.get(i);

        BillingHelper.logVerbose(TAG, "Sku is owned: " + sku);
        Purchase purchase;
        try {
          purchase = new Purchase(purchaseData, signature);
        } catch (JSONException e) {
          BillingHelper.logWarn(TAG, "Got an exception trying to decode the purchase: " + e);
          return new PurchasesResult(null, BillingResponse.ERROR);
        }

        if (TextUtils.isEmpty(purchase.getPurchaseToken())) {
          BillingHelper.logWarn(TAG, "BUG: empty/null token!");
        }

        resultList.add(purchase);
      }

      continueToken = ownedItems.getString(INAPP_CONTINUATION_TOKEN);
      BillingHelper.logVerbose(TAG, "Continuation token: " + continueToken);
    } while (!TextUtils.isEmpty(continueToken));

    return new PurchasesResult(resultList, BillingResponse.OK);
  }

  /** Execute the runnable on the UI/Main Thread */
  private void postToUiThread(Runnable runnable) {
    mUiThreadHandler.post(runnable);
  }

  /** Consume the purchase and execute listener's callback on the Ui/Main thread */
  @WorkerThread
  private void consumeInternal(final String purchaseToken, final ConsumeResponseListener listener) {
    try {
      BillingHelper.logVerbose(TAG, "Consuming purchase with token: " + purchaseToken);
      final @BillingResponse int responseCode =
          mService.consumePurchase(3 /* apiVersion */, mContext.getPackageName(), purchaseToken);

      if (responseCode == BillingResponse.OK) {
        BillingHelper.logVerbose(TAG, "Successfully consumed purchase.");
        if (listener != null) {
          postToUiThread(
              new Runnable() {
                @Override
                public void run() {
                  listener.onConsumeResponse(purchaseToken, responseCode);
                }
              });
        }
      } else {
        BillingHelper.logWarn(
            TAG, "Error consuming purchase with token. Response code: " + responseCode);

        postToUiThread(
            new Runnable() {
              @Override
              public void run() {
                BillingHelper.logWarn(TAG, "Error consuming purchase.");
                listener.onConsumeResponse(purchaseToken, responseCode);
              }
            });
      }
    } catch (final RemoteException e) {
      postToUiThread(
          new Runnable() {
            @Override
            public void run() {
              BillingHelper.logWarn(TAG, "Error consuming purchase; ex: " + e);
              listener.onConsumeResponse(purchaseToken, BillingResponse.SERVICE_DISCONNECTED);
            }
          });
    }
  }

  /** Connect with Billing service and notify listener about important states. */
  private final class BillingServiceConnection implements ServiceConnection {
    private final BillingClientStateListener mListener;

    private BillingServiceConnection(@NonNull BillingClientStateListener listener) {
      if (listener == null) {
        throw new RuntimeException("Please specify a listener to know when init is done.");
      }
      mListener = listener;
    }

    @Override
    public void onServiceDisconnected(ComponentName name) {
      BillingHelper.logWarn(TAG, "Billing service disconnected.");
      mService = null;
      mSetupDone = false;
      mListener.onBillingServiceDisconnected();
    }

    @Override
    public void onServiceConnected(ComponentName name, IBinder service) {
      BillingHelper.logVerbose(TAG, "Billing service connected.");

      mService = IInAppBillingService.Stub.asInterface(service);
      String packageName = mContext.getPackageName();
      mSubscriptionsSupported = false;
      mSubscriptionUpdateSupported = false;
      try {
        BillingHelper.logVerbose(TAG, "Checking for in-app billing 3 support.");

        // check for in-app billing v3 support
        @BillingResponse
        int response = mService.isBillingSupported(3 /* apiVersion */, packageName, SkuType.INAPP);
        if (response != BillingResponse.OK) {
          BillingHelper.logWarn(TAG, "Error checking for billing v3 support.");
          mListener.onBillingSetupFinished(response);
          return;
        } else {
          BillingHelper.logVerbose(TAG, "In-app billing version 3 supported.");
        }

        // Check for v5 subscriptions support. This is needed for
        // getBuyIntentToReplaceSku which allows for subscription update
        // noinspection WrongConstant
        response = mService.isBillingSupported(5, packageName, SUBS);

        if (response == BillingResponse.OK) {
          BillingHelper.logVerbose(TAG, "Subscription re-signup available..");
          mSubscriptionUpdateSupported = true;
          mSubscriptionsSupported = true;
        } else {
          BillingHelper.logVerbose(TAG, "Subscription re-signup not available.");
          mSubscriptionUpdateSupported = false;
        }

        if (!mSubscriptionsSupported) {
          // check for v3 subscriptions support
          // noinspection WrongConstant
          response = mService.isBillingSupported(3 /* apiVersion */, packageName, SUBS);
          if (response == BillingResponse.OK) {
            BillingHelper.logVerbose(TAG, "Subscriptions available.");
            mSubscriptionsSupported = true;
          } else {
            BillingHelper.logVerbose(
                TAG, "Subscriptions not available." + "BillingResponse: " + response);
          }
        }
        mSetupDone = true;
      } catch (final RemoteException e) {
        BillingHelper.logWarn(TAG, "RemoteException while setting up in-app billing" + e);
        mListener.onBillingSetupFinished(BillingResponse.SERVICE_DISCONNECTED);
        mSetupDone = false;
        return;
      }

      BillingHelper.logVerbose(TAG, "Billing client setup was successful!");
      mListener.onBillingSetupFinished(BillingResponse.OK);
    }
  }
}
