package com.adpdigital.push;

import android.annotation.TargetApi;
import android.app.Notification;
import android.app.PendingIntent;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.content.SharedPreferences;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.media.RingtoneManager;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.IBinder;
import android.support.v4.app.NotificationCompat;
import android.support.v4.app.NotificationManagerCompat;
import android.util.Log;

import com.google.android.gms.gcm.GcmListenerService;
import com.strongloop.android.loopback.LocalInstallation;

import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import de.greenrobot.event.EventBus;
import me.leolin.shortcutbadger.ShortcutBadger;

/**
 * Created by behrad on 4/28/16.
 */
public class GcmMessageHandler extends GcmListenerService {

    public static final String TAG = GcmMessageHandler.class.getName();

    private static final boolean CONNECT_FOR_GCM = false;

    private static int NOTIFICATION_ID = 0;

    private EventBus eventBus = EventBus.getDefault();

    private static final ScheduledExecutorService worker =
            Executors.newSingleThreadScheduledExecutor();

    public GcmMessageHandler() {
        eventBus.register(this);
    }


    @Override
    public void onMessageReceived(String from, Bundle data) {
        Logger.d(TAG, "@@@@@@@@@@@@@@@@@ FCM Notification on " + from + ": " + data);

        String deviceId = data.getString("deviceId");
        if( deviceId != null ) {
            String currentDeviceId = AdpPushClient.get().getInstallationId();
            if( !deviceId.equalsIgnoreCase(currentDeviceId) ) {
                Logger.w(TAG,
                    "Ignoring FCM, unmatched deviceId " + deviceId + " != " + currentDeviceId);
                return;
            }
        }
        // Silent FCM
        String alertText = data.getString("message");
        if( alertText == null || alertText.equalsIgnoreCase("") ) {
            Logger.i(TAG, "Silent FCM");
            return;
        }
        // Don't notify GCM for received messages
        final String messageId = data.getString("messageId");
        if( messageId != null && PushService.hasReceivedMessage(messageId) ) {
            Logger.i(TAG, "MessageId already received, Swallow FCM!");
            return;
        }

        final String pushPayload = data.getString("push");

        final String topicName = "app/" + AdpPushClient.get().getAppId()
                + "/user/" + AdpPushClient.get().getUserId()
                + "/" + data.getString("collapse_key"); //TODO collapse_key == public/*

        final ChabokNotification notification = new ChabokNotification(
            messageId,
            data.getString("messageFrom"),
            alertText,
            Integer.valueOf(data.getString("androidBadge", "0"))
                    + getUnseenBadge(),
            data
        );
        notification.setTopicName(data.getString("collapse_key"));

        try {
            if (AdpPushClient.get().isForeground()) {
                Log.w(TAG, "Don't notify GCM when app is foreground!");
                // Ignore GCM
            } else {
                // app is background
                try {
                    // Don't notify GCM for received messages
                    if( messageId != null ) {
                        if( PushService.hasReceivedMessage(messageId) ||
                            AdpPushClient.hasNotified(messageId) ) {
                            // Ignore GCM
                            return;
                        }
                    }

                    final Class activityToOpen = AdpPushClient.get().getNotifActivityClass(data);
                    final Context ctx = this;

                    int delay = 0;
                    if( CONNECT_FOR_GCM ) {
                        PushService.performAction(getApplicationContext(), "START");
                        delay = 15;
                    }
                    Runnable notifTask = new Runnable() {
                        public void run() {
                            if(pushPayload != null) {
                                Logger.w(TAG, "We have push payload... handle as a new chabok in-app message!!");
                                Set<String> inAppMsgs = getSharedPreferences().getStringSet("pendingInAppMsgs", null);
                                if(inAppMsgs == null) {
                                    inAppMsgs = new HashSet<>();
                                }
                                inAppMsgs.add( topicName + "_BAHDRPA_" + pushPayload );
                                getSharedPreferences().edit().putStringSet(
                                        "pendingInAppMsgs",
                                        new HashSet<>(
                                            Arrays.asList(inAppMsgs.toArray(new String[inAppMsgs.size()]))
                                        )
                                ).apply();
                            }
                            GcmMessageHandler.sendNotification(ctx, activityToOpen, notification);
                        }
                    };
                    worker.schedule(notifTask, delay, TimeUnit.SECONDS);

                } catch (ClassNotFoundException e) {
                    Log.e(TAG, "cannot send notification when no subscriber for Intents", e);
                }
            }
        } catch(Exception e) {
            Logger.e(TAG, "Error handling GCM ", e);
            GcmMessageHandler.sendNotification(this, null, notification);
        }
    }


    public void onEvent(createNotificationEvent event) {
        Log.w(TAG, "Should? Notify GCM message (canNotify,isCanceled): " + event.isCanNofity() + "," + event.isCanceled());
        if( !event.isCanceled() ) {
            sendNotification(event);
        }
    }

    private void sendNotification(createNotificationEvent event) {
        ChabokNotification gcmNotif = new ChabokNotification(
            event.getIntent().getExtras().getString("messageId"),
            event.getIntent().getExtras().getString("messageFrom"),
            event.getIntent().getExtras().getString("message"),
            Integer.valueOf(event.getIntent().getExtras().getString("androidBadge", "0"))
                    + getUnseenBadge(),
            event.getIntent().getExtras()
        );
        sendNotification(this, event.getClient().getActivityClass(), gcmNotif);
    }

    // Put the message into a notification and post it.
    @TargetApi(Build.VERSION_CODES.JELLY_BEAN)
    public static void sendNotification(Context ctx, Class viewClass, ChabokNotification msg ) {
        try {
            PackageInfo packageInfo = ctx.getPackageManager().getPackageInfo(ctx.getPackageName(), 0);
            ApplicationInfo appInfo = ctx.getPackageManager().getApplicationInfo(packageInfo.packageName, 0);
            NotificationManagerCompat mNotificationManager = NotificationManagerCompat.from(ctx);
            Intent intent;
            if( viewClass != null ) {
                intent = new Intent(ctx, viewClass);
            } else {
                intent = new Intent();
            }
            intent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);
            intent.putExtra("CHABOK_NOTIFICATION", true);
            if( viewClass != null ) {
                intent.putExtra("CHABOK_NOTIFICATION_HANDLER", viewClass.getName());
            }

            String msgId = msg.getId();
            if(msg.getExtras() != null) {
                Log.i(TAG, "Notification Extras " + msg.getExtras());
                intent.putExtras(msg.getExtras());
                if(msg.getExtras().getString("messageId") != null) {
                    msgId = msg.getExtras().getString("messageId");
                }
            }

            if (msg.getMessage() != null) {
                msg.setTopicName(msg.getMessage().getTopicName());
                intent.putExtra("collapse_key", msg.getMessage().getTopicName());
                intent.putExtra("messageId", msg.getMessage().getId());
                if (msg.getMessage().getData() != null) {
                    intent.putExtra("data", msg.getMessage().getData().toString());
                }
                intent.putExtra("live", msg.getMessage().isLive());
                intent.putExtra("stateful", msg.getMessage().isStateful());
                intent.putExtra("inapp", msg.getMessage().isInApp());
                intent.putExtra("topicName", msg.getMessage().getTopicName());
                intent.putExtra("expireAt", msg.getMessage().getExpireAt());
                intent.putExtra("receivedAt", msg.getMessage().getReceivedAt());
            }

            if(msgId != null) {
                AdpPushClient.get().addNotifiedMessage(msgId);
            }

            int uniqueInt = (int) (System.currentTimeMillis() & 0xfffffff);
            PendingIntent contentIntent = PendingIntent.getActivity(
                    ctx,
                    uniqueInt,
                    intent,
                    PendingIntent.FLAG_UPDATE_CURRENT
            );

            NotificationCompat.Builder mBuilder = new NotificationCompat.Builder(ctx)
                    .setAutoCancel(true)
                    .setPriority(Notification.PRIORITY_HIGH)
                    .setContentTitle(appInfo.nonLocalizedLabel)
                    .setTicker(msg.getText())
                    .setContentText(msg.getText())
//                .setSubText(msg.getText())
                    .setContentIntent(contentIntent);

            mBuilder.setSmallIcon(getNotificationIcon());

            Bitmap bm = BitmapFactory.decodeResource(ctx.getResources(), appInfo.icon);
            mBuilder.setLargeIcon(bm);

            HashMap<String, HashMap> settings = AdpPushClient.get().getNotificationSettings();

            if(settings != null && settings.containsKey(msg.getTopicName())) {
                msg.setAlert(Boolean.getBoolean(settings.get(msg.getTopicName()).get("alert").toString()));
                if (settings.get(msg.getTopicName()).get("sound") != null) {
                    msg.setSound((String)settings.get(msg.getTopicName()).get("sound"));
                }
            } else {
                if (settings != null) {
                    Logger.d(TAG, "No Notification settings? " + Arrays.toString(settings.keySet().toArray()) + ", " + msg.getTopicName());
                } else {
                    Logger.d(TAG, "No Notification settings = null " + msg.getTopicName());
                }
            }

            if (msg.isAlert()) {
                if(msg.getSound() != null) {
                    mBuilder.setSound(
                        Uri.parse("android.resource://" + ctx.getPackageName() + "/" + msg.getSound())
                    );
                } else {
                    mBuilder.setSound(
                        RingtoneManager.getActualDefaultRingtoneUri(
                            ctx, RingtoneManager.TYPE_NOTIFICATION
                        )
                    );
                }
            }


            if( Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP ) { // Android 5.0 or higher
                mBuilder.setVisibility(Notification.VISIBILITY_PUBLIC);
            }

            Notification notif = mBuilder.build();

            //TODO read message's badge count
            AdpPushClient.get().incBadge();
            ShortcutBadger.applyNotification(ctx, notif, AdpPushClient.get().getBadge());

            if(AdpPushClient.get().prepareNotification(msg, mBuilder)) {
                mNotificationManager.notify(++NOTIFICATION_ID, notif);
            }
        } catch( Exception e ) {
            Log.e(TAG, "Error notifying user ", e);
        }
    }


    private static int getNotificationIcon() {
        boolean useSilhouette = Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP;
        return useSilhouette ?
            AdpPushClient.get().getNotificationIconSilhouette()
            :
            AdpPushClient.get().getNotificationIcon();
    }


    private int getUnseenBadge() {
        return getApplicationContext().getSharedPreferences(
                LocalInstallation.SHARED_PREFERENCES_NAME,
                Context.MODE_PRIVATE
        ).getInt( "androidUnseenBadge", 0 );
    }

    private SharedPreferences getSharedPreferences() {
        return getApplicationContext().getSharedPreferences(
                LocalInstallation.SHARED_PREFERENCES_NAME, Context.MODE_PRIVATE
        );
    }


    @Override
    public void onDestroy() {
        eventBus.unregister( this );
        super.onDestroy();
    }

}