package cn.lollypop.android.thermometer.ble.utils;

import java.util.Calendar;

import com.basic.util.ByteUtil;
import com.basic.util.TimeUtil;
import com.orhanobut.logger.Logger;

public class HealthThermometerServiceUtil {

  private static final int HIDE_MSB_8BITS_OUT_OF_32BITS = 0x00FFFFFF;
  private static final int HIDE_MSB_8BITS_OUT_OF_16BITS = 0x00FF;
  private static final int SHIFT_LEFT_8BITS = 8;
  private static final int SHIFT_LEFT_16BITS = 16;
  private static final int GET_BIT24 = 0x00400000;
  private static final int FIRST_BIT_MASK = 0x01; //
  private static final int SECOND_BIT_MASK = 0x02; //是否带时间
  private static final int THIRD_BIT_MASK = 0x04; //是否带类型

  public static boolean checkOver(byte[] data) {
    return checkSpecialValue(data, (byte) 0xFF);
  }

  public static boolean checkMeasuring(byte[] data) {
    return checkSpecialValue(data, (byte) 0xEE);
  }

  private static boolean checkSpecialValue(byte[] data, byte v) {
    boolean flag = true;
    if (data != null && data.length > 4) {
      for (int i = 1; i <= 3; i++) {
        if (data[i] != v) {
          flag = false;
        }
      }
    } else {
      flag = false;
    }
    return flag;
  }

  public static int getTimestamp(byte[] data) {
    if (data.length > 0) {
      byte flag = data[0];
      if ((flag & SECOND_BIT_MASK) != 0) { //带时间
        //去掉温度四个字节
        int year = ByteUtil.getShort(new byte[]{data[5], data[6]});
        int month = data[7];
        int day = data[8];
        int hour = data[9];
        int min = data[10];
        int sec = data[11];
        Calendar calendar = Calendar.getInstance();
        calendar.set(year, month - 1, day, hour, min, sec);
        return TimeUtil.getTimestamp(calendar.getTimeInMillis());
      }
    }
    return 0;
  }

  public static int getValue(byte[] data) {
    return decodeTemperature(data);
  }

  /**
   * This method decode temperature value received from Health Thermometer
   * device First byte {0} of data is flag and first bit of flag shows unit
   * information of temperature. if bit 0 has value 1
   * then unit is Fahrenheit and Celsius otherwise Four bytes {1 to 4} after
   * Flag bytes represent the temperature value in IEEE-11073 32-bit
   * Float format
   */
  public static int decodeTemperature(byte[] data) {
    double temperatureValue;
    byte flag = data[0];
    byte exponential = data[4];
    short firstOctet = convertNegativeByteToPositiveShort(data[1]);
    short secondOctet = convertNegativeByteToPositiveShort(data[2]);
    short thirdOctet = convertNegativeByteToPositiveShort(data[3]);
    int mantissa = ((thirdOctet << SHIFT_LEFT_16BITS)
        | (secondOctet << SHIFT_LEFT_8BITS)
        | (firstOctet)) & HIDE_MSB_8BITS_OUT_OF_32BITS;
    mantissa = getTwosComplimentOfNegativeMantissa(mantissa);
    temperatureValue = (mantissa * Math.pow(10, exponential));

    /**
     * Conversion of temperature unit from Fahrenheit to Celsius
     * if unit is in Fahrenheit.
     * Celsius = (98.6*Fahrenheit -32) 5/9
     */
    if ((flag & FIRST_BIT_MASK) != 0) {
      temperatureValue = (98.6 * temperatureValue - 32) * (5 / 9.0);
    }
    return (int) Math.round(temperatureValue * 100);
  }

  private static short convertNegativeByteToPositiveShort(byte octet) {
    if (octet < 0) {
      return (short) (octet & HIDE_MSB_8BITS_OUT_OF_16BITS);
    } else {
      return octet;
    }
  }

  private static int getTwosComplimentOfNegativeMantissa(int mantissa) {
    if ((mantissa & GET_BIT24) != 0) {
      return ((((~mantissa) & HIDE_MSB_8BITS_OUT_OF_32BITS) + 1) * (-1));
    } else {
      return mantissa;
    }
  }

  public static int getCalculateFlag(byte[] data) {
    if (data == null || data.length <= 11) {
      return 0;
    }

    byte c4 = data[12];
    if ((c4 & 0x80) != 0) { //calculate
      int time = c4 & 0x7f;
      Logger.i("ble calculate flag : " + time);
      return time;
    }

    return 0;
  }

  public static int getSmartThermoValue(byte[] data) {
    double temperatureValue;
    byte exponential = data[4];
    short firstOctet = convertNegativeByteToPositiveShort(data[1]);
    short secondOctet = convertNegativeByteToPositiveShort(data[2]);
    short thirdOctet = convertNegativeByteToPositiveShort(data[3]);
    int mantissa = ((thirdOctet << SHIFT_LEFT_16BITS)
        | (secondOctet << SHIFT_LEFT_8BITS)
        | (firstOctet)) & HIDE_MSB_8BITS_OUT_OF_32BITS;
    mantissa = getTwosComplimentOfNegativeMantissa(mantissa);
    temperatureValue = (mantissa * Math.pow(10, exponential));
    return (int) Math.round(temperatureValue * 100);
  }

  public static int getSmartThermoUserId(byte[] data) {
    if (data == null || data.length <= 11) {
      return 0;
    }
    return data[12];
  }
}