package cn.lollypop.android.thermometer.ble.action;

import java.util.Map;
import java.util.UUID;

import android.bluetooth.BluetoothGatt;
import android.bluetooth.BluetoothGattCharacteristic;
import android.bluetooth.BluetoothGattDescriptor;
import com.orhanobut.logger.Logger;

import cn.lollypop.android.thermometer.ble.BleAutoConnect;
import cn.lollypop.android.thermometer.ble.Constants;
import cn.lollypop.android.thermometer.ble.GattAttributes;
import cn.lollypop.android.thermometer.ble.action.request.BleRequest;
import cn.lollypop.android.thermometer.ble.action.request.BleRequestActionType;
import cn.lollypop.android.thermometer.ble.action.request.BleRequestValueType;
import cn.lollypop.android.thermometer.ble.action.response.BleResponse;

/**
 * Copyright (c) 2015, Bongmi
 * All rights reserved
 * Author: wangjunjie@bongmi.com
 */
public class BleGattInteraction {

  public boolean doInteractive(BluetoothGatt bluetoothGatt,
                               BleRequest bleRequest,
                               Map<UUID, BluetoothGattCharacteristic>
                                   characteristics,
                               BleAutoConnect.IResponse response) {
    BluetoothGattCharacteristic characteristic =
        characteristics.get(bleRequest.getCharacteristicUUID());
    if (characteristic == null) { //特征值不存在
      return false;
    }

    //根据valueType设置value
    if (bleRequest.getValueType().equals(BleRequestValueType.BYTE)) {
      characteristic.setValue(bleRequest.getValue());
    } else if (bleRequest.getValueType().equals(BleRequestValueType.UINT16)) {
      characteristic.setValue(bleRequest.getValueInt(),
          BluetoothGattCharacteristic.FORMAT_UINT16,
          bleRequest.getOffset());
    }

    //根据actionType执行action
    if (bleRequest.getActionType().equals(BleRequestActionType.READ)) {
      readCharacteristic(bluetoothGatt, characteristic);
    } else if (bleRequest.getActionType().equals(BleRequestActionType.WRITE)) {
      writeCharacteristic(bluetoothGatt, characteristic);
    } else if (bleRequest.getActionType().equals(
        BleRequestActionType.SET_NOTIFICATION)) {
      boolean setNotification = setCharacteristicNotification(
          bluetoothGatt, characteristic, bleRequest.isValueBoolean()
      );
      // 设置结束返回信息
      BleResponse bleResponse = new BleResponse(
          characteristic, setNotification ? 1 : 0);
      response.receiveDataFromDevice(Constants.ACTION_GATT_WRITE, bleResponse);
    } else if (bleRequest.getActionType().equals(
        BleRequestActionType.SET_INDICATION)) {
      setIndication(bluetoothGatt, characteristic, bleRequest.getValue());
    }
    return true;
  }

  /**
   * Request a read on a given {@code BluetoothGattCharacteristic}.
   * The read result is reported
   * asynchronously through the
   * {@code BluetoothGattCallback#onCharacteristicRead(
   * android.bluetooth.BluetoothGatt,
   * android.bluetooth.BluetoothGattCharacteristic, int)}
   * callback.
   *
   * @param characteristic The characteristic to read from.
   */
  private void readCharacteristic(BluetoothGatt bluetoothGatt,
                                  BluetoothGattCharacteristic characteristic) {
    if (bluetoothGatt == null) {
      Logger.e("bluetoothGatt is null");
      return;
    }

    bluetoothGatt.readCharacteristic(characteristic);
  }

  private void writeCharacteristic(BluetoothGatt bluetoothGatt,
                                   BluetoothGattCharacteristic characteristic) {
    if (bluetoothGatt == null) {
      Logger.e("bluetoothGatt is null");
      return;
    }

    bluetoothGatt.writeCharacteristic(characteristic);
  }

  /**
   * Enables or disables notification on a give characteristic.
   *
   * @param c Characteristic to act on.
   * @param enabled        If true, enable notification.  False otherwise.
   */
  private boolean setCharacteristicNotification(BluetoothGatt bluetoothGatt,
                                                BluetoothGattCharacteristic c,
                                                boolean enabled) {
    if (bluetoothGatt == null) {
      Logger.e("bluetoothGatt is null");
      return false;
    }

    return bluetoothGatt.setCharacteristicNotification(c, enabled);
  }

  private void setIndication(BluetoothGatt bluetoothGatt,
                             BluetoothGattCharacteristic characteristic,
                             byte[] value) {
    if (bluetoothGatt == null) {
      Logger.e("bluetoothGatt is null");
      return;
    }

    BluetoothGattDescriptor descriptor = characteristic.getDescriptor(
        UUID.fromString(GattAttributes.CLIENT_CHARACTERISTIC_CONFIG));
    descriptor.setValue(value);
    bluetoothGatt.writeDescriptor(descriptor);
  }
}
