package cn.lollypop.android.thermometer.ble;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.UUID;

import android.app.Service;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothGattService;
import android.bluetooth.BluetoothManager;
import android.content.Context;
import android.content.Intent;
import android.os.Binder;
import android.os.Handler;
import android.os.IBinder;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import com.basic.util.Callback;
import com.basic.util.ClsUtil;
import com.basic.util.TimeUtil;
import com.orhanobut.logger.Logger;

import cn.lollypop.android.thermometer.ble.action.BleActionImpl;
import cn.lollypop.android.thermometer.ble.action.BleActionManager;
import cn.lollypop.android.thermometer.ble.action.IBleAction;
import cn.lollypop.android.thermometer.ble.action.request.ReadDeviceTimeRequest;
import cn.lollypop.android.thermometer.ble.action.response.BleResponse;
import cn.lollypop.android.thermometer.ble.action.response.BleResponseType;
import cn.lollypop.android.thermometer.ble.exceptions.BleException;
import cn.lollypop.android.thermometer.ble.model.AlarmTimeModel;
import cn.lollypop.android.thermometer.ble.model.Temperature;
import cn.lollypop.android.thermometer.ble.ota.OtaManager;
import cn.lollypop.android.thermometer.ble.utils.BatteryServiceUtil;
import cn.lollypop.android.thermometer.ble.utils.DeviceInformationServiceUtil;
import cn.lollypop.android.thermometer.ble.utils.HealthThermometerServiceUtil;

/**
 * Copyright (c) 2015, Bongmi
 * All rights reserved
 * Author: wangjunjie@bongmi.com
 */
public class BleAutoConnectService extends Service {

  private final BleAutoConnect bleAutoConnect = new BleAutoConnect();
  private final Handler connectTimeoutHandler = new Handler();

  private BleScan bleScan;
  private BleActionManager bleActionManager;
  private BluetoothAdapter bluetoothAdapter;
  private String deviceAddress;
  private boolean isConnected;
  private boolean originalTemperatureSwitcher = true;
  private final OtaManager otaManager = new OtaManager();

  private final Handler mainThreadHandler = new Handler();
  private final Object bleCallbackLock = new Object();
  private final List<BleCallback> bleCallbackList = new ArrayList<>();

  private final IBinder binder = new LocalBinder();

  public class LocalBinder extends Binder {
    public BleAutoConnectService getService() {
      return BleAutoConnectService.this;
    }
  }

  @Nullable
  @Override
  public IBinder onBind(Intent intent) {
    return binder;
  }

  @Override
  public void onCreate() {
    super.onCreate();

    BluetoothManager bluetoothManager =
        (BluetoothManager) getSystemService(Context.BLUETOOTH_SERVICE);
    if (bluetoothManager == null) {
      Logger.i("Unable to initialize BluetoothManager.");
      return;
    }

    bluetoothAdapter = bluetoothManager.getAdapter();
    if (bluetoothAdapter == null) {
      Logger.i("Unable to obtain a BluetoothAdapter.");
      return;
    }

    bleActionManager = new BleActionManager(this, response, createBleAction());
  }

  protected IBleAction createBleAction() {
    return new BleActionImpl();
  }

  public boolean startScan(String deviceName) {
    if (bluetoothAdapter == null) {
      Logger.i("BluetoothAdapter is null");
      return false;
    }

    if (bleScan == null) {
      bleScan = new BleScan(bluetoothAdapter, deviceName);
    }

    bleScan.doScan(new BleCallback() {
      @Override
      public void callback(BleStatus state, Object object) {
        switch (state) {
          case SCAN_SUC:
            BluetoothDevice device = (BluetoothDevice) object;
            try {
              connect(device.getAddress());
            } catch (BleException e) {
              e.printStackTrace();
            }
            break;
          default:
            break;
        }
      }
    });
    return true;
  }

  public void stopScan() {
    if (bleScan == null) {
      return;
    }

    bleScan.stopScan();
  }

  public void connect(String address) throws BleException {
    if (TextUtils.isEmpty(address)) {
      throw new BleException("bluetooth address is null");
    }

    if (isConnected) {
      return;
    }

    boolean checkAddress = bleAutoConnect.connect(
        BleAutoConnectService.this, bluetoothAdapter, address,
        changeConnectState, response
    );

    if (!checkAddress) {
      doBleCallback(BleCallback.BleStatus.DISCONNECTED, null);
    }
    deviceAddress = address;
  }

  public void destroy() {
    Logger.i("ble connect destroyed");
    if (bleActionManager == null) {
      return;
    }

    stopScan();
    isConnected = false;
    deviceAddress = null;
    bleActionManager.end();
    bleAutoConnect.destroy();

    DeviceInformationServiceUtil.setFirmwareVersion(this, "");
    BatteryServiceUtil.setBatteryLevel(this, 0);
    DeviceInformationServiceUtil.setSN(this, "");
    DeviceInformationServiceUtil.setHardwareVersion(this, "");
  }

  /**
   * 设备重连.
   */
  public void reconnect() {
    bleAutoConnect.delayReconnect();
  }

  private BleAutoConnect.IChangeConnectState changeConnectState =
      new BleAutoConnect.IChangeConnectState() {
        @Override
        public void change(BleAutoConnect.ConnectState connectState) {
          switch (connectState) {
            case CONNECTED:
              Logger.i("ble connect successfully and handle services");
              isConnected = true;
              doBleCallback(BleCallback.BleStatus.CONNECTED, deviceAddress);
              bleActionManager.start(
                  bleAutoConnect.getBluetoothGatt(),
                  originalTemperatureSwitcher,
                  new Callback() {
                    @Override
                    public void doCallback(Boolean result, Object obj) {
                      otaManager.initCharacteristic(
                          (BluetoothGattService) obj);
                    }
                  }
              );
              break;
            case DIS_CONNECT:
              if (isConnected) {
                Logger.i("ble disconnect");
              }
              isConnected = false;
              doBleCallback(BleCallback.BleStatus.DISCONNECTED, null);
              bleActionManager.end();
              break;
            default:
              break;
          }
        }
      };

  private final BleAutoConnect.IResponse response = new BleAutoConnect.IResponse() {
    @Override
    public void receiveDataFromDevice(String action, BleResponse bleResponse) {
      String uuid = null;
      byte[] data = new byte[0];
      if (bleResponse.getType().equals(BleResponseType.CHARACTERISTIC)) {
        //ota暂时单独判断
        if (action.equals(Constants.ACTION_GATT_WRITE)) {
          if (otaManager.ackWrite(bleResponse.getCharacteristic(),
              bleResponse.getStatus())) {
            return;
          }
        } else if (action.equals(Constants.ACTION_DATA_AVAILABLE)) {
          if (otaManager.ackRead(bleResponse.getCharacteristic(),
              bleResponse.getStatus())) {
            return;
          }
        }

        data = bleResponse.getCharacteristic().getValue();
        uuid = bleResponse.getCharacteristic().getUuid().toString();
      } else if (bleResponse.getType().equals(BleResponseType.DESCRIPTOR)) {
        data = bleResponse.getDescriptor().getValue();
        uuid = bleResponse.getDescriptor().getCharacteristic()
            .getUuid().toString();
      }
      if (action.equals(Constants.ACTION_GATT_WRITE)
          || action.equals(Constants.ACTION_DATA_AVAILABLE)) { //直接处理数据

        handleReceiveData(action, uuid, data);
      }
    }

    @Override
    public void refreshView(BleCallback.BleStatus bleState) {
      if (bleState.equals(BleCallback.BleStatus.OPERATE_FAIL)) {
        reconnect();
      }
      doBleCallback(bleState, null);
    }
  };

  @Override
  public void onDestroy() {
    super.onDestroy();
    destroy();
  }

  private void handleReceiveData(String action, String uuidStr, byte[] data) {
    UUID uuid = UUID.fromString(uuidStr);
    if (action.equals(Constants.ACTION_DATA_AVAILABLE)) {
      if (Constants.UUID_TEMPERATE_MEASUREMENT.equals(uuid)) { //收到温度值
        receiveTemperature(data);
        return;
      } else if (Constants.UUID_SAMPLE_DATA.equals(uuid)) { //收到测温过程数据
        receiveSampleData(data);
        return;
      }
    }

    if (bleActionManager.getCurBleRequest() == null) {
      return;
    }

    boolean analyze = bleActionManager.getCurBleRequest()
        .analyzeData(this, data);
    doBleCallback(
        bleActionManager.getCurBleRequest().getCurrentBleStatus(),
        bleActionManager.getCurBleRequest().getObject());
    if (!analyze) {
      //返回数据有问题
      bleActionManager.reDoAction();
      return;
    }

    bleActionManager.doNextAction();
  }

  protected void receiveTemperature(byte[] data) {
    int temperature = HealthThermometerServiceUtil.getValue(data);
    int calculateFlag = HealthThermometerServiceUtil.getCalculateFlag(data);
    int measureTimestamp = HealthThermometerServiceUtil.getTimestamp(data);

    //compensate time offset caused by restart of thermometer
    Calendar calendar = Calendar.getInstance();
    calendar.add(Calendar.DAY_OF_MONTH, 1);
    int currentTime = TimeUtil.getTimestamp(calendar.getTime().getTime());
    if (measureTimestamp < Constants.TIME_LOWER_LIMIT
        || measureTimestamp > currentTime) {
      measureTimestamp += ReadDeviceTimeRequest.deviceTimeOffset;
    }

    String content = String.format(
        "ble receive data: %s, temperature: %d, time: %s",
        ClsUtil.Bytes2HexString(data),
        temperature,
        TimeUtil.getTimeInMillisShow(TimeUtil.getTimeInMillis(measureTimestamp))
    );
    Logger.i(content);

    Temperature temp = new Temperature();
    temp.setCalculateFlag(calculateFlag);
    temp.setTemperatureInt(temperature);
    temp.setMeasureTimestamp(measureTimestamp);

    doBleCallback(BleCallback.BleStatus.MEASURE_GET, temp);
  }

  private void receiveSampleData(byte[] data) {
    doBleCallback(BleCallback.BleStatus.MEASURE_SAMPLE_GET, data);
  }

  /**
   * 是否已连接设备
   *
   * @return boolean
   */
  public boolean isConnected() {
    return isConnected;
  }

  /**
   * 设置超时，5分钟后休眠体温计.
   */
  public void setTimeout() {
    connectTimeoutHandler.postDelayed(
        connectTimeoutRunnable, Constants.CONNECT_TIMEOUT
    );
    bleAutoConnect.removeReconnect();
  }

  /**
   * 取消超时设置.
   */
  public void clearTimeout() {
    if (!isConnected() && !TextUtils.isEmpty(deviceAddress)) {
      bleAutoConnect.delayReconnect();
    }
    connectTimeoutHandler.removeCallbacks(connectTimeoutRunnable);
  }

  private final Runnable connectTimeoutRunnable = new Runnable() {
    @Override
    public void run() {
      if (otaManager.isTransferring()) {
        return;
      }

      if (isConnected()) {
        Logger.i("ble make device go into sleep model");
        makeDeviceSleep();
      }
    }
  };

  /**
   * 设置是否收集过程温度
   *
   * @param originalTemperatureSwitcher true or false
   */
  public void setOriginalTemperatureSwitcher(
      boolean originalTemperatureSwitcher) {
    this.originalTemperatureSwitcher = originalTemperatureSwitcher;
  }

  public void registerBleCallback(BleCallback bleCallback) {
    synchronized (bleCallbackLock) {
      bleCallbackList.add(bleCallback);
    }
  }

  public void unregisterBleCallback(BleCallback bleCallback) {
    synchronized (bleCallbackLock) {
      for (BleCallback callback : bleCallbackList) {
        if (callback.hashCode() == bleCallback.hashCode()) {
          bleCallbackList.remove(callback);
          break;
        }
      }
    }
  }

  protected void doBleCallback(final BleCallback.BleStatus bleState,
                             final Object object) {
    if (bleState == null) {
      return;
    }

    mainThreadHandler.post(new Runnable() {
      @Override
      public void run() {
        synchronized (bleCallbackLock) {
          for (BleCallback bleCallback : bleCallbackList) {
            bleCallback.callback(bleState, object);
          }
        }
      }
    });
  }

  public void startOta(String version) {
    otaManager.start(
        this, version, bleAutoConnect.getBluetoothGatt(),
        new BleCallback() {
          @Override
          public void callback(BleStatus state, Object object) {
            doBleCallback(state, object);
          }
        });
  }

  public void makeDeviceSleep() {
    bleActionManager.makeDeviceSleep();
  }

  public void getBatteryLevel() {
    bleActionManager.getBatteryLevel();
  }

  public void triggerBeep() {
    bleActionManager.triggerBeep();
  }

  public void enterDebug() {
    bleActionManager.enterDebug();
  }

  public void quitDebug() {
    bleActionManager.quitDebug();
  }

  public void doOta() {
    bleActionManager.doOta();
  }

  public void startMeasure(boolean isSwitchOriginalTemperature) {
    bleActionManager.startMeasure(isSwitchOriginalTemperature);
  }

  public void setAlarm(AlarmTimeModel alarmTimeModel) {
    bleActionManager.setAlarm(alarmTimeModel);
  }

  public void refreshCache() {
    bleAutoConnect.refreshCache();
  }
}
