package cn.lollypop.android.thermometer.ble;

import java.lang.reflect.Method;

import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothGatt;
import android.bluetooth.BluetoothGattCallback;
import android.bluetooth.BluetoothGattCharacteristic;
import android.bluetooth.BluetoothGattDescriptor;
import android.bluetooth.BluetoothManager;
import android.bluetooth.BluetoothProfile;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.os.Build;
import android.os.Handler;
import com.orhanobut.logger.Logger;

import cn.lollypop.android.thermometer.ble.action.response.BleResponse;

/**
 * 先用Direct连接（一般6秒内连接成功，小米的有10多秒），20秒连接超时后设置autoConnect（发现设备自动连接）。
 * 某些设备某些情况下自动连接失效，打开app时重复上面的步骤。
 * Copyright (c) 2015, Bongmi
 * All rights reserved
 * Author: wangjunjie@bongmi.com
 */
public class BleAutoConnect {
  public enum ConnectState {
    CONNECTED,
    DIS_CONNECT
  }

  private final Handler mainThreadHandler = new Handler();
  private final Object gattLock = new Object();

  private Context context;
  private BluetoothAdapter bluetoothAdapter;
  private BluetoothDevice device;
  protected BluetoothGatt bluetoothGatt;

  private IChangeConnectState changeConnectState;
  private IResponse response;
  private boolean autoConnect;
  private boolean isReconnecting;

  public boolean connect(Context context,
                         BluetoothAdapter bluetoothAdapter,
                         String deviceAddress,
                         IChangeConnectState changeConnectState,
                         IResponse response) {
    this.context = context;
    this.bluetoothAdapter = bluetoothAdapter;
    this.changeConnectState = changeConnectState;
    this.response = response;
    if (!BluetoothAdapter.checkBluetoothAddress(deviceAddress)) {
      Logger.e("ble connect with wrong address：" + deviceAddress);
      return false;
    }

    device = bluetoothAdapter.getRemoteDevice(deviceAddress);
    context.registerReceiver(
        blueStateBroadcastReceiver,
        new IntentFilter(BluetoothAdapter.ACTION_STATE_CHANGED)
    );

    return reconnect();
  }

  private final BroadcastReceiver blueStateBroadcastReceiver =
      new BroadcastReceiver() {

        @Override
        public void onReceive(Context context, Intent intent) {
          int blueState = intent.getIntExtra(BluetoothAdapter.EXTRA_STATE, 0);
          switch (blueState) {
            case BluetoothAdapter.STATE_OFF:
              Logger.i("ble bleAutoConnect blueState: STATE_OFF");
              disconnect();
              break;
            case BluetoothAdapter.STATE_TURNING_ON:
              break;
            case BluetoothAdapter.STATE_ON:
              Logger.i("ble bleAutoConnect blueState: STATE_ON");
              reconnect();
              break;
            case BluetoothAdapter.STATE_TURNING_OFF:
              break;
            default:
              break;
          }
        }
      };

  public void delayReconnect() {
    mainThreadHandler.postDelayed(delayConnectRunnable, 2000);
  }

  public void removeReconnect() {
    mainThreadHandler.removeCallbacks(delayConnectRunnable);
  }

  private Runnable delayConnectRunnable = new Runnable() {

    @Override
    public void run() {
      if (device == null) {
        return;
      }

      BluetoothManager bluetoothManager =
          (BluetoothManager) context.getSystemService(
              Context.BLUETOOTH_SERVICE);
      int state = bluetoothManager.getConnectionState(
          device, BluetoothProfile.GATT);
      if (state != BluetoothProfile.STATE_CONNECTED) {
        reconnect();
      }
    }
  };

  private boolean reconnect() {
    Logger.d("ble isReconnecting : " + isReconnecting);
    if (!bluetoothAdapter.isEnabled()) {
      return false;
    }

    if (device == null) {
      return false;
    }

    if (isReconnecting) {
      return true;
    }

    isReconnecting = true;

    disconnect();

    mainThreadHandler.postDelayed(new Runnable() {
      @Override
      public void run() {
        synchronized (gattLock) {
          if (!bluetoothAdapter.isEnabled() || device == null) {
            return;
          }

          Logger.i("ble begin connect gatt");
          if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            bluetoothGatt = device.connectGatt(context, false, gattCallback,
                BluetoothDevice.TRANSPORT_LE);
          } else {
            bluetoothGatt = device.connectGatt(context, false, gattCallback);
          }

          mainThreadHandler.postDelayed(connectRunnable, 20000);
        }
      }
    }, 2000);
    return true;
  }

  private Runnable connectRunnable = new Runnable() {
    @Override
    public void run() {
      isReconnecting = false;

      if (bluetoothGatt != null) {
        bluetoothGatt.disconnect();
        autoConnect = bluetoothGatt.connect();
        Logger.d("ble autoConnect : " + autoConnect);
        return;
      }

      reconnect();
    }
  };

  public void refreshCache() {
    if (bluetoothGatt != null) {
      refreshDeviceCache(bluetoothGatt);
    }
  }

  public void destroy() {
    if (context != null && device != null) {
      context.unregisterReceiver(blueStateBroadcastReceiver);
    }
    device = null;
    isReconnecting = false;
    disconnect();
  }

  private void disconnect() {
    mainThreadHandler.removeCallbacks(connectRunnable);
    removeReconnect();
    mainThreadHandler.post(new Runnable() {
      @Override
      public void run() {
        synchronized (gattLock) {
          if (bluetoothGatt == null) {
            return;
          }

          bluetoothGatt.disconnect();
          refreshDeviceCache(bluetoothGatt);
          bluetoothGatt.close();
          bluetoothGatt = null;
          autoConnect = false;
        }
      }
    });
  }

  private boolean refreshDeviceCache(BluetoothGatt gatt) {
    try {
      Method localMethod = gatt.getClass().getMethod("refresh");
      if (localMethod != null) {
        return (boolean) localMethod.invoke(gatt);
      }
    } catch (Exception localException) {
      Logger.e("ble An exception occurs while refreshing device");
    }
    return false;
  }

  protected final BluetoothGattCallback gattCallback =
      new BluetoothGattCallback() {

        @Override
        public void onConnectionStateChange(BluetoothGatt gatt,
                                            int status,
                                            int newState) {
          if (newState == BluetoothProfile.STATE_CONNECTING
              || newState == BluetoothProfile.STATE_DISCONNECTING) {
            Logger.i("ble connection state change, new state : " + newState);
            return;
          }

          isReconnecting = false;

          if (newState == BluetoothProfile.STATE_CONNECTED &&
              status == BluetoothGatt.GATT_SUCCESS) {
            Logger.i("ble begin discoverServices, new state : " + newState);
            bluetoothGatt.discoverServices();
            mainThreadHandler.removeCallbacks(connectRunnable);
            removeReconnect();
          } else {
            Logger.e("ble connection state change, new state : " + newState);
            changeConnectState.change(ConnectState.DIS_CONNECT);
            if (!autoConnect) {
              mainThreadHandler.postDelayed(connectRunnable, 2000);
            }
          }
        }

        @Override
        public void onServicesDiscovered(BluetoothGatt gatt, final int status) {
          mainThreadHandler.post(new Runnable() {
            @Override
            public void run() {
              if (status == BluetoothGatt.GATT_SUCCESS) {
                Logger.i("ble services discovered success");
                changeConnectState.change(ConnectState.CONNECTED);
                removeReconnect();
              } else {
                Logger.e("ble services discovered error status : " + status);
                reconnect();
              }
            }
          });
        }

        @Override
        public void onCharacteristicRead(BluetoothGatt gatt,
                                         BluetoothGattCharacteristic charact,
                                         int status) {
          BleResponse bleResponse = new BleResponse(charact, status);
          response.receiveDataFromDevice(
              Constants.ACTION_DATA_AVAILABLE, bleResponse);
        }

        @Override
        public void onCharacteristicChanged(BluetoothGatt gatt,
                                            BluetoothGattCharacteristic c) {
          BleResponse bleResponse = new BleResponse(c,
              BluetoothGatt.GATT_SUCCESS);
          response.receiveDataFromDevice(
              Constants.ACTION_DATA_AVAILABLE, bleResponse);
        }

        @Override
        public void onCharacteristicWrite(BluetoothGatt gatt,
                                          BluetoothGattCharacteristic c,
                                          int status) {
          BleResponse bleResponse = new BleResponse(c, status);
          response.receiveDataFromDevice(
              Constants.ACTION_GATT_WRITE, bleResponse);
        }

        @Override
        public void onDescriptorWrite(BluetoothGatt gatt,
                                      BluetoothGattDescriptor descriptor,
                                      int status) {
          BleResponse bleResponse = new BleResponse(descriptor, status);
          response.receiveDataFromDevice(
              Constants.ACTION_GATT_WRITE, bleResponse);
        }
      };

  public BluetoothGatt getBluetoothGatt() {
    return bluetoothGatt;
  }

  public interface IChangeConnectState {
    void change(ConnectState connectState);
  }

  public interface IResponse {
    void receiveDataFromDevice(String type, BleResponse bleResponse);

    void refreshView(BleCallback.BleStatus bleState);
  }
}