package com.basic.widgets;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Rect;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.GradientDrawable;
import android.graphics.drawable.GradientDrawable.Orientation;
import android.util.AttributeSet;
import android.view.Gravity;
import android.view.SoundEffectConstants;
import android.view.View;

import com.base.libs.R;

public class WheelView extends TosGallery {
  /**
   * The selector.
   */
  private Drawable mSelectorDrawable = null;

  /**
   * The bound rectangle of selector.
   */
  private Rect mSelectorBound = new Rect();

  /**
   * The top shadow.
   */
  private GradientDrawable mTopShadow = null;

  /**
   * The bottom shadow.
   */
  private GradientDrawable mBottomShadow = null;

  /**
   * Shadow colors
   */
//  private static final int[] SHADOWS_COLORS = { 0xFF111111, 0x00AAAAAA, 0x00AAAAAA };
  private static final int[] SHADOWS_COLORS = {0xFFFFFFFF, 0x00FFFFFF, 0x00FFFFFF};

  /**
   * The constructor method.
   *
   * @param context
   * @param attrs
   */
  public WheelView(Context context, AttributeSet attrs) {
    super(context, attrs);

    this.setVerticalScrollBarEnabled(false);
    this.setSlotInCenter(true);
    this.setOrientation(VERTICAL);
    this.setGravity(Gravity.CENTER_HORIZONTAL);
    this.setUnselectedAlpha(1.0f);

    // This lead the onDraw() will be called.
    this.setWillNotDraw(false);

    TypedArray ta = context.obtainStyledAttributes(attrs, R.styleable.WheelView);
    int bg = ta.getResourceId(R.styleable.WheelView_bg, 0);
    int selector = ta.getResourceId(R.styleable.WheelView_selector, 0);
    ta.recycle();

    // The selector rectangle drawable.
    if (selector > 0) {
      this.mSelectorDrawable = getContext().getResources().getDrawable(selector);
    }
    this.mTopShadow = new GradientDrawable(Orientation.TOP_BOTTOM, SHADOWS_COLORS);
    this.mBottomShadow = new GradientDrawable(Orientation.BOTTOM_TOP, SHADOWS_COLORS);

    // The default background.
    if (bg > 0) {
      this.setBackgroundResource(bg);
    }

    // Disable the sound effect default.
    this.setSoundEffectsEnabled(false);
  }

  /**
   * Called by draw to draw the child views. This may be overridden by derived classes to gain
   * control just before its children are drawn (but after its own view has been drawn).
   */
  @Override
  protected void dispatchDraw(Canvas canvas) {
    super.dispatchDraw(canvas);

    // After draw child, we do the following things:
    // +1, Draw the center rectangle.
    // +2, Draw the shadows on the top and bottom.

    drawCenterRect(canvas);
    drawShadows(canvas);
  }

  /**
   * setOrientation
   */
  @Override
  public void setOrientation(int orientation) {
    if (HORIZONTAL == orientation) {
      throw new IllegalArgumentException("The orientation must be VERTICAL");
    }

    super.setOrientation(orientation);
  }

  /**
   * Call when the ViewGroup is layout.
   */
  @Override
  protected void onLayout(boolean changed, int l, int t, int r, int b) {
    super.onLayout(changed, l, t, r, b);

    int galleryCenter = getCenterOfGallery();
    View v = this.getChildAt(0);

    int height = (null != v) ? v.getMeasuredHeight() : 50;
    int top = galleryCenter - height / 2;
    int bottom = top + height;

    mSelectorBound.set(getPaddingLeft(), top, getWidth() - getPaddingRight(), bottom);
  }

  /**
   * @see TosGallery#setSelectedPositionInt(int)
   */
  @Override
  protected void selectionChanged() {
    super.selectionChanged();

    playSoundEffect(SoundEffectConstants.CLICK);
  }

  /**
   * Draw the selector drawable.
   *
   * @param canvas
   */
  private void drawCenterRect(Canvas canvas) {
    if (null != mSelectorDrawable) {
      mSelectorDrawable.setBounds(mSelectorBound);
      mSelectorDrawable.draw(canvas);
    }
  }

  /**
   * Draw the shadow
   *
   * @param canvas
   */
  private void drawShadows(Canvas canvas) {
    int height = (int) (2.0 * mSelectorBound.height());
    mTopShadow.setBounds(0, 0, getWidth(), height);
    mTopShadow.draw(canvas);

    mBottomShadow.setBounds(0, getHeight() - height, getWidth(), getHeight());
    mBottomShadow.draw(canvas);
  }

}