package com.basic.util;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.security.MessageDigest;
import java.text.DecimalFormat;
import java.text.ParseException;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Random;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import android.app.Activity;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.Intent;
import android.content.res.Resources;
import android.graphics.drawable.Drawable;
import android.net.Uri;
import android.os.Build;
import android.telephony.TelephonyManager;
import android.text.TextUtils;
import android.util.Log;
import android.view.inputmethod.InputMethodManager;
import com.orhanobut.logger.Logger;

import cn.lollypop.be.model.Language;

/**
 * 通用工具类
 * Created by BeyondBright on 2015-02-08.
 */
public class CommonUtil {

  /**
   * 验证邮箱地址是否正确
   */
  public static boolean isEmail(String email) {
    boolean flag;
    try {
      String check = "^\\s*\\w+(?:\\.?[\\w-]+)*@[a-zA-Z0-9]+(?:[-.][a-zA-Z0-9]+)*\\.[a-zA-Z]+\\s*$";
      Pattern regex = Pattern.compile(check);
      Matcher matcher = regex.matcher(email);
      flag = matcher.matches();
    } catch (Exception e) {
      Logger.i("验证邮箱地址错误");
      flag = false;
    }

    return flag;
  }

  /**
   * 数字判断
   *
   * @param number 字符串
   * @return true 数字
   */
  public static boolean isNumber(String number) {
    Pattern p = Pattern.compile("^[0-9]{1,}$");
    Matcher m = p.matcher(number);
    return m.matches();
  }

  private static long lastClickTime;

  /**
   * 通用判断
   *
   * @param telNum 手机号
   * @return true 手机号
   */
  public static boolean isMobilePhoneNum(String telNum) {
    String regex = "^((1[0-9]))\\d{9}$";
    Pattern p = Pattern.compile(regex, Pattern.CASE_INSENSITIVE);
    Matcher m = p.matcher(telNum);
    return m.matches();
  }

  /**
   * 字母、数字、特殊字符至少包含两种以上，8~18位
   *
   * @param words 密码
   * @return true 符合密码规则
   */
  public static boolean isPassword(String words) {
    //必须是字母或者数字，8位以上
    Pattern p = Pattern.compile("(?!^(\\d+|[a-zA-Z]+|[~!@#$%^&*?.]+)$)^[\\w~!@#$%\\^&*?.]{8,18}+$");
    Matcher m = p.matcher(words);
    return m.matches();
  }

  /**
   * 防止连续快速点击
   *
   * @return true 重复点击
   */
  public static boolean isFastDoubleClick() {
    long time = System.currentTimeMillis();
    long timeD = time - lastClickTime;
    if (0 < timeD && timeD < 300) {
      Log.d("TimeUtils", "相隔：" + timeD);
      return true;
    }
    lastClickTime = time;
    Log.d("TimeUtils", "上次点击事件：" + lastClickTime);
    return false;
  }

  /**
   * 防止重复点击
   *
   * @param dis 毫秒
   * @return true 重复点击
   */
  public static boolean isFastDoubleClick(long dis) {
    long time = System.currentTimeMillis();
    long timeD = time - lastClickTime;
    if (0 < timeD && timeD < dis) {
      return true;
    }

    lastClickTime = time;
    Log.d("TimeUtils", "上次点击事件：" + lastClickTime);
    return false;
  }

  /**
   * Hides the input method.
   */
  public static boolean hideInputMethod(Activity activity) {
    if (activity == null) {
      return false;
    }

    InputMethodManager imm = (InputMethodManager) activity.getSystemService(
        Context.INPUT_METHOD_SERVICE);
    if (activity.getCurrentFocus() != null) {
      imm.hideSoftInputFromWindow(
          activity.getCurrentFocus().getWindowToken(), 0
      );
      return true;
    }

    return false;
  }

  public static float pixelToDp(Context context, float val) {
    float density = context.getResources().getDisplayMetrics().density;
    return val * density;
  }

  public static String getHashedFileName(String url) {
    if (url == null || url.endsWith("/")) {
      return null;
    }

    String suffix = getSuffix(url);
    StringBuilder sb = null;

    try {
      MessageDigest digest = MessageDigest.getInstance("MD5");
      byte[] dstbytes = digest.digest(url.getBytes("UTF-8")); // GMaFroid uses UTF-16LE
      sb = new StringBuilder();
      for (int i = 0; i < dstbytes.length; i++) {
        sb.append(Integer.toHexString(dstbytes[i] & 0xff));
      }
    } catch (Exception e) {
      e.printStackTrace();
    }

    if (null != sb && null != suffix) {
      return sb.toString() + "." + suffix;
    }

    return null;
  }

  private static String getSuffix(String fileName) {
    int dot_point = fileName.lastIndexOf(".");
    int sl_point = fileName.lastIndexOf("/");
    if (dot_point < sl_point) {
      return "";
    }

    if (dot_point != -1) {
      return fileName.substring(dot_point + 1);
    }

    return null;
  }

  /**
   * 去评分
   *
   * @param context context
   */
  public static void gotoScore(Context context) throws ActivityNotFoundException {
    Uri uri = Uri.parse("market://details?id=" + context.getPackageName());
    Intent intent = new Intent(Intent.ACTION_VIEW, uri);
    intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
    context.startActivity(intent);
  }

  /**
   * 是否用中文
   *
   * @return true是中文
   */
  public static boolean isChinese() {
    String language = Locale.getDefault().getLanguage();
    String country = Locale.getDefault().getCountry();
    return language.equals(Locale.SIMPLIFIED_CHINESE.getLanguage())
        && country.equals(Locale.SIMPLIFIED_CHINESE.getCountry());
  }

  /**
   * 系统分享功能
   *
   * @param context  上下文
   * @param msgTitle 消息标题
   * @param msgText  消息内容
   */
  public static void shareMsg(Context context, String msgTitle, String msgText) {
    Intent intent = new Intent(Intent.ACTION_SEND);
    intent.setType("text/plain"); // 纯文本
    //当用户选择短信时使用sms_body取得文字
    intent.putExtra("sms_body", msgText);
    intent.putExtra(Intent.EXTRA_SUBJECT, msgTitle);
    intent.putExtra(Intent.EXTRA_TEXT, msgText);
    intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
    context.startActivity(Intent.createChooser(intent, msgTitle));
  }

  public static void shareImg(Context context, String imgPath) {
    Intent intent = new Intent(Intent.ACTION_SEND);
    if (!TextUtils.isEmpty(imgPath)) {
      File f = new File(imgPath);
      if (f.exists() && f.isFile()) {
        intent.setType("image/jpg");
        Uri u = Uri.fromFile(f);
        intent.putExtra(Intent.EXTRA_STREAM, u);
      }
    }
    intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
    context.startActivity(intent);
  }

  public static int randomInt(int len) {
    Random random = new Random();
    int min = Double.valueOf(Math.pow(10, len - 1)).intValue();
    int max = Double.valueOf(Math.pow(10, len)).intValue();
    return min + random.nextInt(max);
  }

  public static byte[] convertInputStreamToByteArray(InputStream inputStream) {
    byte[] bytes = null;
    try {
      ByteArrayOutputStream bos = new ByteArrayOutputStream();

      byte data[] = new byte[1024];
      int count;

      while ((count = inputStream.read(data)) != -1) {
        bos.write(data, 0, count);
      }

      bos.flush();
      bos.close();
      inputStream.close();

      bytes = bos.toByteArray();
    } catch (IOException e) {
      e.printStackTrace();
    }
    return bytes;
  }

  public static float convertFloat(float f) {
    DecimalFormat df = new DecimalFormat("#.##");
    try {
      return df.parse(df.format(f)).floatValue();
    } catch (ParseException e) {
      e.printStackTrace();
    }
    return f;
  }

  public static int floatToInt(float f, int scale) {
    return (int) Math.round(Math.pow(10, scale) * f);
  }

  public static float convertDoubleToFloat(double d) {
    DecimalFormat df = new DecimalFormat("#.##");
    try {
      return df.parse(df.format(d)).floatValue();
    } catch (ParseException e) {
      e.printStackTrace();
    }
    return Double.valueOf(d).floatValue();
  }

  public static int convertFloatToInt(float f, int multiple) {
    Double d = Double.valueOf(String.valueOf(f)) * multiple;
    return (int) Math.round(d);
  }

  public static String convert10ToString(int ten) {
    if (ten < 10 && ten >= 0) {
      return "0" + ten;
    }
    return String.valueOf(ten);
  }

  public static String convertTemperature(float temperature) {
    DecimalFormat mFormat = new DecimalFormat("0.00");
    return mFormat.format(temperature);
  }

  public static String convertRate(float rate) {
    DecimalFormat mFormat = new DecimalFormat("0.0");
    return mFormat.format(rate);
  }

  public static Drawable getDrawable(Context context, int id) {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
      return context.getResources().getDrawable(id, context.getTheme());
    } else {
      return context.getResources().getDrawable(id);
    }
  }

  public static int getColor(Context context, int id) {
    return context.getResources().getColor(id);
  }

  /**
   * 获取手机服务商信息
   */
  public static String getProvidersName(Context context) {
    TelephonyManager telephonyManager = (TelephonyManager) context
        .getSystemService(Context.TELEPHONY_SERVICE);
    return telephonyManager.getSimOperatorName();
  }

  public static int getIntegerFromDouble(double num) {
    return (int) num;
  }

  public static int getDecimalFromDouble(double num) {
    return (int) ((num - getIntegerFromDouble(num) + 0.01) * 10);
  }

  public static double getDoubleFromDecimal(int integer, int decimal) {
    return (decimal + 0.0) / 10 + integer;
  }

  public static int convertToRealBalance(int number) {
    return (int) (Math.round(number / 100.0));
  }

  public static int convertToServerBalance(double number) {
    return (int) Math.round(number * 100);
  }

  public static String encriptAccountDisplay(String number, AccountType accountType) {
    String starString = "****";
    StringBuilder stringBuffer = new StringBuilder();
    if (accountType.getValue() == AccountType.ALIPAY_ACCOUNT.getValue()) {
      if (number.contains("@")) {
        accountType = AccountType.EMAIL;
      } else {
        accountType = AccountType.PHONE_NUMBER;
      }
    }
    switch (accountType) {
      case USER_NAME:
      case PHONE_NUMBER:
        if (number.length() < 7) {
          for (int i = 0; i < 3 && i < number.length(); i++) {
            stringBuffer.append(number.charAt(i));
          }
          stringBuffer.append(starString);
        } else {
          stringBuffer.append(number.substring(0, 3)).append(starString)
              .append(number.substring(7, number.length()));
        }
        break;

      case EMAIL:
        int divisionPosition = number.indexOf("@");
        for (int i = 0; i < 3 && i < divisionPosition; i++) {
          stringBuffer.append(number.charAt(i));
        }
        stringBuffer.append(starString);
        stringBuffer.append(number.substring(divisionPosition, number.length()));
        break;

      default:
        break;
    }

    return stringBuffer.toString();
  }

  public enum AccountType {
    EMAIL(1),
    PHONE_NUMBER(2),
    USER_NAME(3),
    ALIPAY_ACCOUNT(4);

    private final int value;

    AccountType(int value) {
      this.value = value;
    }

    public int getValue() {
      return this.value;
    }
  }

  public static int getLanguage() {
    return CommonUtil.isChinese() ? Language.CHINESE.getLanguage()
        : Language.ENGLISH.getLanguage();
  }

  public static int dpToPx(int dp) {
    return (int) (dp * Resources.getSystem().getDisplayMetrics().density);
  }

  public static int pxToDp(int px) {
    return (int) (px / Resources.getSystem().getDisplayMetrics().density);
  }

  public static <T> List<T> reverseList(List<T> list) {
    if (list == null || list.size() == 0) {
      return null;
    }

    int size = list.size();
    List<T> reverseList = new LinkedList<>();
    for (int i = size - 1; i >= 0; i--) {
      reverseList.add(list.get(i));
    }

    return reverseList;
  }

  public static float scaleValue(double value, int scale) {
    String scales = "#.";
    for (int i = 1; i <= scale; i++) {
      scales += "#";
    }

    DecimalFormat df = new DecimalFormat(scales);

    try {
      return df.parse(df.format(value)).floatValue();
    } catch (ParseException e) {
      e.printStackTrace();
    }

    return 0;
  }
}