package app.raybritton.elog.data

import android.content.ContentValues
import android.content.Context
import android.database.Cursor
import android.database.sqlite.SQLiteDatabase
import android.database.sqlite.SQLiteOpenHelper
import app.raybritton.elog.arch.LogModule
import java.io.File

internal class LogDatabase {
    private val openHelper by lazy {
        LogDbOpenHelper(LogModule.context, "elog.db", 2)
    }

    internal fun getLogs(): List<LogFile> {
        val cursor = openHelper.readableDatabase.rawQuery("SELECT * FROM logs", arrayOf())
        val results = mutableListOf<LogFile>()

        while (cursor.moveToNext()) {
            results.add(cursor.toLogFile())
        }

        cursor.close()
        return results
    }

    internal fun insertLog(logFile: LogFile) {
        openHelper.writableDatabase.insert(TABLE_NAME, null, logFile.toContentValues())
    }

    internal fun updateLog(logFile: LogFile) {
        openHelper.writableDatabase.update(TABLE_NAME, logFile.toContentValues(), "$COL_ID = ?", arrayOf(logFile.id))
    }

    internal fun deleteLog(id: String) {
        openHelper.writableDatabase.delete(TABLE_NAME, "$COL_ID = ?", arrayOf(id))
    }

    internal fun deleteAll() {
        openHelper.writableDatabase.execSQL("DELETE FROM $TABLE_NAME")
    }

    fun getLatestLog(): LogFile? {
        val cursor = openHelper.readableDatabase.rawQuery("SELECT * FROM logs ORDER BY $COL_TIMESTAMP DESC LIMIT 1", arrayOf())
        val results = mutableListOf<LogFile>()

        if (cursor.count == 0) {
            return null
        }

        while (cursor.moveToNext()) {
            results.add(cursor.toLogFile())
        }

        cursor.close()
        return results[0]
    }

    private fun LogFile.toContentValues(): ContentValues {
        val values = ContentValues()
        values.put(COL_ID, id)
        values.put(COL_FILE, file.absolutePath)
        values.put(COL_TIMESTAMP, created)
        values.put(COL_CODE, code)
        values.put(COL_CONT_CODE, continuationCode)
        return values
    }

    private fun Cursor.toLogFile(): LogFile {
        return LogFile(
            getString(getColumnIndex(COL_ID)),
            File(getString(getColumnIndex(COL_FILE))),
            getLong(getColumnIndex(COL_TIMESTAMP)),
            getString(getColumnIndex(COL_CODE)),
            getString(getColumnIndex(COL_CONT_CODE))
        )
    }

    private class LogDbOpenHelper(ctx: Context, name: String, version: Int) : SQLiteOpenHelper(ctx, name, null, version) {
        override fun onCreate(db: SQLiteDatabase) {
            db.execSQL("CREATE TABLE $TABLE_NAME ($COL_ID TEXT PRIMARY KEY, $COL_FILE TEXT, $COL_CODE TEXT, $COL_TIMESTAMP INTEGER, $COL_CONT_CODE TEXT)")
        }

        override fun onUpgrade(db: SQLiteDatabase, oldVer: Int, newVer: Int) {
            if (oldVer == 1 && newVer == 2) {
                db.execSQL("ALTER TABLE $TABLE_NAME ADD COLUMN $COL_CONT_CODE TEXT")
            }
        }
    }

    private companion object {
        private const val TABLE_NAME = "logs"
        private const val COL_ID = "id"
        private const val COL_FILE = "file"
        private const val COL_CODE = "code"
        private const val COL_CONT_CODE = "contCode"
        private const val COL_TIMESTAMP = "dateFormatter"
    }
}