package app.raybritton.elog.data

import app.raybritton.elog.BuildConfig
import app.raybritton.elog.ELog
import app.raybritton.elog.ELogConfig
import okhttp3.*
import okhttp3.MediaType.Companion.toMediaType
import okhttp3.RequestBody.Companion.asRequestBody
import java.io.File
import java.io.IOException
import java.text.SimpleDateFormat
import java.time.Instant
import java.util.*
import java.util.concurrent.TimeUnit

class LogUpload {
    private val dateFormatter = SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ", Locale.UK)

    private val okhttp by lazy {
        OkHttpClient.Builder()
            .readTimeout(ELogConfig.readTimeout, TimeUnit.SECONDS)
            .writeTimeout(ELogConfig.writeTimeout, TimeUnit.SECONDS)
            .connectTimeout(ELogConfig.connectTimeout, TimeUnit.SECONDS)
            .apply {
                ELogConfig.interceptor?.let {
                    addInterceptor(it)
                }
            }
            .build()
    }

    /**
     * if logId is not provided an id will be generated automatically using the ELog.idGen setting
     *
     * onComplete may be called from any thread
     */
    fun upload(log: LogFile, code: String = ELog.generateCode(), onComplete: (result: Result) -> Unit) {
        if (ELog.uploadServerUrl.isBlank()) {
            throw IllegalStateException("uploadServerUrl has not been set")
        }
        if (ELogConfig.apiPath.isBlank()) {
            throw IllegalStateException("apiPath has not been set")
        }
        val request = Request.Builder()
            .url(ELog.uploadServerUrl + ELogConfig.apiPath)
            .header("x-code", code)
            .header("x-version", BuildConfig.VERSION_NAME)
            .header("x-version-code", BuildConfig.VERSION_CODE.toString())
            .header("x-app", BuildConfig.APPLICATION_ID)
            .header("x-build", BuildConfig.BUILD_TYPE + BuildConfig.FLAVOR)
            .header("x-created", dateFormatter.format(Date(log.created)))
            .post(log.file.asRequestBody("text/plain".toMediaType()))
            .build()

        okhttp.newCall(request)
            .enqueue(object : Callback {
                override fun onFailure(call: Call, e: IOException) {
                    onComplete(Result.Error(e))
                }

                override fun onResponse(call: Call, response: Response) {
                    if (response.isSuccessful) {
                        onComplete(Result.Success(code))
                    } else {
                        onComplete(Result.Error(IOException("${response.code} ${response.message}")))
                    }
                }
            })
    }
}