/*
 * Decompiled with CFR 0.152.
 */
package world.data.jdbc.internal.transport;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.zip.GZIPInputStream;
import world.data.jdbc.internal.transport.ErrorMessageParser;
import world.data.jdbc.internal.transport.FileBackedInputStream;
import world.data.jdbc.internal.transport.QueryApi;
import world.data.jdbc.internal.transport.RdfParser;
import world.data.jdbc.internal.transport.Response;
import world.data.jdbc.internal.transport.SparqlResultsParser;
import world.data.jdbc.internal.transport.StreamParser;
import world.data.jdbc.internal.util.CloseableRef;
import world.data.jdbc.internal.util.Conditions;
import world.data.jdbc.internal.util.Optionals;
import world.data.jdbc.model.Node;

public final class HttpQueryApi
implements QueryApi {
    private static final AtomicLong THREAD_COUNTER = new AtomicLong(0L);
    private static final List<StreamParser<Response>> STANDARD_PARSERS = Arrays.asList(new RdfParser(), new SparqlResultsParser());
    private final URL queryEndpoint;
    private final String userAgent;
    private final String authToken;
    private final ExecutorService cachedThreadPool = Executors.newCachedThreadPool(target -> new Thread(target, String.format("dw-jdbc-%d", THREAD_COUNTER.getAndIncrement())));

    public HttpQueryApi(URL queryEndpoint, String userAgent, String authToken) {
        this.queryEndpoint = Objects.requireNonNull(queryEndpoint, "queryEndpoint");
        this.userAgent = Objects.requireNonNull(userAgent, "userAgent");
        this.authToken = authToken;
    }

    @Override
    public void close() {
        this.cachedThreadPool.shutdown();
    }

    @Override
    public Response executeQuery(String query, Map<String, Node> parameters, Integer maxRowsToReturn, Integer timeoutSeconds) throws SQLException {
        Objects.requireNonNull(query, "query");
        Objects.requireNonNull(parameters, "parameters");
        LinkedHashMap<String, String> requestParams = new LinkedHashMap<String, String>();
        requestParams.put("query", query);
        for (Map.Entry<String, Node> entry : parameters.entrySet()) {
            String name = entry.getKey();
            Node value = entry.getValue();
            Conditions.check(name.startsWith("$") && name.length() > 1, "Illegal parameter name: %s", name);
            if (value == null) continue;
            requestParams.put(name, value.toString());
        }
        if (maxRowsToReturn != null) {
            requestParams.put("maxRowsReturned", Integer.toString(maxRowsToReturn));
        }
        return (Response)this.post(requestParams, timeoutSeconds, STANDARD_PARSERS);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private <T> T post(Map<String, String> requestParams, Integer timeoutSeconds, List<StreamParser<T>> responseParsers) throws SQLException {
        try {
            StringBuilder buf = new StringBuilder();
            for (Map.Entry<String, String> entry : requestParams.entrySet()) {
                if (buf.length() > 0) {
                    buf.append('&');
                }
                buf.append(this.encode(entry.getKey())).append('=').append(this.encode(entry.getValue()));
            }
            byte[] requestBody = buf.toString().getBytes(StandardCharsets.UTF_8);
            String acceptTypes = responseParsers.stream().map(StreamParser::getAcceptType).collect(Collectors.joining(", "));
            int readTimeout = Math.min(Optionals.or(timeoutSeconds, 60), 60);
            int connectTimeout = Math.min(readTimeout, 5);
            HttpURLConnection connection = (HttpURLConnection)this.queryEndpoint.openConnection();
            connection.setConnectTimeout((int)TimeUnit.SECONDS.toMillis(connectTimeout));
            connection.setReadTimeout((int)TimeUnit.SECONDS.toMillis(readTimeout));
            connection.addRequestProperty("Content-Type", "application/x-www-form-urlencoded; charset=utf-8");
            connection.addRequestProperty("Accept", acceptTypes);
            connection.addRequestProperty("Accept-Encoding", "gzip");
            connection.addRequestProperty("User-Agent", this.userAgent);
            if (this.authToken != null) {
                connection.addRequestProperty("Authorization", "Bearer " + this.authToken);
            }
            connection.setRequestMethod("POST");
            connection.setDoOutput(true);
            connection.setFixedLengthStreamingMode(requestBody.length);
            connection.getOutputStream().write(requestBody);
            int status = connection.getResponseCode();
            String message = connection.getResponseMessage();
            String contentType = this.trimHeader(connection.getHeaderField("Content-Type"));
            if (status >= 400) {
                String details;
                InputStream err = connection.getErrorStream();
                try (CloseableRef ignored = new CloseableRef(err);){
                    details = err != null ? new ErrorMessageParser().parse(err, contentType) : null;
                }
                if (details != null && !details.isEmpty()) {
                    throw new SQLException(String.format("HTTP request to '%s' failed with response %d: %s; %s", this.queryEndpoint, status, message, details));
                }
                throw new SQLException(String.format("HTTP request to '%s' failed with response %d: %s", this.queryEndpoint, status, message));
            }
            if (status != 200) {
                throw new SQLException(String.format("HTTP request to '%s' failed with unexpected response %d: %s", this.queryEndpoint, status, message));
            }
            InputStream in = connection.getInputStream();
            try (CloseableRef cleanup = new CloseableRef(in);){
                in = cleanup.set(new FileBackedInputStream(in, 16384, this.cachedThreadPool));
                if ("gzip".equals(this.trimHeader(connection.getHeaderField("Content-Encoding")))) {
                    in = cleanup.set(new GZIPInputStream(new BufferedInputStream(in)));
                }
                T t = cleanup.detach(this.parseResponse(in, contentType, responseParsers));
                return t;
            }
            catch (SQLException e) {
                throw e;
            }
            catch (IOException e) {
                throw new SQLException("I/O exception while parsing HTTP response from server: " + this.queryEndpoint, e);
            }
            catch (Exception e) {
                throw new SQLException("Unexpected exception parsing HTTP response from server: " + this.queryEndpoint, e);
            }
        }
        catch (SQLException e) {
            throw e;
        }
        catch (IOException e) {
            throw new SQLException("I/O exception while making HTTP request to server: " + this.queryEndpoint, e);
        }
        catch (Exception e) {
            throw new SQLException("Unexpected exception while making HTTP request to server: " + this.queryEndpoint, e);
        }
    }

    private <T> T parseResponse(InputStream in, String contentType, List<StreamParser<T>> responseParsers) throws Exception {
        for (StreamParser<T> responseParser : responseParsers) {
            for (String acceptType : responseParser.getAcceptType().split(",")) {
                if (!this.trimHeader(acceptType).equals(contentType)) continue;
                return responseParser.parse(in, contentType);
            }
        }
        throw new SQLException(String.format("HTTP request to '%s' failed with unexpected content type: %s", this.queryEndpoint, contentType));
    }

    private String trimHeader(String header) {
        return header != null ? header.replaceFirst(";.*", "").trim() : null;
    }

    private String encode(String string) {
        try {
            return URLEncoder.encode(string, "UTF-8");
        }
        catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
    }
}

