/*
 * Decompiled with CFR 0.152.
 */
package us.ihmc.robotics.math.trajectories.waypoints;

import gnu.trove.list.array.TDoubleArrayList;
import java.util.ArrayList;
import org.apache.commons.lang3.tuple.Pair;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import us.ihmc.robotics.Assert;
import us.ihmc.robotics.math.trajectories.generators.TrajectoryPointOptimizer;
import us.ihmc.yoVariables.registry.YoRegistry;
import us.ihmc.yoVariables.variable.YoDouble;

public class TrajectoryPointOptimizerTest {
    private static final double epsilon = 1.0E-6;

    @Test
    public void testEndPointSetters() {
        Assertions.assertThrows(RuntimeException.class, () -> {
            int dimensions = 3;
            YoRegistry registry = new YoRegistry("Dummy");
            TrajectoryPointOptimizer optimizer = new TrajectoryPointOptimizer(dimensions, registry);
            TDoubleArrayList rightSize = new TDoubleArrayList(dimensions);
            TDoubleArrayList wrongSize = new TDoubleArrayList(dimensions - 1);
            for (int i = 0; i < dimensions; ++i) {
                rightSize.add(0.0);
            }
            optimizer.setEndPoints(rightSize, rightSize, rightSize, rightSize);
            optimizer.setEndPoints(rightSize, wrongSize, rightSize, rightSize);
        });
    }

    @Test
    public void testWaypointSetters() {
        Assertions.assertThrows(RuntimeException.class, () -> {
            int dimensions = 1;
            TrajectoryPointOptimizer optimizer = new TrajectoryPointOptimizer(dimensions);
            ArrayList<TDoubleArrayList> waypoints = new ArrayList<TDoubleArrayList>();
            TDoubleArrayList waypointA = new TDoubleArrayList();
            TDoubleArrayList waypointB = new TDoubleArrayList();
            TDoubleArrayList waypointC = new TDoubleArrayList();
            waypointA.add(1.0);
            waypointB.add(5.0);
            waypointC.add(21.0);
            waypoints.add(waypointA);
            waypoints.add(waypointB);
            waypoints.add(waypointC);
            optimizer.setWaypoints(waypoints);
            TDoubleArrayList waypointD = new TDoubleArrayList();
            waypointD.add(5.0);
            waypointD.add(1.7);
            waypoints.add(waypointD);
            optimizer.setWaypoints(waypoints);
        });
    }

    @Test
    public void testYoVariables() {
        int dimensions = 3;
        YoRegistry registry = new YoRegistry("Dummy");
        new TrajectoryPointOptimizer(dimensions, registry);
        YoDouble timeGain = (YoDouble)registry.findVariable("TimeGain");
        Assert.assertTrue(timeGain.getDoubleValue() != 0.0);
    }

    @Test
    public void testSimpleSymmetricProblem() {
        int dimensions = 1;
        TrajectoryPointOptimizer optimizer = new TrajectoryPointOptimizer(dimensions);
        TDoubleArrayList x0 = new TDoubleArrayList(dimensions);
        TDoubleArrayList xd0 = new TDoubleArrayList(dimensions);
        TDoubleArrayList x1 = new TDoubleArrayList(dimensions);
        TDoubleArrayList xd1 = new TDoubleArrayList(dimensions);
        for (int i = 0; i < dimensions; ++i) {
            x0.add(0.0);
            xd0.add(0.0);
            x1.add(0.0);
            xd1.add(0.0);
        }
        x1.set(0, 1.0);
        optimizer.setEndPoints(x0, xd0, x1, xd1);
        optimizer.compute();
        ArrayList<TDoubleArrayList> dummyWaypoints = new ArrayList<TDoubleArrayList>();
        TDoubleArrayList waypointA = new TDoubleArrayList();
        TDoubleArrayList waypointB = new TDoubleArrayList();
        waypointA.add(0.25);
        waypointB.add(0.3);
        dummyWaypoints.add(waypointA);
        dummyWaypoints.add(waypointB);
        optimizer.setWaypoints(dummyWaypoints);
        optimizer.compute();
        ArrayList<TDoubleArrayList> waypoints = new ArrayList<TDoubleArrayList>();
        TDoubleArrayList waypointC = new TDoubleArrayList();
        waypointC.add(0.5);
        waypoints.add(waypointC);
        optimizer.setWaypoints(waypoints);
        optimizer.compute();
        ArrayList<TDoubleArrayList> coefficients = new ArrayList<TDoubleArrayList>();
        for (int i = 0; i < waypoints.size() + 1; ++i) {
            coefficients.add(new TDoubleArrayList(4));
        }
        optimizer.getPolynomialCoefficients(coefficients, 0);
        double[] expected = new double[]{-2.0, 3.0, 0.0, 0.0};
        TDoubleArrayList waypointTimes = new TDoubleArrayList(waypoints.size());
        optimizer.getWaypointTimes(waypointTimes);
        this.printResults(waypointTimes, coefficients);
        for (int i = 0; i < 4; ++i) {
            Assert.assertEquals(coefficients.get(0).get(i), expected[i], 1.0E-6);
            Assert.assertEquals(coefficients.get(0).get(i), coefficients.get(1).get(i), 1.0E-6);
        }
        TDoubleArrayList waypointVelocity = new TDoubleArrayList();
        optimizer.getWaypointVelocity(waypointVelocity, 0);
        Assert.assertEquals(waypointVelocity.get(0), 1.5, 1.0E-6);
    }

    @Test
    public void testSimpleProblem() {
        int dimensions = 1;
        TrajectoryPointOptimizer optimizer = new TrajectoryPointOptimizer(dimensions);
        TDoubleArrayList x0 = new TDoubleArrayList(dimensions);
        TDoubleArrayList xd0 = new TDoubleArrayList(dimensions);
        TDoubleArrayList x1 = new TDoubleArrayList(dimensions);
        TDoubleArrayList xd1 = new TDoubleArrayList(dimensions);
        x0.add(1.0);
        xd0.add(1.0);
        x1.add(2.0);
        xd1.add(1.0);
        optimizer.setEndPoints(x0, xd0, x1, xd1);
        optimizer.compute();
        ArrayList<TDoubleArrayList> coefficients = new ArrayList<TDoubleArrayList>();
        coefficients.add(new TDoubleArrayList());
        optimizer.getPolynomialCoefficients(coefficients, 0);
        double[] expected = new double[]{0.0, 0.0, 1.0, 1.0};
        for (int i = 0; i < 4; ++i) {
            Assert.assertEquals(((TDoubleArrayList)coefficients.get(0)).get(i), expected[i], 1.0E-6);
        }
    }

    @Test
    public void testTrivialProblem() {
        int dimensions = 1;
        TrajectoryPointOptimizer optimizer = new TrajectoryPointOptimizer(dimensions);
        TDoubleArrayList x0 = new TDoubleArrayList(dimensions);
        TDoubleArrayList xd0 = new TDoubleArrayList(dimensions);
        TDoubleArrayList x1 = new TDoubleArrayList(dimensions);
        TDoubleArrayList xd1 = new TDoubleArrayList(dimensions);
        x0.add(0.0);
        xd0.add(1.0);
        x1.add(1.0);
        xd1.add(1.0);
        optimizer.setEndPoints(x0, xd0, x1, xd1);
        optimizer.compute(0);
        ArrayList<TDoubleArrayList> coefficients = new ArrayList<TDoubleArrayList>();
        coefficients.add(new TDoubleArrayList(0));
        optimizer.getPolynomialCoefficients(coefficients, 0);
        double[] expected = new double[]{0.0, 0.0, 1.0, 0.0};
        for (int i = 0; i < 4; ++i) {
            Assert.assertEquals(expected[i], ((TDoubleArrayList)coefficients.get(0)).get(i), 1.0E-6);
        }
    }

    @Test
    public void testTimeDescent() {
        int dimensions = 2;
        TrajectoryPointOptimizer optimizer = new TrajectoryPointOptimizer(dimensions);
        TDoubleArrayList x0 = new TDoubleArrayList(dimensions);
        TDoubleArrayList xd0 = new TDoubleArrayList(dimensions);
        TDoubleArrayList x1 = new TDoubleArrayList(dimensions);
        TDoubleArrayList xd1 = new TDoubleArrayList(dimensions);
        for (int i = 0; i < dimensions; ++i) {
            x0.add(0.0);
            xd0.add(0.0);
            x1.add(0.0);
            xd1.add(0.0);
        }
        x1.set(0, 1.0);
        x1.set(1, 1.0);
        optimizer.setEndPoints(x0, xd0, x1, xd1);
        ArrayList<TDoubleArrayList> waypoints = new ArrayList<TDoubleArrayList>();
        TDoubleArrayList waypointA = new TDoubleArrayList();
        TDoubleArrayList waypointB = new TDoubleArrayList();
        waypointA.add(0.001);
        waypointA.add(0.001);
        waypointB.add(0.8);
        waypointB.add(0.8);
        waypoints.add(waypointA);
        waypoints.add(waypointB);
        optimizer.setWaypoints(waypoints);
        optimizer.compute();
        TDoubleArrayList waypointTimes = new TDoubleArrayList();
        optimizer.getWaypointTimes(waypointTimes);
        Assert.assertTrue(waypointTimes.get(0) < 0.1);
        Assert.assertTrue(waypointTimes.get(0) > 0.0);
    }

    @Test
    public void testGetEndpoints() {
        int dimensions = 3;
        TrajectoryPointOptimizer optimizer = new TrajectoryPointOptimizer(dimensions);
        TDoubleArrayList input_x0 = new TDoubleArrayList(new double[]{0.654, 0.129, -0.004});
        TDoubleArrayList input_xd0 = new TDoubleArrayList(new double[]{0.045, 0.015, 0.207});
        TDoubleArrayList input_x1 = new TDoubleArrayList(new double[]{1.416, 0.125, -0.022});
        TDoubleArrayList input_xd1 = new TDoubleArrayList(new double[]{0.0, 0.0, -0.3});
        ArrayList<TDoubleArrayList> input_waypoints = new ArrayList<TDoubleArrayList>();
        input_waypoints.add(new TDoubleArrayList(new double[]{0.768, 0.129, 0.093}));
        input_waypoints.add(new TDoubleArrayList(new double[]{1.301, 0.126, 0.081}));
        optimizer.setEndPoints(input_x0, input_xd0, input_x1, input_xd1);
        optimizer.setWaypoints(input_waypoints);
        optimizer.compute();
        TDoubleArrayList output_x0 = new TDoubleArrayList();
        TDoubleArrayList output_xd0 = new TDoubleArrayList();
        TDoubleArrayList output_x1 = new TDoubleArrayList();
        TDoubleArrayList output_xd1 = new TDoubleArrayList();
        optimizer.getStartPosition(output_x0);
        optimizer.getStartVelocity(output_xd0);
        optimizer.getTargetPosition(output_x1);
        optimizer.getTargetVelocity(output_xd1);
        Assert.assertArrayEquals(input_x0.toArray(), output_x0.toArray(), 1.0E-6);
        Assert.assertArrayEquals(input_xd0.toArray(), output_xd0.toArray(), 1.0E-6);
        Assert.assertArrayEquals(input_x1.toArray(), output_x1.toArray(), 1.0E-6);
        Assert.assertArrayEquals(input_xd1.toArray(), output_xd1.toArray(), 1.0E-6);
    }

    private void printResults(TDoubleArrayList waypointTimes, ArrayList<TDoubleArrayList> coefficients) {
        Object timesString = "";
        for (int i = 0; i < waypointTimes.size(); ++i) {
            timesString = (String)timesString + waypointTimes.get(i) + " ";
        }
        System.out.println("Waypoint Times: " + (String)timesString);
        for (TDoubleArrayList coeffs : coefficients) {
            Object coeffString = "";
            for (int i = 0; i < coeffs.size(); ++i) {
                coeffString = (String)coeffString + coeffs.get(i) + " ";
            }
            System.out.println("Polynomial Coefficients: " + (String)coeffString);
        }
    }

    private static void runTimingTest() {
        int i;
        int maxWaypointsToTest = 12;
        boolean fixedTimes = true;
        ArrayList<Pair<TrajectoryPointOptimizer, TDoubleArrayList>> optimizers = new ArrayList<Pair<TrajectoryPointOptimizer, TDoubleArrayList>>();
        long[] timingData = new long[maxWaypointsToTest];
        for (int i2 = 0; i2 < maxWaypointsToTest; ++i2) {
            optimizers.add(TrajectoryPointOptimizerTest.setupOptimizerForSemicircularTrajectory(i2 + 1));
        }
        Pair<TrajectoryPointOptimizer, TDoubleArrayList> tempOptimizer = TrajectoryPointOptimizerTest.setupOptimizerForSemicircularTrajectory(1);
        for (int i3 = 0; i3 < 250; ++i3) {
            ((TrajectoryPointOptimizer)tempOptimizer.getLeft()).compute();
        }
        int numberOfOptimizationsToRun = 20;
        for (i = 0; i < optimizers.size(); ++i) {
            long diff;
            long start = System.nanoTime();
            Pair optimizer = (Pair)optimizers.get(i);
            if (fixedTimes) {
                ((TrajectoryPointOptimizer)optimizer.getLeft()).computeForFixedTime((TDoubleArrayList)optimizer.getRight());
            } else {
                ((TrajectoryPointOptimizer)optimizer.getLeft()).compute(numberOfOptimizationsToRun);
            }
            long stop = System.nanoTime();
            timingData[i] = diff = stop - start;
        }
        for (i = 0; i < optimizers.size(); ++i) {
            System.out.println(i + 1 + "\t " + (double)timingData[i] / ((double)numberOfOptimizationsToRun * 1000000.0) + " ms");
        }
    }

    private static Pair<TrajectoryPointOptimizer, TDoubleArrayList> setupOptimizerForSemicircularTrajectory(int numberOfWaypoints) {
        int dimensions = 3;
        TrajectoryPointOptimizer optimizer = new TrajectoryPointOptimizer(dimensions);
        TDoubleArrayList x0 = new TDoubleArrayList(new double[]{0.0, 0.0, 0.0});
        TDoubleArrayList xd0 = new TDoubleArrayList(new double[]{0.0, 0.0, 1.0});
        TDoubleArrayList x1 = new TDoubleArrayList(new double[]{1.0, 1.0, 0.0});
        TDoubleArrayList xd1 = new TDoubleArrayList(new double[]{0.0, 0.0, -1.0});
        ArrayList<TDoubleArrayList> waypoints = new ArrayList<TDoubleArrayList>();
        double theta = Math.PI / (double)(numberOfWaypoints + 1);
        for (int i = 0; i < numberOfWaypoints; ++i) {
            double xyValue = Math.cos((double)(i + 1) * theta);
            double zValue = Math.sin((double)(i + 1) * theta);
            TDoubleArrayList waypoint = new TDoubleArrayList(new double[]{xyValue, xyValue, zValue});
            waypoints.add(waypoint);
        }
        optimizer.setEndPoints(x0, xd0, x1, xd1);
        optimizer.setWaypoints(waypoints);
        TDoubleArrayList times = new TDoubleArrayList();
        for (int i = 0; i < numberOfWaypoints; ++i) {
            times.add((double)(i + 1) / (double)(numberOfWaypoints + 1));
        }
        return Pair.of((Object)optimizer, (Object)times);
    }

    public static void main(String[] args) {
        TrajectoryPointOptimizerTest.runTimingTest();
    }
}

