/*
 * Decompiled with CFR 0.152.
 */
package us.ihmc.euclid.geometry.tools;

import java.util.Collections;
import java.util.List;
import java.util.Random;
import us.ihmc.euclid.geometry.Bound;
import us.ihmc.euclid.geometry.interfaces.Vertex2DSupplier;
import us.ihmc.euclid.geometry.tools.EuclidGeometryTools;
import us.ihmc.euclid.interfaces.EuclidGeometry;
import us.ihmc.euclid.tools.EuclidCoreTools;
import us.ihmc.euclid.tuple2D.Point2D;
import us.ihmc.euclid.tuple2D.interfaces.Point2DBasics;
import us.ihmc.euclid.tuple2D.interfaces.Point2DReadOnly;
import us.ihmc.euclid.tuple2D.interfaces.Tuple2DReadOnly;
import us.ihmc.euclid.tuple2D.interfaces.Vector2DBasics;
import us.ihmc.euclid.tuple2D.interfaces.Vector2DReadOnly;

public class EuclidGeometryPolygonTools {
    private static final Random random = new Random();
    static final double EPSILON = 1.0E-7;

    private EuclidGeometryPolygonTools() {
    }

    public static boolean isPolygon2DConvexAtVertex(int vertexIndex, List<? extends Point2DReadOnly> vertices, boolean clockwiseOrdered) {
        return EuclidGeometryPolygonTools.isPolygon2DConvexAtVertex(vertexIndex, vertices, vertices.size(), clockwiseOrdered);
    }

    public static boolean isPolygon2DConvexAtVertex(int vertexIndex, List<? extends Point2DReadOnly> vertices, int numberOfVertices, boolean clockwiseOrdered) {
        EuclidGeometryPolygonTools.checkNumberOfVertices(vertices, numberOfVertices);
        if (numberOfVertices <= 2) {
            return true;
        }
        Point2DReadOnly vertex = vertices.get(vertexIndex);
        Point2DReadOnly previousVertex = vertices.get(EuclidGeometryPolygonTools.previous(vertexIndex, numberOfVertices));
        Point2DReadOnly nextVertex = vertices.get(EuclidGeometryPolygonTools.next(vertexIndex, numberOfVertices));
        return EuclidGeometryTools.isPoint2DOnSideOfLine2D(vertex, previousVertex, nextVertex, clockwiseOrdered);
    }

    public static int inPlaceGiftWrapConvexHull2D(List<? extends Point2DReadOnly> vertices) {
        return EuclidGeometryPolygonTools.inPlaceGiftWrapConvexHull2D(vertices, vertices.size());
    }

    public static int inPlaceGiftWrapConvexHull2D(List<? extends Point2DReadOnly> vertices, int numberOfVertices) {
        if (numberOfVertices == 0) {
            return 0;
        }
        EuclidGeometryPolygonTools.checkNumberOfVertices(vertices, numberOfVertices);
        Collections.swap(vertices, EuclidGeometryPolygonTools.findMinXMaxYVertexIndex(vertices, numberOfVertices), 0);
        Point2DReadOnly firstVertex = vertices.get(0);
        int lastHullVertexIndex = 0;
        while (lastHullVertexIndex < numberOfVertices - 1) {
            Point2DReadOnly lastHullVertex = vertices.get(lastHullVertexIndex);
            int candidateIndex = lastHullVertexIndex + 1;
            Point2DReadOnly candidateVertex = vertices.get(candidateIndex);
            while (candidateVertex.epsilonEquals((EuclidGeometry)lastHullVertex, 1.0E-7)) {
                Collections.swap(vertices, candidateIndex, --numberOfVertices);
                candidateVertex = vertices.get(candidateIndex);
                if (numberOfVertices != 1) continue;
                return numberOfVertices;
            }
            int vertexIndex = lastHullVertexIndex + 2;
            while (vertexIndex <= numberOfVertices) {
                int wrappedIndex = EuclidGeometryPolygonTools.wrap(vertexIndex, numberOfVertices);
                Point2DReadOnly vertex = vertices.get(wrappedIndex);
                if (vertex.epsilonEquals((EuclidGeometry)candidateVertex, 1.0E-7) || wrappedIndex != 0 && vertex.epsilonEquals((EuclidGeometry)firstVertex, 1.0E-7)) {
                    Collections.swap(vertices, wrappedIndex, --numberOfVertices);
                    continue;
                }
                if (EuclidGeometryTools.isPoint2DOnLeftSideOfLine2D(vertex, lastHullVertex, candidateVertex)) {
                    candidateIndex = wrappedIndex;
                    candidateVertex = vertex;
                }
                ++vertexIndex;
            }
            if (candidateIndex == 0) {
                return lastHullVertexIndex + 1;
            }
            Collections.swap(vertices, ++lastHullVertexIndex, candidateIndex);
        }
        return numberOfVertices;
    }

    public static int inPlaceGrahamScanConvexHull2D(List<? extends Point2DReadOnly> vertices) {
        return EuclidGeometryPolygonTools.inPlaceGrahamScanConvexHull2D(vertices, vertices.size());
    }

    public static int inPlaceGrahamScanConvexHull2D(List<? extends Point2DReadOnly> vertices, int numberOfVertices) {
        if (numberOfVertices == 0) {
            return 0;
        }
        EuclidGeometryPolygonTools.checkNumberOfVertices(vertices, numberOfVertices);
        EuclidGeometryPolygonTools.grahamScanAngleSort(vertices, numberOfVertices);
        if (numberOfVertices <= 3) {
            boolean areAllVerticesEqual = true;
            Point2DReadOnly firstVertex = vertices.get(0);
            for (int vertexIndex = 1; vertexIndex < numberOfVertices; ++vertexIndex) {
                if (firstVertex.epsilonEquals((EuclidGeometry)vertices.get(vertexIndex), 1.0E-7)) continue;
                areAllVerticesEqual = false;
                break;
            }
            return areAllVerticesEqual ? 1 : numberOfVertices;
        }
        int currentIndex = 1;
        while (currentIndex < numberOfVertices) {
            if (vertices.get(currentIndex).epsilonEquals((EuclidGeometry)vertices.get(EuclidGeometryPolygonTools.previous(currentIndex, numberOfVertices)), 1.0E-7)) {
                EuclidGeometryPolygonTools.moveElementToEnd(vertices, currentIndex, numberOfVertices);
                --numberOfVertices;
                continue;
            }
            if (EuclidGeometryPolygonTools.isPolygon2DConvexAtVertex(currentIndex, vertices, numberOfVertices, true)) {
                ++currentIndex;
                continue;
            }
            EuclidGeometryPolygonTools.moveElementToEnd(vertices, currentIndex, numberOfVertices);
            if (--numberOfVertices <= 2) {
                return numberOfVertices;
            }
            currentIndex = Math.max(0, currentIndex - 1);
        }
        return numberOfVertices;
    }

    public static double computeConvexPolygon2DArea(List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered, Point2DBasics centroidToPack) {
        EuclidGeometryPolygonTools.checkNumberOfVertices(convexPolygon2D, numberOfVertices);
        if (numberOfVertices == 0) {
            if (centroidToPack != null) {
                centroidToPack.setToNaN();
            }
            return Double.NaN;
        }
        if (numberOfVertices < 3) {
            if (centroidToPack != null) {
                centroidToPack.setToZero();
                for (int i = 0; i < numberOfVertices; ++i) {
                    centroidToPack.add((Tuple2DReadOnly)convexPolygon2D.get(i));
                }
                centroidToPack.scale(1.0 / (double)numberOfVertices);
            }
            return 0.0;
        }
        double area = 0.0;
        double Cx = 0.0;
        double Cy = 0.0;
        if (clockwiseOrdered) {
            for (int i = 0; i < numberOfVertices; ++i) {
                Point2DReadOnly ci = convexPolygon2D.get(i);
                Point2DReadOnly ciMinus1 = convexPolygon2D.get(EuclidGeometryPolygonTools.previous(i, numberOfVertices));
                double weight = ci.getX() * ciMinus1.getY() - ciMinus1.getX() * ci.getY();
                Cx += (ci.getX() + ciMinus1.getX()) * weight;
                Cy += (ci.getY() + ciMinus1.getY()) * weight;
                area += weight;
            }
        } else {
            for (int i = 0; i < numberOfVertices; ++i) {
                Point2DReadOnly ci = convexPolygon2D.get(i);
                Point2DReadOnly ciPlus1 = convexPolygon2D.get(EuclidGeometryPolygonTools.next(i, numberOfVertices));
                double weight = ci.getX() * ciPlus1.getY() - ciPlus1.getX() * ci.getY();
                Cx += (ci.getX() + ciPlus1.getX()) * weight;
                Cy += (ci.getY() + ciPlus1.getY()) * weight;
                area += weight;
            }
        }
        area *= 0.5;
        if (centroidToPack != null) {
            if (area < 1.0E-5) {
                centroidToPack.set((Tuple2DReadOnly)convexPolygon2D.get(0));
            } else {
                centroidToPack.set(Cx, Cy);
                centroidToPack.scale(1.0 / (6.0 * area));
            }
        }
        return area;
    }

    public static boolean edgeNormal(int edgeIndex, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered, Vector2DBasics normalToPack) {
        EuclidGeometryPolygonTools.checkNumberOfVertices(convexPolygon2D, numberOfVertices);
        if (numberOfVertices <= 1) {
            return false;
        }
        EuclidGeometryPolygonTools.checkEdgeIndex(edgeIndex, numberOfVertices);
        Point2DReadOnly edgeStart = convexPolygon2D.get(edgeIndex);
        Point2DReadOnly edgeEnd = convexPolygon2D.get(EuclidGeometryPolygonTools.next(edgeIndex, numberOfVertices));
        normalToPack.sub((Tuple2DReadOnly)edgeEnd, (Tuple2DReadOnly)edgeStart);
        normalToPack.normalize();
        EuclidGeometryTools.perpendicularVector2D((Vector2DReadOnly)normalToPack, normalToPack);
        if (!clockwiseOrdered) {
            normalToPack.negate();
        }
        return true;
    }

    public static boolean isPoint2DInsideConvexPolygon2D(double pointX, double pointY, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered) {
        EuclidGeometryPolygonTools.checkNumberOfVertices(convexPolygon2D, numberOfVertices);
        if (numberOfVertices == 0) {
            return false;
        }
        if (numberOfVertices == 1) {
            Point2DReadOnly vertex = convexPolygon2D.get(0);
            return pointX == vertex.getX() && pointY == vertex.getY();
        }
        Point2DReadOnly edgeStart = convexPolygon2D.get(0);
        Point2DReadOnly edgeEnd = convexPolygon2D.get(1);
        if (numberOfVertices == 2) {
            if (pointX == edgeStart.getX() && pointY == edgeStart.getY()) {
                return true;
            }
            if (pointX == edgeEnd.getX() && pointY == edgeEnd.getY()) {
                return true;
            }
            if (edgeStart.getX() < edgeEnd.getX() ? pointX < edgeStart.getX() || pointX > edgeEnd.getX() : pointX < edgeEnd.getX() || pointX > edgeStart.getX()) {
                return false;
            }
            if (edgeStart.getY() < edgeEnd.getY() ? pointY < edgeStart.getY() || pointY > edgeEnd.getY() : pointY < edgeEnd.getY() || pointY > edgeStart.getY()) {
                return false;
            }
            double dx = pointX - edgeStart.getX();
            double dy = pointY - edgeStart.getY();
            double edgeDirectionX = edgeEnd.getX() - edgeStart.getX();
            double edgeDirectionY = edgeEnd.getY() - edgeStart.getY();
            double crossProduct = edgeDirectionY * dy - dx * edgeDirectionX;
            return crossProduct == 0.0;
        }
        if (EuclidGeometryTools.isPoint2DOnSideOfLine2D(pointX, pointY, edgeStart, edgeEnd, clockwiseOrdered)) {
            return false;
        }
        for (int index = 1; index < numberOfVertices; ++index) {
            edgeStart = edgeEnd;
            if (!EuclidGeometryTools.isPoint2DOnSideOfLine2D(pointX, pointY, edgeStart, edgeEnd = convexPolygon2D.get(EuclidGeometryPolygonTools.next(index, numberOfVertices)), clockwiseOrdered)) continue;
            return false;
        }
        return true;
    }

    public static boolean isPoint2DInsideConvexPolygon2D(Point2DReadOnly point, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered) {
        return EuclidGeometryPolygonTools.isPoint2DInsideConvexPolygon2D(point.getX(), point.getY(), convexPolygon2D, numberOfVertices, clockwiseOrdered);
    }

    public static boolean isPoint2DInsideConvexPolygon2D(double pointX, double pointY, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered, double epsilon) {
        return EuclidGeometryPolygonTools.signedDistanceFromPoint2DToConvexPolygon2D(pointX, pointY, convexPolygon2D, numberOfVertices, clockwiseOrdered) <= epsilon;
    }

    public static boolean isPoint2DInsideConvexPolygon2D(Point2DReadOnly point, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered, double epsilon) {
        return EuclidGeometryPolygonTools.isPoint2DInsideConvexPolygon2D(point.getX(), point.getY(), convexPolygon2D, numberOfVertices, clockwiseOrdered, epsilon);
    }

    public static int intersectionBetweenLine2DAndConvexPolygon2D(Point2DReadOnly pointOnLine, Vector2DReadOnly lineDirection, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered, Point2DBasics firstIntersectionToPack, Point2DBasics secondIntersectionToPack) {
        double pointOnLineX = pointOnLine.getX();
        double pointOnLineY = pointOnLine.getY();
        double lineDirectionX = lineDirection.getX();
        double lineDirectionY = lineDirection.getY();
        return EuclidGeometryPolygonTools.intersectionBetweenLine2DAndConvexPolygon2D(pointOnLineX, pointOnLineY, lineDirectionX, lineDirectionY, convexPolygon2D, numberOfVertices, clockwiseOrdered, firstIntersectionToPack, secondIntersectionToPack);
    }

    public static int intersectionBetweenLine2DAndConvexPolygon2D(double pointOnLineX, double pointOnLineY, double lineDirectionX, double lineDirectionY, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered, Point2DBasics firstIntersectionToPack, Point2DBasics secondIntersectionToPack) {
        EuclidGeometryPolygonTools.checkNumberOfVertices(convexPolygon2D, numberOfVertices);
        if (numberOfVertices == 0) {
            return 0;
        }
        if (numberOfVertices == 1) {
            Point2DReadOnly vertex = convexPolygon2D.get(0);
            if (EuclidGeometryTools.isPoint2DOnLine2D(vertex.getX(), vertex.getY(), pointOnLineX, pointOnLineY, lineDirectionX, lineDirectionY)) {
                firstIntersectionToPack.set((Tuple2DReadOnly)convexPolygon2D.get(0));
                return 1;
            }
            return 0;
        }
        int firstEdgeIndex = EuclidGeometryPolygonTools.nextEdgeIndexIntersectingWithLine2D(-1, pointOnLineX, pointOnLineY, lineDirectionX, lineDirectionY, convexPolygon2D, numberOfVertices);
        if (firstEdgeIndex < 0) {
            return 0;
        }
        Point2DReadOnly edgeStart = convexPolygon2D.get(firstEdgeIndex);
        Point2DReadOnly edgeEnd = convexPolygon2D.get(EuclidGeometryPolygonTools.next(firstEdgeIndex, numberOfVertices));
        boolean success = EuclidGeometryTools.intersectionBetweenLine2DAndLineSegment2D(pointOnLineX, pointOnLineY, lineDirectionX, lineDirectionY, edgeStart.getX(), edgeStart.getY(), edgeEnd.getX(), edgeEnd.getY(), firstIntersectionToPack);
        if (!success) {
            throw new RuntimeException("Inconsistency in algorithms.");
        }
        int secondEdgeIndex = EuclidGeometryPolygonTools.nextEdgeIndexIntersectingWithLine2D(firstEdgeIndex, pointOnLineX, pointOnLineY, lineDirectionX, lineDirectionY, convexPolygon2D, numberOfVertices);
        if (secondEdgeIndex < 0) {
            return 1;
        }
        edgeStart = convexPolygon2D.get(secondEdgeIndex);
        edgeEnd = convexPolygon2D.get(EuclidGeometryPolygonTools.next(secondEdgeIndex, numberOfVertices));
        success = EuclidGeometryTools.intersectionBetweenLine2DAndLineSegment2D(pointOnLineX, pointOnLineY, lineDirectionX, lineDirectionY, edgeStart.getX(), edgeStart.getY(), edgeEnd.getX(), edgeEnd.getY(), secondIntersectionToPack);
        if (!success) {
            throw new RuntimeException("Inconsistency in algorithms.");
        }
        if (!firstIntersectionToPack.epsilonEquals((EuclidGeometry)secondIntersectionToPack, 1.0E-7)) {
            return 2;
        }
        if ((secondEdgeIndex = EuclidGeometryPolygonTools.nextEdgeIndexIntersectingWithLine2D(secondEdgeIndex, pointOnLineX, pointOnLineY, lineDirectionX, lineDirectionY, convexPolygon2D, numberOfVertices)) < 0 || secondEdgeIndex == firstEdgeIndex) {
            return 1;
        }
        edgeStart = convexPolygon2D.get(secondEdgeIndex);
        edgeEnd = convexPolygon2D.get(EuclidGeometryPolygonTools.next(secondEdgeIndex, numberOfVertices));
        success = EuclidGeometryTools.intersectionBetweenLine2DAndLineSegment2D(pointOnLineX, pointOnLineY, lineDirectionX, lineDirectionY, edgeStart.getX(), edgeStart.getY(), edgeEnd.getX(), edgeEnd.getY(), secondIntersectionToPack);
        if (!success) {
            throw new RuntimeException("Inconsistency in algorithms.");
        }
        return 2;
    }

    public static Point2D[] intersectionBetweenLine2DAndConvexPolygon2D(Point2DReadOnly pointOnLine, Vector2DReadOnly lineDirection, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered) {
        Point2D firstIntersection = new Point2D();
        Point2D secondIntersection = new Point2D();
        int numberOfIntersections = EuclidGeometryPolygonTools.intersectionBetweenLine2DAndConvexPolygon2D(pointOnLine, lineDirection, convexPolygon2D, numberOfVertices, clockwiseOrdered, (Point2DBasics)firstIntersection, (Point2DBasics)secondIntersection);
        switch (numberOfIntersections) {
            case 1: {
                return new Point2D[]{firstIntersection};
            }
            case 2: {
                return new Point2D[]{firstIntersection, secondIntersection};
            }
        }
        return null;
    }

    public static int intersectionBetweenLineSegment2DAndConvexPolygon2D(Point2DReadOnly lineSegmentStart, Point2DReadOnly lineSegmentEnd, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered, Point2DBasics firstIntersectionToPack, Point2DBasics secondIntersectionToPack) {
        EuclidGeometryPolygonTools.checkNumberOfVertices(convexPolygon2D, numberOfVertices);
        if (numberOfVertices == 0) {
            return 0;
        }
        if (numberOfVertices == 1) {
            Point2DReadOnly vertex = convexPolygon2D.get(0);
            if (EuclidGeometryTools.distanceFromPoint2DToLineSegment2D(vertex, lineSegmentStart, lineSegmentEnd) < 1.0E-12) {
                firstIntersectionToPack.set((Tuple2DReadOnly)convexPolygon2D.get(0));
                return 1;
            }
            return 0;
        }
        double lineSegmentDx = lineSegmentEnd.getX() - lineSegmentStart.getX();
        double lineSegmentDy = lineSegmentEnd.getY() - lineSegmentStart.getY();
        int foundIntersections = 0;
        if (numberOfVertices == 2) {
            Point2DReadOnly vertex0 = convexPolygon2D.get(0);
            Point2DReadOnly vertex1 = convexPolygon2D.get(1);
            if (EuclidGeometryTools.distanceFromPoint2DToLineSegment2D(vertex0, lineSegmentStart, lineSegmentEnd) < 1.0E-12) {
                firstIntersectionToPack.set((Tuple2DReadOnly)vertex0);
                ++foundIntersections;
            }
            if (EuclidGeometryTools.distanceFromPoint2DToLineSegment2D(vertex1, lineSegmentStart, lineSegmentEnd) < 1.0E-12) {
                if (foundIntersections == 0) {
                    firstIntersectionToPack.set((Tuple2DReadOnly)vertex1);
                } else {
                    secondIntersectionToPack.set((Tuple2DReadOnly)vertex1);
                }
                ++foundIntersections;
            }
            if (foundIntersections == 2) {
                return 2;
            }
        }
        for (int edgeIndex = 0; edgeIndex < numberOfVertices; ++edgeIndex) {
            Point2DReadOnly edgeEnd;
            Point2DReadOnly edgeStart = convexPolygon2D.get(edgeIndex);
            if (EuclidGeometryTools.distanceFromPoint2DToLineSegment2D(lineSegmentStart, edgeStart, edgeEnd = convexPolygon2D.get(EuclidGeometryPolygonTools.next(edgeIndex, numberOfVertices))) < 1.0E-12) {
                if (foundIntersections == 0) {
                    firstIntersectionToPack.set((Tuple2DReadOnly)lineSegmentStart);
                } else {
                    secondIntersectionToPack.set((Tuple2DReadOnly)lineSegmentStart);
                }
                if (++foundIntersections == 2 && firstIntersectionToPack.epsilonEquals((EuclidGeometry)secondIntersectionToPack, 1.0E-12)) {
                    --foundIntersections;
                }
                if (foundIntersections == 2) {
                    return foundIntersections;
                }
            }
            if (EuclidGeometryTools.distanceFromPoint2DToLineSegment2D(lineSegmentEnd, edgeStart, edgeEnd) < 1.0E-12) {
                if (foundIntersections == 0) {
                    firstIntersectionToPack.set((Tuple2DReadOnly)lineSegmentEnd);
                } else {
                    secondIntersectionToPack.set((Tuple2DReadOnly)lineSegmentEnd);
                }
                if (++foundIntersections == 2 && firstIntersectionToPack.epsilonEquals((EuclidGeometry)secondIntersectionToPack, 1.0E-12)) {
                    --foundIntersections;
                }
                if (foundIntersections == 2) {
                    return foundIntersections;
                }
            }
            double edgeVectorX = edgeEnd.getX() - edgeStart.getX();
            double edgeVectorY = edgeEnd.getY() - edgeStart.getY();
            double lambda = EuclidGeometryTools.percentageOfIntersectionBetweenTwoLine2Ds(edgeStart.getX(), edgeStart.getY(), edgeVectorX, edgeVectorY, lineSegmentStart.getX(), lineSegmentStart.getY(), lineSegmentDx, lineSegmentDy);
            if (Double.isInfinite(lambda)) {
                lambda = 0.0;
            }
            if (Double.isNaN(lambda) || lambda < -1.0E-12 || lambda > 1.000000000001) continue;
            if (lambda < 0.0) {
                lambda = 0.0;
            } else if (lambda > 1.0) {
                lambda = 1.0;
            }
            double candidateX = edgeStart.getX() + lambda * edgeVectorX;
            double candidateY = edgeStart.getY() + lambda * edgeVectorY;
            if (!EuclidGeometryTools.isPoint2DInFrontOfRay2D(candidateX, candidateY, lineSegmentStart.getX(), lineSegmentStart.getY(), lineSegmentDx, lineSegmentDy) || !EuclidGeometryTools.isPoint2DInFrontOfRay2D(candidateX, candidateY, lineSegmentEnd.getX(), lineSegmentEnd.getY(), -lineSegmentDx, -lineSegmentDy)) continue;
            if (foundIntersections == 0) {
                firstIntersectionToPack.set(candidateX, candidateY);
            } else {
                secondIntersectionToPack.set(candidateX, candidateY);
            }
            if (++foundIntersections == 2 && firstIntersectionToPack.epsilonEquals((EuclidGeometry)secondIntersectionToPack, 1.0E-12)) {
                --foundIntersections;
            }
            if (foundIntersections != 2) continue;
            return foundIntersections;
        }
        return foundIntersections;
    }

    public static Point2D[] intersectionBetweenLineSegment2DAndConvexPolygon2D(Point2DReadOnly lineSegmentStart, Point2DReadOnly lineSegmentEnd, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered) {
        Point2D firstIntersection = new Point2D();
        Point2D secondIntersection = new Point2D();
        int numberOfIntersections = EuclidGeometryPolygonTools.intersectionBetweenLineSegment2DAndConvexPolygon2D(lineSegmentStart, lineSegmentEnd, convexPolygon2D, numberOfVertices, clockwiseOrdered, (Point2DBasics)firstIntersection, (Point2DBasics)secondIntersection);
        switch (numberOfIntersections) {
            case 1: {
                return new Point2D[]{firstIntersection};
            }
            case 2: {
                return new Point2D[]{firstIntersection, secondIntersection};
            }
        }
        return null;
    }

    public static int intersectionBetweenRay2DAndConvexPolygon2D(Point2DReadOnly rayOrigin, Vector2DReadOnly rayDirection, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered, Point2DBasics firstIntersectionToPack, Point2DBasics secondIntersectionToPack) {
        int numberOfIntersections = EuclidGeometryPolygonTools.intersectionBetweenLine2DAndConvexPolygon2D(rayOrigin, rayDirection, convexPolygon2D, numberOfVertices, clockwiseOrdered, firstIntersectionToPack, secondIntersectionToPack);
        if (numberOfIntersections == 2 && !EuclidGeometryTools.isPoint2DInFrontOfRay2D((Point2DReadOnly)secondIntersectionToPack, rayOrigin, rayDirection)) {
            --numberOfIntersections;
        }
        if (numberOfIntersections >= 1 && !EuclidGeometryTools.isPoint2DInFrontOfRay2D((Point2DReadOnly)firstIntersectionToPack, rayOrigin, rayDirection)) {
            --numberOfIntersections;
            firstIntersectionToPack.set((Tuple2DReadOnly)secondIntersectionToPack);
        }
        return numberOfIntersections;
    }

    public static Point2D[] intersectionBetweenRay2DAndConvexPolygon2D(Point2DReadOnly rayOrigin, Vector2DReadOnly rayDirection, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered) {
        Point2D firstIntersection = new Point2D();
        Point2D secondIntersection = new Point2D();
        int numberOfIntersections = EuclidGeometryPolygonTools.intersectionBetweenRay2DAndConvexPolygon2D(rayOrigin, rayDirection, convexPolygon2D, numberOfVertices, clockwiseOrdered, (Point2DBasics)firstIntersection, (Point2DBasics)secondIntersection);
        switch (numberOfIntersections) {
            case 1: {
                return new Point2D[]{firstIntersection};
            }
            case 2: {
                return new Point2D[]{firstIntersection, secondIntersection};
            }
        }
        return null;
    }

    public static double signedDistanceFromPoint2DToConvexPolygon2D(double pointX, double pointY, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered) {
        EuclidGeometryPolygonTools.checkNumberOfVertices(convexPolygon2D, numberOfVertices);
        if (numberOfVertices == 0) {
            return Double.NaN;
        }
        if (numberOfVertices == 1) {
            return EuclidGeometryTools.distanceBetweenPoint2Ds(pointX, pointY, convexPolygon2D.get(0));
        }
        if (numberOfVertices == 2) {
            return EuclidGeometryTools.distanceFromPoint2DToLineSegment2D(pointX, pointY, convexPolygon2D.get(0), convexPolygon2D.get(1));
        }
        boolean isQueryOutsidePolygon = false;
        double minDistance = Double.POSITIVE_INFINITY;
        for (int index = 0; index < numberOfVertices; ++index) {
            Point2DReadOnly edgeStart = convexPolygon2D.get(index);
            Point2DReadOnly edgeEnd = convexPolygon2D.get(EuclidGeometryPolygonTools.next(index, numberOfVertices));
            isQueryOutsidePolygon |= EuclidGeometryTools.isPoint2DOnSideOfLine2D(pointX, pointY, edgeStart, edgeEnd, clockwiseOrdered);
            minDistance = Math.min(minDistance, EuclidGeometryTools.distanceSquaredFromPoint2DToLineSegment2D(pointX, pointY, edgeStart, edgeEnd));
        }
        minDistance = EuclidCoreTools.squareRoot((double)minDistance);
        if (!isQueryOutsidePolygon) {
            minDistance = -minDistance;
        }
        return minDistance;
    }

    public static double signedDistanceFromPoint2DToConvexPolygon2D(Point2DReadOnly point, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered) {
        return EuclidGeometryPolygonTools.signedDistanceFromPoint2DToConvexPolygon2D(point.getX(), point.getY(), convexPolygon2D, numberOfVertices, clockwiseOrdered);
    }

    public static boolean closestPointToNonInterectingRay2D(Point2DReadOnly rayOrigin, Vector2DReadOnly rayDirection, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered, Point2DBasics closestPointToPack) {
        double distanceToProjection;
        double previousEdgeDirectionY;
        double nextEdgeDirectionY;
        int closestVertexIndexToLine = EuclidGeometryPolygonTools.closestVertexIndexToLine2D(rayOrigin, rayDirection, convexPolygon2D, numberOfVertices);
        if (closestVertexIndexToLine == -1) {
            return false;
        }
        Point2DReadOnly closestVertexToLine = convexPolygon2D.get(closestVertexIndexToLine);
        boolean success = EuclidGeometryPolygonTools.orthogonalProjectionOnConvexPolygon2D(rayOrigin, convexPolygon2D, numberOfVertices, clockwiseOrdered, closestPointToPack);
        if (!success) {
            return false;
        }
        Point2DReadOnly nextEdgeStart = closestVertexToLine;
        Point2DReadOnly nextEdgeEnd = convexPolygon2D.get(EuclidGeometryPolygonTools.next(closestVertexIndexToLine, numberOfVertices));
        double nextEdgeDirectionX = nextEdgeEnd.getX() - nextEdgeStart.getX();
        boolean areRayAndNextEdgeParallel = EuclidGeometryTools.areVector2DsParallel(nextEdgeDirectionX, nextEdgeDirectionY = nextEdgeEnd.getY() - nextEdgeStart.getY(), rayDirection.getX(), rayDirection.getY(), 1.0E-7);
        if (areRayAndNextEdgeParallel) {
            return true;
        }
        Point2DReadOnly previousEdgeStart = convexPolygon2D.get(EuclidGeometryPolygonTools.previous(closestVertexIndexToLine, numberOfVertices));
        Point2DReadOnly previousEdgeEnd = closestVertexToLine;
        double previousEdgeDirectionX = previousEdgeEnd.getX() - previousEdgeStart.getX();
        boolean areRayAndPreviousEdgeParallel = EuclidGeometryTools.areVector2DsParallel(previousEdgeDirectionX, previousEdgeDirectionY = previousEdgeEnd.getY() - previousEdgeStart.getY(), rayDirection.getX(), rayDirection.getY(), 1.0E-7);
        if (areRayAndPreviousEdgeParallel) {
            return true;
        }
        double distanceToClosestVertex = EuclidGeometryTools.distanceFromPoint2DToRay2D(closestVertexToLine, rayOrigin, rayDirection);
        if (distanceToClosestVertex < (distanceToProjection = EuclidGeometryTools.distanceFromPoint2DToRay2D((Point2DReadOnly)closestPointToPack, rayOrigin, rayDirection))) {
            closestPointToPack.set((Tuple2DReadOnly)closestVertexToLine);
        }
        return true;
    }

    public static Point2D closestPointToNonInterectingRay2D(Point2DReadOnly rayOrigin, Vector2DReadOnly rayDirection, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered) {
        Point2D closestPoint = new Point2D();
        boolean success = EuclidGeometryPolygonTools.closestPointToNonInterectingRay2D(rayOrigin, rayDirection, convexPolygon2D, numberOfVertices, clockwiseOrdered, (Point2DBasics)closestPoint);
        return success ? closestPoint : null;
    }

    public static int closestVertexIndexToLine2D(double pointOnLineX, double pointOnLineY, double lineDirectionX, double lineDirectionY, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices) {
        EuclidGeometryPolygonTools.checkNumberOfVertices(convexPolygon2D, numberOfVertices);
        int index = -1;
        double minDistance = Double.POSITIVE_INFINITY;
        for (int i = 0; i < numberOfVertices; ++i) {
            Point2DReadOnly vertex = convexPolygon2D.get(i);
            double distance = EuclidGeometryTools.distanceFromPoint2DToLine2D(vertex.getX(), vertex.getY(), pointOnLineX, pointOnLineY, lineDirectionX, lineDirectionY);
            if (!(distance < minDistance)) continue;
            index = i;
            minDistance = distance;
        }
        return index;
    }

    public static int closestVertexIndexToLine2D(Point2DReadOnly firstPointOnLine, Point2DReadOnly secondPointOnLine, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices) {
        double pointOnLineX = firstPointOnLine.getX();
        double pointOnLineY = firstPointOnLine.getY();
        double lineDirectionX = secondPointOnLine.getX() - firstPointOnLine.getX();
        double lineDirectionY = secondPointOnLine.getY() - firstPointOnLine.getY();
        return EuclidGeometryPolygonTools.closestVertexIndexToLine2D(pointOnLineX, pointOnLineY, lineDirectionX, lineDirectionY, convexPolygon2D, numberOfVertices);
    }

    public static int closestVertexIndexToLine2D(Point2DReadOnly pointOnLine, Vector2DReadOnly lineDirection, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices) {
        return EuclidGeometryPolygonTools.closestVertexIndexToLine2D(pointOnLine.getX(), pointOnLine.getY(), lineDirection.getX(), lineDirection.getY(), convexPolygon2D, numberOfVertices);
    }

    public static int closestVertexIndexToRay2D(Point2DReadOnly rayOrigin, Vector2DReadOnly rayDirection, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered) {
        EuclidGeometryPolygonTools.checkNumberOfVertices(convexPolygon2D, numberOfVertices);
        int index = -1;
        double minDistance = Double.POSITIVE_INFINITY;
        for (int i = 0; i < numberOfVertices; ++i) {
            Point2DReadOnly vertex = convexPolygon2D.get(i);
            double distance = EuclidGeometryTools.distanceFromPoint2DToRay2D(vertex, rayOrigin, rayDirection);
            if (!(distance < minDistance)) continue;
            index = i;
            minDistance = distance;
        }
        return index;
    }

    public static int closestVertexIndexToPoint2D(Point2DReadOnly point, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices) {
        return EuclidGeometryPolygonTools.closestVertexIndexToPoint2D(point.getX(), point.getY(), convexPolygon2D, numberOfVertices);
    }

    public static int closestVertexIndexToPoint2D(double pointX, double pointY, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices) {
        EuclidGeometryPolygonTools.checkNumberOfVertices(convexPolygon2D, numberOfVertices);
        int index = -1;
        double minDistanceSquared = Double.POSITIVE_INFINITY;
        for (int i = 0; i < numberOfVertices; ++i) {
            double distanceSquared = EuclidGeometryTools.distanceSquaredBetweenPoint2Ds(pointX, pointY, convexPolygon2D.get(i));
            if (!(distanceSquared < minDistanceSquared)) continue;
            index = i;
            minDistanceSquared = distanceSquared;
        }
        return index;
    }

    public static int closestEdgeIndexToPoint2D(double pointX, double pointY, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered) {
        EuclidGeometryPolygonTools.checkNumberOfVertices(convexPolygon2D, numberOfVertices);
        if (numberOfVertices <= 1) {
            return -1;
        }
        boolean isQueryOutsidePolygon = false;
        int insideIndex = -1;
        int outsideIndex = -1;
        double minOutsideDistanceSquared = Double.POSITIVE_INFINITY;
        double minInsideDistanceSquared = Double.POSITIVE_INFINITY;
        for (int edgeIndex = 0; edgeIndex < numberOfVertices; ++edgeIndex) {
            Point2DReadOnly edgeStart = convexPolygon2D.get(edgeIndex);
            Point2DReadOnly edgeEnd = convexPolygon2D.get(EuclidGeometryPolygonTools.next(edgeIndex, numberOfVertices));
            double distanceSquared = EuclidGeometryTools.distanceSquaredFromPoint2DToLineSegment2D(pointX, pointY, edgeStart, edgeEnd);
            boolean isOutsideEdge = EuclidGeometryTools.isPoint2DOnSideOfLine2D(pointX, pointY, edgeStart, edgeEnd, clockwiseOrdered);
            isQueryOutsidePolygon |= isOutsideEdge;
            if (isOutsideEdge) {
                if (!(distanceSquared < minOutsideDistanceSquared)) continue;
                outsideIndex = edgeIndex;
                minOutsideDistanceSquared = distanceSquared;
                continue;
            }
            if (!(distanceSquared < minInsideDistanceSquared)) continue;
            insideIndex = edgeIndex;
            minInsideDistanceSquared = distanceSquared;
        }
        return isQueryOutsidePolygon ? outsideIndex : insideIndex;
    }

    public static int closestEdgeIndexToPoint2D(Point2DReadOnly point, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered) {
        return EuclidGeometryPolygonTools.closestEdgeIndexToPoint2D(point.getX(), point.getY(), convexPolygon2D, numberOfVertices, clockwiseOrdered);
    }

    public static int lineOfSightStartIndex(double observerX, double observerY, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered) {
        EuclidGeometryPolygonTools.checkNumberOfVertices(convexPolygon2D, numberOfVertices);
        if (numberOfVertices == 0) {
            return -1;
        }
        if (numberOfVertices == 1) {
            Point2DReadOnly vertex = convexPolygon2D.get(0);
            if (vertex.getX() == observerX && vertex.getY() == observerY) {
                return -1;
            }
            return 0;
        }
        if (numberOfVertices == 2) {
            if (EuclidGeometryPolygonTools.isPoint2DInsideConvexPolygon2D(observerX, observerY, convexPolygon2D, numberOfVertices, clockwiseOrdered, 0.0)) {
                return -1;
            }
            boolean isOnSide = EuclidGeometryTools.isPoint2DOnSideOfLine2D(observerX, observerY, convexPolygon2D.get(0), convexPolygon2D.get(1), clockwiseOrdered);
            return isOnSide ? 0 : 1;
        }
        boolean previousEdgeVisible = EuclidGeometryPolygonTools.canObserverSeeEdge(numberOfVertices - 1, observerX, observerY, convexPolygon2D, numberOfVertices, clockwiseOrdered);
        for (int edgeIndex = 0; edgeIndex < numberOfVertices; ++edgeIndex) {
            boolean edgeVisible = EuclidGeometryPolygonTools.canObserverSeeEdge(edgeIndex, observerX, observerY, convexPolygon2D, numberOfVertices, clockwiseOrdered);
            if (!previousEdgeVisible && edgeVisible) {
                return edgeIndex;
            }
            previousEdgeVisible = edgeVisible;
        }
        return -1;
    }

    public static int lineOfSightStartIndex(Point2DReadOnly observer, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered) {
        return EuclidGeometryPolygonTools.lineOfSightStartIndex(observer.getX(), observer.getY(), convexPolygon2D, numberOfVertices, clockwiseOrdered);
    }

    public static int lineOfSightEndIndex(double observerX, double observerY, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered) {
        EuclidGeometryPolygonTools.checkNumberOfVertices(convexPolygon2D, numberOfVertices);
        if (numberOfVertices == 0) {
            return -1;
        }
        if (numberOfVertices == 1) {
            Point2DReadOnly vertex = convexPolygon2D.get(0);
            if (vertex.getX() == observerX && vertex.getY() == observerY) {
                return -1;
            }
            return 0;
        }
        if (numberOfVertices == 2) {
            if (EuclidGeometryPolygonTools.isPoint2DInsideConvexPolygon2D(observerX, observerY, convexPolygon2D, numberOfVertices, clockwiseOrdered, 0.0)) {
                return -1;
            }
            boolean isOnSide = EuclidGeometryTools.isPoint2DOnSideOfLine2D(observerX, observerY, convexPolygon2D.get(0), convexPolygon2D.get(1), clockwiseOrdered);
            return isOnSide ? 1 : 0;
        }
        boolean previousEdgeVisible = EuclidGeometryPolygonTools.canObserverSeeEdge(numberOfVertices - 1, observerX, observerY, convexPolygon2D, numberOfVertices, clockwiseOrdered);
        for (int edgeIndex = 0; edgeIndex < numberOfVertices; ++edgeIndex) {
            boolean edgeVisible = EuclidGeometryPolygonTools.canObserverSeeEdge(edgeIndex, observerX, observerY, convexPolygon2D, numberOfVertices, clockwiseOrdered);
            if (previousEdgeVisible && !edgeVisible) {
                return edgeIndex;
            }
            previousEdgeVisible = edgeVisible;
        }
        return -1;
    }

    public static int lineOfSightEndIndex(Point2DReadOnly observer, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered) {
        return EuclidGeometryPolygonTools.lineOfSightEndIndex(observer.getX(), observer.getY(), convexPolygon2D, numberOfVertices, clockwiseOrdered);
    }

    public static int[] lineOfSightIndices(Point2DReadOnly observer, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered) {
        int lineOfSightStartIndex = EuclidGeometryPolygonTools.lineOfSightStartIndex(observer, convexPolygon2D, numberOfVertices, clockwiseOrdered);
        int lineOfSightEndIndex = EuclidGeometryPolygonTools.lineOfSightEndIndex(observer, convexPolygon2D, numberOfVertices, clockwiseOrdered);
        if (lineOfSightStartIndex == -1 || lineOfSightEndIndex == -1) {
            return null;
        }
        return new int[]{lineOfSightStartIndex, lineOfSightEndIndex};
    }

    public static int nextEdgeIndexIntersectingWithLine2D(int previousEdgeIndex, Point2DReadOnly pointOnLine, Vector2DReadOnly lineDirection, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices) {
        return EuclidGeometryPolygonTools.nextEdgeIndexIntersectingWithLine2D(previousEdgeIndex, pointOnLine.getX(), pointOnLine.getY(), lineDirection.getX(), lineDirection.getY(), convexPolygon2D, numberOfVertices);
    }

    public static int nextEdgeIndexIntersectingWithLine2D(int previousEdgeIndex, double pointOnLineX, double pointOnLineY, double lineDirectionX, double lineDirectionY, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices) {
        EuclidGeometryPolygonTools.checkNumberOfVertices(convexPolygon2D, numberOfVertices);
        if (previousEdgeIndex < -2 || previousEdgeIndex >= numberOfVertices) {
            throw new IndexOutOfBoundsException("Expected previousEdgeIndex to be in [-2; numberOfVertices[, but was: " + previousEdgeIndex);
        }
        if (numberOfVertices <= 1 || previousEdgeIndex == -2) {
            return -2;
        }
        if (numberOfVertices == 2) {
            Point2DReadOnly edgeEnd;
            if (previousEdgeIndex == 0) {
                return 1;
            }
            Point2DReadOnly edgeStart = convexPolygon2D.get(0);
            if (EuclidGeometryTools.doLine2DAndLineSegment2DIntersect(pointOnLineX, pointOnLineY, lineDirectionX, lineDirectionY, edgeStart, edgeEnd = convexPolygon2D.get(1))) {
                return 0;
            }
        }
        int edgeIndex = previousEdgeIndex;
        for (int i = 0; i < numberOfVertices; ++i) {
            Point2DReadOnly edgeEnd;
            Point2DReadOnly edgeStart = convexPolygon2D.get(edgeIndex = EuclidGeometryPolygonTools.next(edgeIndex, numberOfVertices));
            boolean doLineAndEdgeIntersect = EuclidGeometryTools.doLine2DAndLineSegment2DIntersect(pointOnLineX, pointOnLineY, lineDirectionX, lineDirectionY, edgeStart, edgeEnd = convexPolygon2D.get(EuclidGeometryPolygonTools.next(edgeIndex, numberOfVertices)));
            if (!doLineAndEdgeIntersect) continue;
            return edgeIndex;
        }
        return -2;
    }

    public static boolean orthogonalProjectionOnConvexPolygon2D(double pointToProjectX, double pointToProjectY, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered, Point2DBasics projectionToPack) {
        EuclidGeometryPolygonTools.checkNumberOfVertices(convexPolygon2D, numberOfVertices);
        if (numberOfVertices == 0) {
            return false;
        }
        if (numberOfVertices == 1) {
            projectionToPack.set((Tuple2DReadOnly)convexPolygon2D.get(0));
            return true;
        }
        if (numberOfVertices == 2) {
            return EuclidGeometryTools.orthogonalProjectionOnLineSegment2D(pointToProjectX, pointToProjectY, convexPolygon2D.get(0), convexPolygon2D.get(1), projectionToPack);
        }
        int closestEdgeIndex = EuclidGeometryPolygonTools.closestEdgeIndexToPoint2D(pointToProjectX, pointToProjectY, convexPolygon2D, numberOfVertices, clockwiseOrdered);
        if (closestEdgeIndex == -1) {
            return false;
        }
        if (!EuclidGeometryPolygonTools.canObserverSeeEdge(closestEdgeIndex, pointToProjectX, pointToProjectY, convexPolygon2D, numberOfVertices, clockwiseOrdered)) {
            return false;
        }
        Point2DReadOnly edgeStart = convexPolygon2D.get(closestEdgeIndex);
        Point2DReadOnly edgeEnd = convexPolygon2D.get(EuclidGeometryPolygonTools.next(closestEdgeIndex, numberOfVertices));
        return EuclidGeometryTools.orthogonalProjectionOnLineSegment2D(pointToProjectX, pointToProjectY, edgeStart, edgeEnd, projectionToPack);
    }

    public static boolean orthogonalProjectionOnConvexPolygon2D(Point2DReadOnly pointToProject, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered, Point2DBasics projectionToPack) {
        return EuclidGeometryPolygonTools.orthogonalProjectionOnConvexPolygon2D(pointToProject.getX(), pointToProject.getY(), convexPolygon2D, numberOfVertices, clockwiseOrdered, projectionToPack);
    }

    public static Point2D orthogonalProjectionOnConvexPolygon2D(Point2DReadOnly pointToProject, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered) {
        Point2D projection = new Point2D();
        boolean success = EuclidGeometryPolygonTools.orthogonalProjectionOnConvexPolygon2D(pointToProject.getX(), pointToProject.getY(), convexPolygon2D, numberOfVertices, clockwiseOrdered, (Point2DBasics)projection);
        return success ? projection : null;
    }

    public static boolean canObserverSeeEdge(int edgeIndex, Point2DReadOnly observer, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered) {
        return EuclidGeometryPolygonTools.canObserverSeeEdge(edgeIndex, observer.getX(), observer.getY(), convexPolygon2D, numberOfVertices, clockwiseOrdered);
    }

    public static boolean canObserverSeeEdge(int edgeIndex, double observerX, double observerY, List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, boolean clockwiseOrdered) {
        EuclidGeometryPolygonTools.checkNumberOfVertices(convexPolygon2D, numberOfVertices);
        EuclidGeometryPolygonTools.checkEdgeIndex(edgeIndex, numberOfVertices);
        Point2DReadOnly edgeStart = convexPolygon2D.get(edgeIndex);
        Point2DReadOnly edgeEnd = convexPolygon2D.get(EuclidGeometryPolygonTools.next(edgeIndex, numberOfVertices));
        return EuclidGeometryTools.isPoint2DOnSideOfLine2D(observerX, observerY, edgeStart, edgeEnd, clockwiseOrdered);
    }

    public static boolean isConvexPolygon2DConcyclic(List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices, double epsilon) {
        EuclidGeometryPolygonTools.checkNumberOfVertices(convexPolygon2D, numberOfVertices);
        if (numberOfVertices == 0) {
            return false;
        }
        if (numberOfVertices <= 3) {
            return true;
        }
        int interval = Math.max(1, numberOfVertices / 3);
        Point2DReadOnly A = convexPolygon2D.get(0);
        int indexB = interval;
        Point2DReadOnly B = convexPolygon2D.get(indexB);
        int indexC = 2 * interval;
        Point2DReadOnly C = convexPolygon2D.get(indexC);
        double ASquared = A.distanceFromOriginSquared();
        double BSquared = B.distanceFromOriginSquared();
        double CSquared = C.distanceFromOriginSquared();
        double ByCy = B.getY() - C.getY();
        double CxBx = C.getX() - B.getX();
        double a = 0.5 / (A.getX() * ByCy + A.getY() * CxBx + B.getX() * C.getY() - B.getY() * C.getX());
        if (!Double.isFinite(a)) {
            return false;
        }
        double CSquaredBSqured = CSquared - BSquared;
        double sx = a * (ASquared * ByCy + A.getY() * CSquaredBSqured + BSquared * C.getY() - B.getY() * CSquared);
        double sy = a * (-A.getX() * CSquaredBSqured + ASquared * CxBx + B.getX() * CSquared - BSquared * C.getX());
        double distanceSquaredReference = EuclidCoreTools.normSquared((double)(sx - C.getX()), (double)(sy - C.getY()));
        for (int i = 1; i < numberOfVertices; ++i) {
            Point2DReadOnly vertex;
            double distanceSquared;
            if (i == indexB || i == indexC || EuclidCoreTools.epsilonEquals((double)distanceSquaredReference, (double)(distanceSquared = EuclidCoreTools.normSquared((double)(sx - (vertex = convexPolygon2D.get(i)).getX()), (double)(sy - vertex.getY()))), (double)epsilon)) continue;
            return false;
        }
        return true;
    }

    static void grahamScanAngleSort(List<? extends Point2DReadOnly> vertices, int numberOfVertices) {
        Point2DReadOnly minXMaxYVertex = vertices.get(EuclidGeometryPolygonTools.findMinXMaxYVertexIndex(vertices, numberOfVertices));
        EuclidGeometryPolygonTools.grahamScanAngleSortRecursive(vertices, minXMaxYVertex, 0, numberOfVertices - 1);
    }

    private static void grahamScanAngleSortRecursive(List<? extends Point2DReadOnly> vertices, Point2DReadOnly minXMaxYVertex, int startIndex, int endIndex) {
        if (endIndex <= startIndex) {
            return;
        }
        int pivotIndex = random.nextInt(endIndex - startIndex) + startIndex;
        pivotIndex = EuclidGeometryPolygonTools.grahamScanAnglePartition(vertices, minXMaxYVertex, startIndex, endIndex, pivotIndex);
        EuclidGeometryPolygonTools.grahamScanAngleSortRecursive(vertices, minXMaxYVertex, startIndex, pivotIndex - 1);
        EuclidGeometryPolygonTools.grahamScanAngleSortRecursive(vertices, minXMaxYVertex, pivotIndex + 1, endIndex);
    }

    private static int grahamScanAnglePartition(List<? extends Point2DReadOnly> list, Point2DReadOnly minXMaxYVertex, int startIndex, int endIndex, int pivotIndex) {
        Point2DReadOnly pivot = list.get(pivotIndex);
        Collections.swap(list, pivotIndex, endIndex);
        int partitionIndex = startIndex;
        for (int i = startIndex; i < endIndex; ++i) {
            if (EuclidGeometryPolygonTools.grahamScanAngleCompare(minXMaxYVertex, list.get(i), pivot) > 0) continue;
            Collections.swap(list, i, partitionIndex);
            ++partitionIndex;
        }
        Collections.swap(list, endIndex, partitionIndex);
        return partitionIndex;
    }

    static int grahamScanAngleCompare(Point2DReadOnly minXMaxYVertex, Point2DReadOnly vertex1, Point2DReadOnly vertex2) {
        if (vertex1 == minXMaxYVertex) {
            return -1;
        }
        if (vertex2 == minXMaxYVertex) {
            return 1;
        }
        if (vertex1.epsilonEquals((EuclidGeometry)vertex2, 1.0E-7)) {
            return 0;
        }
        if (EuclidGeometryTools.isPoint2DOnLeftSideOfLine2D(vertex1, minXMaxYVertex, vertex2)) {
            return -1;
        }
        return 1;
    }

    static <T> void moveElementToEnd(List<T> list, int indexOfElementToMove, int listSize) {
        T elementToMove = list.get(indexOfElementToMove);
        while (indexOfElementToMove < listSize - 1) {
            list.set(indexOfElementToMove++, list.get(indexOfElementToMove));
        }
        list.set(indexOfElementToMove, elementToMove);
    }

    static int findMinXMaxYVertexIndex(List<? extends Point2DReadOnly> vertices, int numberOfVertices) {
        if (numberOfVertices == 0) {
            return -1;
        }
        int minXMaxYIndex = 0;
        Point2DReadOnly minXMaxY = vertices.get(minXMaxYIndex);
        for (int vertexIndex = 1; vertexIndex < numberOfVertices; ++vertexIndex) {
            Point2DReadOnly candidate = vertices.get(vertexIndex);
            if (candidate.getX() < minXMaxY.getX()) {
                minXMaxYIndex = vertexIndex;
                minXMaxY = candidate;
                continue;
            }
            if (candidate.getX() != minXMaxY.getX() || !(candidate.getY() > minXMaxY.getY())) continue;
            minXMaxYIndex = vertexIndex;
            minXMaxY = candidate;
        }
        return minXMaxYIndex;
    }

    public static int findVertexIndex(Vertex2DSupplier vertex2DSupplier, boolean isXPriority, Bound xBound, Bound yBound) {
        if (vertex2DSupplier.getNumberOfVertices() == 0) {
            return -1;
        }
        Bound bound1 = isXPriority ? xBound : yBound;
        Bound bound2 = isXPriority ? yBound : xBound;
        int coord1 = isXPriority ? 0 : 1;
        int coord2 = isXPriority ? 1 : 0;
        int bestIndex = 0;
        Point2DReadOnly vertex = vertex2DSupplier.getVertex(bestIndex);
        double bestCoord1 = vertex.getElement(coord1);
        double bestCoord2 = vertex.getElement(coord2);
        for (int vertexIndex = 1; vertexIndex < vertex2DSupplier.getNumberOfVertices(); ++vertexIndex) {
            vertex = vertex2DSupplier.getVertex(vertexIndex);
            double candidateCoord1 = vertex.getElement(coord1);
            double candidateCoord2 = vertex.getElement(coord2);
            if (bound1.isFirstBetter(candidateCoord1, bestCoord1)) {
                bestIndex = vertexIndex;
                bestCoord1 = candidateCoord1;
                bestCoord2 = candidateCoord2;
                continue;
            }
            if (candidateCoord1 != bestCoord1 || !bound2.isFirstBetter(candidateCoord2, bestCoord2)) continue;
            bestIndex = vertexIndex;
            bestCoord1 = candidateCoord1;
            bestCoord2 = candidateCoord2;
        }
        return bestIndex;
    }

    public static int wrap(int index, int listSize) {
        if ((index %= listSize) < 0) {
            index += listSize;
        }
        return index;
    }

    public static int next(int index, int listSize) {
        return EuclidGeometryPolygonTools.wrap(index + 1, listSize);
    }

    public static int previous(int index, int listSize) {
        return EuclidGeometryPolygonTools.wrap(index - 1, listSize);
    }

    private static void checkEdgeIndex(int edgeIndex, int numberOfVertices) {
        if (edgeIndex < 0 || edgeIndex >= numberOfVertices) {
            throw new IndexOutOfBoundsException("Expected edgeIndex to be in [0; numberOfVertices[, but was: " + edgeIndex);
        }
    }

    private static void checkNumberOfVertices(List<? extends Point2DReadOnly> convexPolygon2D, int numberOfVertices) {
        if (numberOfVertices < 0 || numberOfVertices > convexPolygon2D.size()) {
            throw new IllegalArgumentException("Illegal numberOfVertices: " + numberOfVertices + ", expected a value in ] 0, " + convexPolygon2D.size() + "].");
        }
    }
}

