/*
========================================================================
SchemaCrawler
http://www.schemacrawler.com
Copyright (c) 2000-2017, Sualeh Fatehi <sualeh@hotmail.com>.
All rights reserved.
------------------------------------------------------------------------

SchemaCrawler is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

SchemaCrawler and the accompanying materials are made available under
the terms of the Eclipse Public License v1.0, GNU General Public License
v3 or GNU Lesser General Public License v3.

You may elect to redistribute this code under any of these licenses.

The Eclipse Public License is available at:
http://www.eclipse.org/legal/epl-v10.html

The GNU General Public License v3 and the GNU Lesser General Public
License v3 are available at:
http://www.gnu.org/licenses/

========================================================================
*/

package schemacrawler.tools.text.base;


import static sf.util.IOUtility.readResourceFully;
import static sf.util.Utility.isBlank;

import schemacrawler.schema.CrawlInfo;
import schemacrawler.schema.DatabaseInfo;
import schemacrawler.schema.JdbcDriverInfo;
import schemacrawler.schema.SchemaCrawlerInfo;
import schemacrawler.schemacrawler.SchemaCrawlerException;
import schemacrawler.tools.options.OutputOptions;
import schemacrawler.tools.options.TextOutputFormat;
import schemacrawler.tools.text.utility.html.Alignment;
import schemacrawler.tools.text.utility.html.TableCell;
import schemacrawler.tools.text.utility.html.TableRow;
import sf.util.Color;

/**
 * Text formatting of schema.
 *
 * @author Sualeh Fatehi
 */
public abstract class BaseDotFormatter<O extends BaseTextOptions>
  extends BaseFormatter<O>
{

  protected BaseDotFormatter(final O options,
                             final boolean printVerboseDatabaseInfo,
                             final OutputOptions outputOptions)
    throws SchemaCrawlerException
  {
    super(options, printVerboseDatabaseInfo, outputOptions);
  }

  @Override
  public void begin()
  {
    final String text = readResourceFully("/dot.header.txt");
    formattingHelper.append(text).println();
  }

  @Override
  public void end()
    throws SchemaCrawlerException
  {
    formattingHelper.append("}").println();

    super.end();
  }

  @Override
  public void handle(final CrawlInfo crawlInfo)
  {
    if (options.isNoInfo() || crawlInfo == null)
    {
      return;
    }

    TableRow row;

    final String title = crawlInfo.getTitle();
    if (!isBlank(title))
    {
      row = new TableRow(TextOutputFormat.html);
      row.add(newTableCell(title, Alignment.left, true, Color.white, 2));

      formattingHelper.append(row.toString()).println();
    }

    row = new TableRow(TextOutputFormat.html);
    row.add(newTableCell("generated by",
                         Alignment.right,
                         false,
                         Color.white,
                         1));
    row.add(newTableCell(crawlInfo.getSchemaCrawlerInfo(),
                         Alignment.left,
                         false,
                         Color.white,
                         1));

    formattingHelper.append(row.toString()).println();

    row = new TableRow(TextOutputFormat.html);
    row.add(newTableCell("generated on",
                         Alignment.right,
                         false,
                         Color.white,
                         1));
    row.add(newTableCell(formatTimestamp(crawlInfo.getCrawlTimestamp()),
                         Alignment.left,
                         false,
                         Color.white,
                         1));

    formattingHelper.append(row.toString()).println();

    row = new TableRow(TextOutputFormat.html);
    row.add(newTableCell("database version",
                         Alignment.right,
                         false,
                         Color.white,
                         1));
    row.add(newTableCell(crawlInfo.getDatabaseInfo(),
                         Alignment.left,
                         false,
                         Color.white,
                         1));

    formattingHelper.append(row.toString()).println();
  }

  @Override
  public void handle(final DatabaseInfo dbInfo)
  {
    // No-op
  }

  @Override
  public void handle(final JdbcDriverInfo driverInfo)
  {
    // No-op
  }

  @Override
  public void handle(final SchemaCrawlerInfo schemaCrawlerInfo)
  {
    // No-op
  }

  @Override
  public void handleHeaderEnd()
    throws SchemaCrawlerException
  {
    if (options.isNoInfo())
    {
      return;
    }

    formattingHelper.append("      </table>    >").println();
    formattingHelper.append("    labeljust=r").println();
    formattingHelper.append("    labelloc=b").println();
    formattingHelper.append("  ];").println();
    formattingHelper.println();
  }

  @Override
  public void handleHeaderStart()
    throws SchemaCrawlerException
  {
    if (options.isNoInfo())
    {
      return;
    }
    formattingHelper.append("  graph [fontcolor=\"#888888\", ").println();
    formattingHelper.append("    label=<").println();
    formattingHelper
      .append("<table color=\"#888888\" border=\"1\" cellborder=\"0\" cellspacing=\"0\">")
      .println();
  }

  @Override
  public void handleInfoEnd()
    throws SchemaCrawlerException
  {
    // No-op
  }

  @Override
  public void handleInfoStart()
    throws SchemaCrawlerException
  {
    // No-op
  }

  protected TableCell newTableCell(final String text,
                                   final Alignment align,
                                   final boolean emphasizeText,
                                   final Color bgColor,
                                   final int colspan)
  {
    return new TableCell(text,
                         true,
                         0,
                         align,
                         emphasizeText,
                         "",
                         bgColor,
                         colspan,
                         TextOutputFormat.html);
  }

}
