/*
 * Copyright (c) 2017, John May
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 *   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 *   WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *   DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
 *   ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *   (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *   LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 *   ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *   The views and conclusions contained in the software and documentation are those
 *   of the authors and should not be interpreted as representing official policies,
 *   either expressed or implied, of the FreeBSD Project.
 */

package uk.ac.ebi.beam;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

class ReassignDbStereo
{

    enum Config {
        Together,
        Opposite,
        Undefined
    }

    // helper class allows fast access to substituents
    private static final class DbStereo {
        Edge[] uEdges, vEdges;
        int u, v;
        Config  config;
        boolean mark;

        private DbStereo(int u, int v, Edge[] uEdges, Edge[] vEdges) {
            this.u = u;
            this.v = v;
            this.uEdges = uEdges;
            this.vEdges = vEdges;

            Bond fst = dirOfFirst(uEdges, u);
            Bond snd = dirOfFirst(vEdges, v);

            if (!fst.directional() && snd.directional()) {
                config = Config.Undefined;
            }
            else {
                if (fst == snd)
                    config = Config.Together;
                else
                    config = Config.Opposite;
            }
        }

        private static Bond dirOfFirst(Edge[] e, int u) {
            if (e.length == 1)
                return e[0].bond(u);
            if (e[0].bond().directional())
                return e[0].bond(u);
            else
                return e[1].bond(u).inverse();
        }

        private static DbStereo create(Graph g, Edge e) {
            assert e.bond().order() == 2;
            int u = e.either();
            int v = e.other(u);
            
            // ensure we read the same way round
            if (u > v) {
                u = v;
                v = e.either();
            }

            Edge[] uEdges = new Edge[g.degree(u) - 1];
            Edge[] vEdges = new Edge[g.degree(v) - 1];

            int i = 0;
            for (Edge f : g.edges(u)) {
                if (f != e)
                    uEdges[i++] = f;
            }
            i = 0;
            for (Edge f : g.edges(v)) {
                if (f != e)
                    vEdges[i++] = f;
            }

            return new DbStereo(u, v, uEdges, vEdges);
        }

        @Override public String toString() {
            return u + "=" + v + ": " + config;
        }
    }

    static void assign(final Graph g) {

        if (g.getFlags(Graph.HAS_BND_STRO) == 0)
            return;

        Map<Integer, DbStereo> dbMap  = new HashMap<>();
        List<DbStereo>         dbList = new ArrayList<>();

        // find all double bonds
        for (int v = 0; v < g.order(); ++v) {
            final int d = g.degree(v);
            if (g.bondedValence(v) - d == 1) {
                for (int j = 0; j < d; j++) {
                    Edge edge = g.edgeAt(v, j);
                    if (edge.bond().order() == 2 && edge.other(v) > v && g.degree(v) > 1 && g.degree(edge.other(v)) > 1) {
                        
                        DbStereo dbStereo = DbStereo.create(g, edge);
                        if (inSmallRing(g, edge)) {
                            dbStereo.config = Config.Undefined;
                        }
                        dbMap.put(dbStereo.u, dbStereo);
                        dbMap.put(dbStereo.v, dbStereo);
                        dbList.add(dbStereo);
                    }
                }
            }
        }

        // remove current directional bonds
        for (DbStereo dbStereo : dbList) {
            for (Edge e : dbStereo.uEdges)
                e.bond(Bond.IMPLICIT);
            for (Edge f : dbStereo.vEdges)
                f.bond(Bond.IMPLICIT);
        }

        // reassign from the lowest one first
        Collections.sort(dbList, new Comparator<DbStereo>() {
            @Override public int compare(DbStereo a, DbStereo b) {
                int cmp = a.u - b.u;
                if (cmp != 0) return cmp;
                return a.v - b.v;
            }
        });
        
        for (DbStereo dbStereo : dbList) {
            if (dbStereo.mark || dbStereo.config == Config.Undefined)
                continue;
            // set and propagate
            if (setConfig(g, dbStereo, dbMap)) {
                System.err.println("Warning - Double-bond stereo assignment");
            }
        }

    }

    static boolean inSmallRing(Graph g, Edge e) {
        return inSmallRing(g, e.either(), e.other(e.either()), e.other(e.either()), 1, new BitSet());
    }

    static boolean inSmallRing(Graph g, int v, int prev, int t, int d, BitSet visit) {
        if (d > 7)
            return false;
        if (v == t)
            return true;
        if (visit.get(v))
            return false;
        visit.set(v);
        final int deg = g.degree(v);
        for (int j = 0; j < deg; ++j) {
            final Edge e = g.edgeAt(v, j);
            int w = e.other(v);
            if (w == prev) continue;
            if (inSmallRing(g, w, v, t, d + 1, visit)) {
                return true;
            }
        }
        visit.clear(v);
        return false;
    }

    static void setFirstConfig(final DbStereo dbStereo) {
        Bond fst = Bond.DOWN;
        Bond snd = fst;

        if (dbStereo.config == Config.Opposite)
            snd = snd.inverse();

        dbStereo.uEdges[0].bond(dbStereo.uEdges[0].either() == dbStereo.u ? fst : fst.inverse());
        dbStereo.vEdges[0].bond(dbStereo.vEdges[0].either() == dbStereo.v ? snd : snd.inverse());
        dbStereo.mark = true;
    }

    static boolean setConfig(Graph g, DbStereo dbStereo, Map<Integer, DbStereo> dbStereoMap) {

        if (dbStereo.mark || dbStereo.config == Config.Undefined)
            return false;
        
        Bond left = DbStereo.dirOfFirst(dbStereo.uEdges, dbStereo.u);
        Bond right = DbStereo.dirOfFirst(dbStereo.vEdges, dbStereo.v);

        // none defined
        if (!left.directional() && !right.directional()) {
            left = Bond.DOWN;
            right = dbStereo.config == Config.Opposite ? left.inverse() : left;
            dbStereo.uEdges[0].bond(dbStereo.uEdges[0].either() == dbStereo.u ? left : left.inverse());
            dbStereo.vEdges[0].bond(dbStereo.vEdges[0].either() == dbStereo.v ? right : right.inverse());
        }
        // left has not been defined
        else if (!left.directional()) {
            left = dbStereo.config == Config.Opposite ? right.inverse() : right;
            dbStereo.uEdges[0].bond(dbStereo.uEdges[0].either() == dbStereo.u ? left : left.inverse());
        }
        // right is not defined
        else if (!right.directional()) {
            right = dbStereo.config == Config.Opposite ? left.inverse() : left;
            dbStereo.vEdges[0].bond(dbStereo.vEdges[0].either() == dbStereo.v ? right : right.inverse());
        }
        // both defined check they are correct
        else {
            if (dbStereo.config == Config.Together && left != right)
                return true;
        }
        dbStereo.mark = true;

        boolean res = false;

        int d = g.degree(dbStereo.u);
        for (int j = 0; j < d; j++) {
            final Edge e = g.edgeAt(dbStereo.u, j);
            if (dbStereoMap.containsKey(e.other(dbStereo.u))) {
                res = setConfig(g, dbStereoMap.get(e.other(dbStereo.u)), dbStereoMap) || res;
            }
        }
        d = g.degree(dbStereo.v);
        for (int j = 0; j < d; j++) {
            final Edge e = g.edgeAt(dbStereo.v, j);
            if (dbStereoMap.containsKey(e.other(dbStereo.v))) {
                res = setConfig(g, dbStereoMap.get(e.other(dbStereo.v)), dbStereoMap) || res;
            }
        }

        return res;
    }

    static Config config(final Graph g, final Edge e) {
        final int u = e.either();
        final int v = e.other(u);
        final Bond ub = firstBond(g, u);
        final Bond vb = firstBond(g, v);
        if (ub == Bond.IMPLICIT || vb == Bond.IMPLICIT)
            return Config.Undefined;
        return ub == vb ? Config.Together : Config.Opposite;
    }

    static Bond firstBond(final Graph g, final int v) {
        final int d = g.degree(v);
        boolean seenimplied = false;
        for (int j = 0; j < d; ++j) {
            final Edge e = g.edgeAt(v, j);
            if (e.bond().directional())
                return seenimplied ? e.bond(v).inverse() : e.bond(v);
            else if (e.bond().order() == 1)
                seenimplied = true;
        }
        return Bond.IMPLICIT;
    }
}
