/*
 * Decompiled with CFR 0.152.
 */
package com.mongodb.internal.connection;

import com.mongodb.AuthenticationMechanism;
import com.mongodb.MongoCredential;
import com.mongodb.MongoException;
import com.mongodb.MongoInternalException;
import com.mongodb.ServerAddress;
import com.mongodb.internal.connection.AuthorizationHeader;
import com.mongodb.internal.connection.MongoCredentialWithCache;
import com.mongodb.internal.connection.SaslAuthenticator;
import com.mongodb.lang.NonNull;
import com.mongodb.lang.Nullable;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import javax.security.sasl.SaslClient;
import javax.security.sasl.SaslException;
import org.bson.BsonBinary;
import org.bson.BsonBinaryWriter;
import org.bson.BsonDocument;
import org.bson.BsonInt32;
import org.bson.BsonString;
import org.bson.BsonWriter;
import org.bson.RawBsonDocument;
import org.bson.codecs.BsonDocumentCodec;
import org.bson.codecs.EncoderContext;
import org.bson.io.BasicOutputBuffer;

public class AwsAuthenticator
extends SaslAuthenticator {
    private static final String MONGODB_AWS_MECHANISM_NAME = "MONGODB-AWS";
    private static final int RANDOM_LENGTH = 32;

    public AwsAuthenticator(MongoCredentialWithCache credential) {
        super(credential);
        if (this.getMongoCredential().getAuthenticationMechanism() != AuthenticationMechanism.MONGODB_AWS) {
            throw new MongoException("Incorrect mechanism: " + this.getMongoCredential().getMechanism());
        }
    }

    @Override
    public String getMechanismName() {
        return MONGODB_AWS_MECHANISM_NAME;
    }

    @Override
    protected SaslClient createSaslClient(ServerAddress serverAddress) {
        return new AwsSaslClient(this.getMongoCredential());
    }

    private static class AwsSaslClient
    implements SaslClient {
        private final MongoCredential credential;
        private final byte[] clientNonce = new byte[32];
        private int step = -1;
        private String httpResponse;

        AwsSaslClient(MongoCredential credential) {
            this.credential = credential;
        }

        @Override
        public String getMechanismName() {
            AuthenticationMechanism authMechanism = this.credential.getAuthenticationMechanism();
            if (authMechanism == null) {
                throw new IllegalArgumentException("Authentication mechanism cannot be null");
            }
            return authMechanism.getMechanismName();
        }

        @Override
        public boolean hasInitialResponse() {
            return true;
        }

        @Override
        public byte[] evaluateChallenge(byte[] challenge) throws SaslException {
            ++this.step;
            if (this.step == 0) {
                return this.computeClientFirstMessage();
            }
            if (this.step == 1) {
                return this.computeClientFinalMessage(challenge);
            }
            throw new SaslException(String.format("Too many steps involved in the %s negotiation.", this.getMechanismName()));
        }

        @Override
        public boolean isComplete() {
            return this.step == 1;
        }

        @Override
        public byte[] unwrap(byte[] bytes, int i, int i1) {
            throw new UnsupportedOperationException("Not implemented yet!");
        }

        @Override
        public byte[] wrap(byte[] bytes, int i, int i1) {
            throw new UnsupportedOperationException("Not implemented yet!");
        }

        @Override
        public Object getNegotiatedProperty(String s) {
            throw new UnsupportedOperationException("Not implemented yet!");
        }

        @Override
        public void dispose() {
        }

        private byte[] computeClientFirstMessage() {
            new SecureRandom().nextBytes(this.clientNonce);
            BsonDocument document = new BsonDocument().append("r", new BsonBinary(this.clientNonce)).append("p", new BsonInt32(110));
            return this.toBson(document);
        }

        private byte[] computeClientFinalMessage(byte[] serverFirst) throws SaslException {
            RawBsonDocument document = new RawBsonDocument(serverFirst);
            String host = document.getString("h").getValue();
            byte[] serverNonce = document.getBinary("s").getData();
            if (serverNonce.length != 64) {
                throw new SaslException(String.format("Server nonce must be %d bytes", 64));
            }
            if (!Arrays.equals(Arrays.copyOf(serverNonce, 32), this.clientNonce)) {
                throw new SaslException(String.format("The first %d bytes of the server nonce must be the client nonce", 32));
            }
            String timestamp = DateTimeFormatter.ofPattern("yyyyMMdd'T'HHmmss'Z'").withZone(ZoneId.of("UTC")).format(Instant.now());
            String token = this.getSessionToken();
            AuthorizationHeader authorizationHeader = AuthorizationHeader.builder().setAccessKeyID(this.getUserName()).setSecretKey(this.getPassword()).setSessionToken(token).setHost(host).setNonce(serverNonce).setTimestamp(timestamp).build();
            BsonDocument ret = new BsonDocument().append("a", new BsonString(authorizationHeader.toString())).append("d", new BsonString(authorizationHeader.getTimestamp()));
            if (token != null) {
                ret.append("t", new BsonString(token));
            }
            return this.toBson(ret);
        }

        private byte[] toBson(BsonDocument document) {
            BasicOutputBuffer buffer = new BasicOutputBuffer();
            new BsonDocumentCodec().encode((BsonWriter)new BsonBinaryWriter(buffer), document, EncoderContext.builder().build());
            byte[] bytes = new byte[buffer.size()];
            System.arraycopy(buffer.getInternalBuffer(), 0, bytes, 0, buffer.getSize());
            return bytes;
        }

        @NonNull
        String getUserName() {
            String userName = this.credential.getUserName();
            if (userName == null && (userName = System.getenv("AWS_ACCESS_KEY_ID")) == null) {
                userName = BsonDocument.parse(this.getHttpResponse()).getString("AccessKeyId").getValue();
            }
            return userName;
        }

        @NonNull
        private String getPassword() {
            char[] password = this.credential.getPassword();
            if (password == null) {
                password = System.getenv("AWS_SECRET_ACCESS_KEY") != null ? System.getenv("AWS_SECRET_ACCESS_KEY").toCharArray() : BsonDocument.parse(this.getHttpResponse()).getString("SecretAccessKey").getValue().toCharArray();
            }
            return new String(password);
        }

        @Nullable
        private String getSessionToken() {
            String token = this.credential.getMechanismProperty("AWS_SESSION_TOKEN", null);
            if (this.credential.getUserName() != null) {
                return token;
            }
            if (token != null) {
                throw new IllegalArgumentException("The connection string contains a session token without credentials");
            }
            if (System.getenv("AWS_SECRET_ACCESS_KEY") != null || System.getenv("AWS_ACCESS_KEY_ID") != null || System.getenv("AWS_SESSION_TOKEN") != null) {
                if (System.getenv("AWS_SECRET_ACCESS_KEY") == null || System.getenv("AWS_ACCESS_KEY_ID") == null) {
                    throw new IllegalArgumentException("The environment variables 'AWS_ACCESS_KEY_ID' and 'AWS_SECRET_ACCESS_KEY' must either both be set or both be null");
                }
                return System.getenv("AWS_SESSION_TOKEN");
            }
            return BsonDocument.parse(this.getHttpResponse()).getString("Token").getValue();
        }

        @NonNull
        private String getHttpResponse() {
            if (this.httpResponse != null) {
                return this.httpResponse;
            }
            String path = System.getenv("AWS_CONTAINER_CREDENTIALS_RELATIVE_URI");
            this.httpResponse = path == null ? this.getEc2Response() : AwsSaslClient.getHttpContents("GET", "http://169.254.170.2" + path, null);
            return this.httpResponse;
        }

        private String getEc2Response() {
            String endpoint = "http://169.254.169.254";
            String path = "/latest/meta-data/iam/security-credentials/";
            HashMap<String, String> header = new HashMap<String, String>();
            header.put("X-aws-ec2-metadata-token-ttl-seconds", "30");
            String token = AwsSaslClient.getHttpContents("PUT", "http://169.254.169.254/latest/api/token", header);
            header.clear();
            header.put("X-aws-ec2-metadata-token", token);
            String role = AwsSaslClient.getHttpContents("GET", "http://169.254.169.254/latest/meta-data/iam/security-credentials/", header);
            return AwsSaslClient.getHttpContents("GET", "http://169.254.169.254/latest/meta-data/iam/security-credentials/" + role, header);
        }

        @NonNull
        private static String getHttpContents(String method, String endpoint, Map<String, String> headers) {
            StringBuilder content = new StringBuilder();
            HttpURLConnection conn = null;
            try {
                int status;
                conn = (HttpURLConnection)new URL(endpoint).openConnection();
                conn.setRequestMethod(method);
                conn.setReadTimeout(10000);
                if (headers != null) {
                    for (Map.Entry<String, String> kvp : headers.entrySet()) {
                        conn.setRequestProperty(kvp.getKey(), kvp.getValue());
                    }
                }
                if ((status = conn.getResponseCode()) != 200) {
                    throw new IOException(String.format("%d %s", status, conn.getResponseMessage()));
                }
                try (BufferedReader in = new BufferedReader(new InputStreamReader(conn.getInputStream(), StandardCharsets.UTF_8));){
                    String inputLine;
                    while ((inputLine = in.readLine()) != null) {
                        content.append(inputLine);
                    }
                }
            }
            catch (IOException e) {
                throw new MongoInternalException("Unexpected IOException", e);
            }
            finally {
                if (conn != null) {
                    conn.disconnect();
                }
            }
            return content.toString();
        }
    }
}

