/*
 * Decompiled with CFR 0.152.
 */
package top.fullj.timer;

import java.util.Collections;
import java.util.HashSet;
import java.util.Objects;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import top.fullj.timer.Timeout;
import top.fullj.timer.Timer;
import top.fullj.timer.TimerTask;

public class HashedWheelTimer
implements Timer {
    private static final int WORKER_INIT = 0;
    private static final int WORKER_STARTED = 1;
    private static final int WORKER_SHUTDOWN = 2;
    private final Worker worker = new Worker();
    private final AtomicInteger workerState = new AtomicInteger(0);
    private final CountDownLatch workerInitialized = new CountDownLatch(1);
    private final Thread workThread;
    private final long tickDuration;
    private final TimeWheelBucket[] wheel;
    private final int mask;
    private final Queue<TimeWheelTimeout> jobs = new LinkedBlockingQueue<TimeWheelTimeout>();
    private final Queue<TimeWheelTimeout> cancelledJobs = new LinkedBlockingQueue<TimeWheelTimeout>();
    private volatile long startTime;

    public HashedWheelTimer() {
        this(Executors.defaultThreadFactory());
    }

    public HashedWheelTimer(long tickDuration, TimeUnit unit) {
        this(Executors.defaultThreadFactory(), tickDuration, unit);
    }

    public HashedWheelTimer(long tickDuration, TimeUnit unit, int ticksPerWheel) {
        this(Executors.defaultThreadFactory(), tickDuration, unit, ticksPerWheel);
    }

    public HashedWheelTimer(ThreadFactory threadFactory) {
        this(threadFactory, 100L, TimeUnit.MILLISECONDS);
    }

    public HashedWheelTimer(ThreadFactory threadFactory, long tickDuration, TimeUnit unit) {
        this(threadFactory, tickDuration, unit, 512);
    }

    public HashedWheelTimer(ThreadFactory threadFactory, long tickDuration, TimeUnit unit, int ticksPerWheel) {
        Objects.requireNonNull(threadFactory, "threadFactory");
        Objects.requireNonNull(unit, "unit");
        if (tickDuration <= 0L) {
            throw new IllegalArgumentException("tickDuration must be greater than 0: " + tickDuration);
        }
        if (ticksPerWheel <= 0) {
            throw new IllegalArgumentException("ticksPerWheel must be greater than 0: " + ticksPerWheel);
        }
        this.wheel = HashedWheelTimer.createWheel(ticksPerWheel);
        this.mask = this.wheel.length - 1;
        this.tickDuration = unit.toNanos(tickDuration);
        this.workThread = threadFactory.newThread(this.worker);
    }

    private static TimeWheelBucket[] createWheel(int ticksPerWheel) {
        int normalizedTicksPerWheel;
        if (ticksPerWheel <= 0) {
            throw new IllegalArgumentException("ticksPerWheel must be greater than 0: " + ticksPerWheel);
        }
        if (ticksPerWheel > 0x10000000) {
            throw new IllegalArgumentException("ticksPerWheel must be less than 0x10000000: " + ticksPerWheel);
        }
        for (normalizedTicksPerWheel = 1; normalizedTicksPerWheel < ticksPerWheel; normalizedTicksPerWheel <<= 1) {
        }
        TimeWheelBucket[] wheel = new TimeWheelBucket[normalizedTicksPerWheel];
        for (int i = 0; i < wheel.length; ++i) {
            wheel[i] = new TimeWheelBucket();
        }
        return wheel;
    }

    private void start() {
        switch (this.workerState.get()) {
            case 0: {
                if (!this.workerState.compareAndSet(0, 1)) break;
                this.workThread.start();
                break;
            }
            case 1: {
                break;
            }
            case 2: {
                throw new IllegalStateException("Cannot start once stopped");
            }
            default: {
                throw new Error("Invalid worker state");
            }
        }
        while (this.startTime == 0L) {
            try {
                this.workerInitialized.await();
            }
            catch (InterruptedException interruptedException) {}
        }
    }

    @Override
    public Timeout newJob(TimerTask task, long delay, TimeUnit unit) {
        Objects.requireNonNull(task, "task");
        Objects.requireNonNull(unit, "unit");
        this.start();
        long deadline = System.nanoTime() + unit.toNanos(delay) - this.startTime;
        if (delay > 0L && deadline < 0L) {
            deadline = Long.MAX_VALUE;
        }
        TimeWheelTimeout job = new TimeWheelTimeout(this, task, deadline);
        this.jobs.offer(job);
        return job;
    }

    @Override
    public Set<Timeout> stop() {
        if (Thread.currentThread() == this.workThread) {
            throw new IllegalStateException(HashedWheelTimer.class.getSimpleName() + ".stop() can not be called from " + TimerTask.class.getSimpleName());
        }
        if (!this.workerState.compareAndSet(1, 2)) {
            if (this.workerState.getAndSet(2) != 2) {
                // empty if block
            }
            return Collections.emptySet();
        }
        while (this.workThread.isAlive()) {
            this.workThread.interrupt();
            try {
                this.workThread.join(100L);
            }
            catch (InterruptedException interruptedException) {}
        }
        return this.worker.getUnprocessedTimeouts();
    }

    private static final class TimeWheelTimeout
    implements Timeout {
        private static final int STATE_INIT = 0;
        private static final int STATE_CANCELLED = 1;
        private static final int STATE_EXPIRED = 2;
        private final HashedWheelTimer timer;
        private final TimerTask task;
        private final long deadline;
        private final AtomicInteger state = new AtomicInteger(0);
        long remainingRounds;
        TimeWheelTimeout next;
        TimeWheelTimeout prev;
        TimeWheelBucket bucket;

        TimeWheelTimeout(HashedWheelTimer timer, TimerTask task, long deadline) {
            this.timer = timer;
            this.task = task;
            this.deadline = deadline;
        }

        int state() {
            return this.state.get();
        }

        @Override
        public Timer timer() {
            return this.timer;
        }

        @Override
        public TimerTask task() {
            return this.task;
        }

        @Override
        public boolean isExpired() {
            return this.state.get() == 2;
        }

        @Override
        public boolean isCancelled() {
            return this.state.get() == 1;
        }

        @Override
        public boolean cancel() {
            if (!this.state.compareAndSet(0, 1)) {
                return false;
            }
            this.timer.cancelledJobs.offer(this);
            return true;
        }

        void expire() {
            if (!this.state.compareAndSet(0, 2)) {
                return;
            }
            try {
                this.task.run(this);
            }
            catch (Exception e) {
                e.printStackTrace();
            }
        }

        void remove() {
            TimeWheelBucket bucket = this.bucket;
            if (bucket != null) {
                bucket.remove(this);
            }
        }

        public String toString() {
            long currentTime = System.nanoTime();
            long remaining = this.deadline - currentTime + this.timer.startTime;
            StringBuilder sb = new StringBuilder(128).append(this.getClass().getName()).append('(').append("deadline: ").append(remaining).append(" ns");
            if (this.isCancelled()) {
                sb.append(", cancelled");
            }
            return sb.append(", task: ").append(this.task()).append(')').toString();
        }
    }

    private static final class TimeWheelBucket {
        private TimeWheelTimeout head;
        private TimeWheelTimeout tail;

        private TimeWheelBucket() {
        }

        void append(TimeWheelTimeout job) {
            assert (job.bucket == null);
            job.bucket = this;
            if (this.head == null) {
                this.head = this.tail = job;
            } else {
                this.tail.next = job;
                job.prev = this.tail;
                this.tail = job;
            }
        }

        /*
         * Enabled force condition propagation
         * Lifted jumps to return sites
         */
        void expire(long deadline) {
            TimeWheelTimeout job = this.head;
            while (job != null) {
                TimeWheelTimeout next = job.next;
                if (job.remainingRounds <= 0L) {
                    next = this.remove(job);
                    if (job.deadline > deadline) throw new IllegalStateException(String.format("job.deadline (%d) > deadline (%d)", job.deadline, deadline));
                    job.expire();
                } else if (job.isCancelled()) {
                    next = this.remove(job);
                } else {
                    --job.remainingRounds;
                }
                job = next;
            }
        }

        TimeWheelTimeout remove(TimeWheelTimeout job) {
            TimeWheelTimeout next = job.next;
            if (job.prev != null) {
                job.prev.next = next;
            }
            if (job.next != null) {
                job.next.prev = job.prev;
            }
            if (job == this.head) {
                if (job == this.tail) {
                    this.tail = null;
                    this.head = null;
                } else {
                    this.head = next;
                }
            } else if (job == this.tail) {
                this.tail = job.prev;
            }
            job.prev = null;
            job.next = null;
            job.bucket = null;
            return next;
        }

        void clear(Set<Timeout> set) {
            TimeWheelTimeout top;
            while ((top = this.poll()) != null) {
                if (top.isExpired() || top.isCancelled()) continue;
                set.add(top);
            }
        }

        private TimeWheelTimeout poll() {
            TimeWheelTimeout head = this.head;
            if (head == null) {
                return null;
            }
            TimeWheelTimeout next = head.next;
            if (next == null) {
                this.head = null;
                this.tail = null;
            } else {
                this.head = next;
                next.prev = null;
            }
            head.prev = null;
            head.next = null;
            head.bucket = null;
            return head;
        }
    }

    private final class Worker
    implements Runnable {
        private final Set<Timeout> unprocessedTimeouts = new HashSet<Timeout>();
        private long tick;

        private Worker() {
        }

        @Override
        public void run() {
            TimeWheelTimeout job;
            Thread.currentThread().setName(HashedWheelTimer.class.getSimpleName());
            HashedWheelTimer.this.startTime = System.nanoTime();
            if (HashedWheelTimer.this.startTime == 0L) {
                HashedWheelTimer.this.startTime = 1L;
            }
            HashedWheelTimer.this.workerInitialized.countDown();
            do {
                long deadline;
                if ((deadline = this.waitForNextTick()) <= 0L) continue;
                this.removeCancelledJobs();
                this.transferJobsToBucket();
                int idx = (int)(this.tick & (long)HashedWheelTimer.this.mask);
                HashedWheelTimer.this.wheel[idx].expire(deadline);
                ++this.tick;
            } while (HashedWheelTimer.this.workerState.get() == 1);
            for (TimeWheelBucket bucket : HashedWheelTimer.this.wheel) {
                bucket.clear(this.unprocessedTimeouts);
            }
            while ((job = (TimeWheelTimeout)HashedWheelTimer.this.jobs.poll()) != null) {
                if (job.isCancelled()) continue;
                this.unprocessedTimeouts.add(job);
            }
            this.removeCancelledJobs();
        }

        private void transferJobsToBucket() {
            TimeWheelTimeout job;
            for (int i = 0; i < 10000 && (job = (TimeWheelTimeout)HashedWheelTimer.this.jobs.poll()) != null; ++i) {
                if (job.state() == 1) continue;
                long totalTick = job.deadline / HashedWheelTimer.this.tickDuration;
                job.remainingRounds = (totalTick - this.tick) / (long)HashedWheelTimer.this.wheel.length;
                totalTick = Math.max(totalTick, this.tick);
                int idx = (int)(totalTick & (long)HashedWheelTimer.this.mask);
                HashedWheelTimer.this.wheel[idx].append(job);
            }
        }

        private void removeCancelledJobs() {
            TimeWheelTimeout job;
            while ((job = (TimeWheelTimeout)HashedWheelTimer.this.cancelledJobs.poll()) != null) {
                job.remove();
            }
        }

        /*
         * Enabled force condition propagation
         * Lifted jumps to return sites
         */
        private long waitForNextTick() {
            long deadline = HashedWheelTimer.this.tickDuration * (this.tick + 1L);
            while (true) {
                long currentTime;
                long sleepMs;
                if ((sleepMs = (deadline - (currentTime = System.nanoTime() - HashedWheelTimer.this.startTime) + 999999L) / 1000000L) <= 0L) {
                    if (currentTime == Long.MIN_VALUE) {
                        return -9223372036854775807L;
                    }
                    long l = currentTime;
                    return l;
                }
                try {
                    Thread.sleep(sleepMs);
                    continue;
                }
                catch (InterruptedException ignored) {
                    if (HashedWheelTimer.this.workerState.get() == 2) return Long.MIN_VALUE;
                    continue;
                }
                break;
            }
        }

        Set<Timeout> getUnprocessedTimeouts() {
            return Collections.unmodifiableSet(this.unprocessedTimeouts);
        }
    }
}

