package threads.webrtc;

import android.content.Context;
import android.content.SharedPreferences;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import org.webrtc.IceCandidate;
import org.webrtc.SessionDescription;

import static androidx.core.util.Preconditions.checkArgument;
import static androidx.core.util.Preconditions.checkNotNull;


public class RTCClient implements RTCSession.Listener {
    private static final String AUDIO_CODEC_KEY = "audioCodecKey";
    private static final String VIDEO_CODEC_KEY = "videoCodecKey";
    private static final String AUDIO_PROCESSING_KEY = "audioProcessingEnabledKey";
    private static final String OPEN_SL_ES_KEY = "openSlEsKey";
    private static final String AEC_KEY = "aecKey";
    private static final String AGC_KEY = "agcKey";
    private static final String HNS_KEY = "hnsKey";
    private static final String PREF_KEY = "prefKey";
    private static final String TIMEOUT_KEY = "timeoutKey";
    private final SignalingEvents events;
    private final String topic;

    public RTCClient(@NonNull String topic, @NonNull SignalingEvents events) {
        checkNotNull(topic);
        checkNotNull(events);
        this.events = events;
        this.topic = topic;

        RTCSession.getInstance().setListener(this);
    }

    @NonNull
    public static String getAudioCodec(@NonNull Context context) {
        checkNotNull(context);
        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        return sharedPref.getString(AUDIO_CODEC_KEY, "opus");
    }

    public static void setAudioCodec(@NonNull Context context, @NonNull String codec) {
        checkNotNull(context);
        checkNotNull(codec);
        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putString(AUDIO_CODEC_KEY, codec);
        editor.apply();
    }

    @NonNull
    public static String getVideoCodec(@NonNull Context context) {
        checkNotNull(context);
        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        return sharedPref.getString(VIDEO_CODEC_KEY, "VP9");
    }

    public static void setVideoCodec(@NonNull Context context, @NonNull String codec) {
        checkNotNull(context);
        checkNotNull(codec);
        SharedPreferences sharedPref = context.getSharedPreferences(
                PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putString(VIDEO_CODEC_KEY, codec);
        editor.apply();
    }

    public static boolean isAutomaticGainControlEnabled(@NonNull Context context) {
        checkNotNull(context);
        SharedPreferences sharedPref = context.getSharedPreferences(PREF_KEY, Context.MODE_PRIVATE);
        return sharedPref.getBoolean(AGC_KEY, true);
    }

    public static void setAutomaticGainControlEnabled(@NonNull Context context, boolean enable) {
        checkNotNull(context);
        SharedPreferences sharedPref = context.getSharedPreferences(PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putBoolean(AGC_KEY, enable);
        editor.apply();
    }

    public static int getConnectionTimeout(@NonNull Context context) {
        checkNotNull(context);
        SharedPreferences sharedPref = context.getSharedPreferences(PREF_KEY, Context.MODE_PRIVATE);
        return sharedPref.getInt(TIMEOUT_KEY, 30);
    }

    public static void setConnectionTimeout(@NonNull Context context, int timeout) {
        checkNotNull(context);
        checkArgument(timeout >= 0);
        SharedPreferences sharedPref = context.getSharedPreferences(PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putInt(TIMEOUT_KEY, timeout);
        editor.apply();
    }

    public static boolean isHardwareNoiseSuppressorEnabled(@NonNull Context context) {
        checkNotNull(context);
        SharedPreferences sharedPref = context.getSharedPreferences(PREF_KEY, Context.MODE_PRIVATE);
        return sharedPref.getBoolean(HNS_KEY, true);
    }

    public static void setHardwareNoiseSuppressorEnabled(@NonNull Context context, boolean enable) {
        checkNotNull(context);
        SharedPreferences sharedPref = context.getSharedPreferences(PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putBoolean(HNS_KEY, enable);
        editor.apply();
    }

    public static boolean isAcousticEchoCancelerEnabled(@NonNull Context context) {
        checkNotNull(context);
        SharedPreferences sharedPref = context.getSharedPreferences(PREF_KEY, Context.MODE_PRIVATE);
        return sharedPref.getBoolean(AEC_KEY, true);
    }

    public static void setAcousticEchoCancelerEnabled(@NonNull Context context, boolean enable) {
        checkNotNull(context);
        SharedPreferences sharedPref = context.getSharedPreferences(PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putBoolean(AEC_KEY, enable);
        editor.apply();
    }

    public static boolean isOpenSlESEnabled(@NonNull Context context) {
        checkNotNull(context);
        SharedPreferences sharedPref = context.getSharedPreferences(PREF_KEY, Context.MODE_PRIVATE);
        return sharedPref.getBoolean(OPEN_SL_ES_KEY, true);
    }

    public static void setOpenSlESEnabled(@NonNull Context context, boolean enable) {
        checkNotNull(context);
        SharedPreferences sharedPref = context.getSharedPreferences(PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putBoolean(OPEN_SL_ES_KEY, enable);
        editor.apply();
    }

    public static boolean isAudioProcessingEnabled(@NonNull Context context) {
        checkNotNull(context);
        SharedPreferences sharedPref = context.getSharedPreferences(PREF_KEY, Context.MODE_PRIVATE);
        return sharedPref.getBoolean(AUDIO_PROCESSING_KEY, true);
    }

    public static void setAudioProcessingEnabled(@NonNull Context context, boolean enable) {
        checkNotNull(context);
        SharedPreferences sharedPref = context.getSharedPreferences(PREF_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putBoolean(AUDIO_PROCESSING_KEY, enable);
        editor.apply();
    }

    @Override
    public void busy(@NonNull String pid) {
        events.onChannelBusy();
    }

    @Override
    public void accept(@NonNull String pid, @Nullable String[] ices) {
        events.onChannelAccepted(ices);
    }

    @Override
    public void reject(@NonNull String pid) {
        events.onChannelReject();
    }

    @Override
    public void offer(@NonNull String pid, @NonNull String sdp) {
        checkNotNull(sdp);

        events.onChannelOffer(new SessionDescription(
                SessionDescription.Type.OFFER, sdp));
    }

    @Override
    public void answer(@NonNull String pid,
                       @NonNull String sdp,
                       @NonNull String type) {
        checkNotNull(sdp);
        checkNotNull(type);
        events.onRemoteDescription(new SessionDescription(
                SessionDescription.Type.fromCanonicalForm(type), sdp));
    }

    @Override
    public void candidate(@NonNull String pid,
                          @NonNull String sdp,
                          @NonNull String mid,
                          @NonNull String index) {
        checkNotNull(sdp);
        checkNotNull(mid);
        checkNotNull(index);
        events.onRemoteIceCandidate(new IceCandidate(
                mid,
                Integer.valueOf(index),
                sdp));
    }

    @Override
    public void candidate_remove(@NonNull String topic,
                                 @NonNull String sdp,
                                 @NonNull String mid,
                                 @NonNull String index) {
        checkNotNull(sdp);
        checkNotNull(mid);
        checkNotNull(index);
        events.onRemoteIceCandidateRemoved(new IceCandidate(
                mid,
                Integer.valueOf(index),
                sdp));
    }

    @Override
    public void close(@NonNull String pid) {
        events.onChannelClose();
    }

    @Override
    public void timeout(@NonNull String pid) {
        events.onChannelTimeout();
    }


    public void sendOfferSdp(@NonNull Context context, @NonNull final SessionDescription sdp) {
        checkNotNull(context);
        checkNotNull(sdp);
        RTCSession.getInstance().emitSessionOffer(context, topic, sdp);
    }


    public void sendAnswerSdp(@NonNull Context context, @NonNull final SessionDescription sdp) {
        checkNotNull(context);
        checkNotNull(sdp);
        RTCSession.getInstance().emitSessionAnswer(context, topic, sdp);
    }


    public void sendLocalIceCandidate(@NonNull Context context, @NonNull final IceCandidate candidate) {
        checkNotNull(context);
        checkNotNull(candidate);
        RTCSession.getInstance().emitIceCandidate(context, topic, candidate);
    }

    public void sendLocalIceCandidateRemovals(@NonNull Context context,
                                              @Nullable final IceCandidate[] candidates) {
        checkNotNull(context);
        if (candidates != null) {
            RTCSession.getInstance().emitIceCandidatesRemove(context,
                    topic, candidates);
        }
    }


    /**
     * Callback interface for messages delivered on signaling channel.
     *
     * <p>Methods are guaranteed to be invoked on the UI thread of |activity|.
     */
    public interface SignalingEvents {


        /**
         * Callback fired once the room's signaling parameters
         * SignalingParameters are extracted.
         */
        void onChannelOffer(final SessionDescription sdp);

        /**
         * Callback fired once remote SDP is received.
         */
        void onRemoteDescription(final SessionDescription sdp);

        /**
         * Callback fired once remote Ice candidate is received.
         */
        void onRemoteIceCandidate(final IceCandidate candidate);

        /**
         * Callback fired once remote Ice candidate removals are received.
         */
        void onRemoteIceCandidateRemoved(final IceCandidate candidate);

        /**
         * Callback fired once channel is closed.
         */
        void onChannelClose();

        /**
         * Callback fired once channel is busy.
         */
        void onChannelBusy();

        void onChannelTimeout();

        void onChannelReject();

        void onChannelAccepted(@Nullable String[] ices);
    }


}
