package threads.webrtc;

import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.content.Context;
import android.media.AudioAttributes;
import android.media.AudioManager;
import android.net.Uri;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.util.Preconditions;

import com.google.gson.Gson;

import org.webrtc.IceCandidate;
import org.webrtc.SessionDescription;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicBoolean;

import threads.core.Singleton;
import threads.core.THREADS;
import threads.core.api.Addresses;
import threads.core.api.Content;
import threads.ipfs.IPFS;
import threads.ipfs.api.PeerInfo;
import threads.ipfs.api.PubsubInfo;
import threads.ipfs.api.PubsubReader;

import static androidx.core.util.Preconditions.checkArgument;
import static androidx.core.util.Preconditions.checkNotNull;

public class RTCSession {
    public static final String RTC_CHANNEL_ID = "RTC_CHANNEL_ID";
    public static final int RTC_CALL_ID = 100;
    private static final int PUBSUB_TIMEOUT = 50;
    private static final String TAG = RTCSession.class.getSimpleName();
    private static final Gson gson = new Gson();
    private static final Hashtable<String, Future> topics = new Hashtable<>();
    private static RTCSession INSTANCE = new RTCSession();
    private final AtomicBoolean busy = new AtomicBoolean(false);
    @Nullable
    private Listener listener = null;

    private RTCSession() {
    }

    public static void createRTCChannel(@NonNull Context context) {
        checkNotNull(context);
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.O) {
            try {
                CharSequence name = context.getString(R.string.rtc_channel_name);
                String description = context.getString(R.string.rtc_channel_description);
                int importance = NotificationManager.IMPORTANCE_HIGH;
                NotificationChannel mChannel = new NotificationChannel(RTC_CHANNEL_ID, name, importance);
                mChannel.setDescription(description);

                mChannel.setSound(
                        Uri.parse("android.resource://" + context.getPackageName() + "/" + R.raw.incoming),
                        new AudioAttributes.Builder().setContentType(
                                AudioAttributes.CONTENT_TYPE_SONIFICATION)
                                .setLegacyStreamType(AudioManager.STREAM_VOICE_CALL)
                                .setUsage(AudioAttributes.USAGE_VOICE_COMMUNICATION).build());

                NotificationManager notificationManager = (NotificationManager) context.getSystemService(
                        Context.NOTIFICATION_SERVICE);
                if (notificationManager != null) {
                    notificationManager.createNotificationChannel(mChannel);
                }

            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage(), e);
            }
        }
    }

    public static RTCSession getInstance() {
        return INSTANCE;
    }

    public static void handleContent(@NonNull Context context,
                                     @NonNull String topic,
                                     @NonNull Content content) {
        checkNotNull(context);
        checkNotNull(topic);
        checkNotNull(content);
        try {
            String est = content.get(Content.EST);
            RTCSession.Event type = RTCSession.Event.valueOf(est);
            switch (type) {
                case SESSION_VIDEO_CALL: {

                    if (RTCSession.getInstance().isBusy()) {
                        RTCSession.getInstance().emitSessionBusy(context, topic);
                    } else {


                        String adds = content.get(Content.ICES);
                        String[] ices = null;
                        if (adds != null) {
                            Addresses addresses = Addresses.toAddresses(adds);
                            ices = RTCSession.getInstance().turnUris(addresses);
                        }

                        RTCVideoCallActivity.createVideoCallNotification(context, topic, ices);
                    }
                    break;
                }
                case SESSION_CALL: {

                    if (RTCSession.getInstance().isBusy()) {
                        RTCSession.getInstance().emitSessionBusy(context, topic);
                    } else {

                        String adds = content.get(Content.ICES);
                        String[] ices = null;
                        if (adds != null) {
                            Addresses addresses = Addresses.toAddresses(adds);
                            ices = RTCSession.getInstance().turnUris(addresses);
                        }

                        RTCCallActivity.createCallNotification(context, topic, ices);
                    }
                    break;
                }
                case SESSION_TIMEOUT: {
                    RTCSession.getInstance().timeout(topic);
                    NotificationManager notificationManager = (NotificationManager)
                            context.getSystemService(Context.NOTIFICATION_SERVICE);
                    if (notificationManager != null) {
                        notificationManager.cancel(RTC_CALL_ID);
                    }
                    break;
                }
                case SESSION_BUSY: {
                    RTCSession.getInstance().busy(topic);
                    NotificationManager notificationManager = (NotificationManager)
                            context.getSystemService(Context.NOTIFICATION_SERVICE);
                    if (notificationManager != null) {
                        notificationManager.cancel(RTC_CALL_ID);
                    }
                    break;
                }
                case SESSION_ACCEPT: {
                    String[] ices = null;
                    String adds = content.get(Content.ICES);
                    if (adds != null) {
                        Addresses addresses = Addresses.toAddresses(adds);
                        ices = RTCSession.getInstance().turnUris(addresses);
                    }
                    RTCSession.getInstance().accept(topic, ices);
                    break;
                }
                case SESSION_REJECT: {
                    RTCSession.getInstance().reject(topic);
                    break;
                }
                case SESSION_OFFER: {
                    String sdp = content.get(Content.SDP);
                    checkNotNull(sdp);
                    RTCSession.getInstance().offer(topic, sdp);
                    break;
                }
                case SESSION_ANSWER: {
                    String sdp = content.get(Content.SDP);
                    checkNotNull(sdp);
                    String esk = content.get(Content.TYPE);
                    checkNotNull(esk);
                    RTCSession.getInstance().answer(topic, sdp, esk);
                    break;
                }
                case SESSION_CANDIDATE_REMOVE: {
                    String sdp = content.get(Content.SDP);
                    checkNotNull(sdp);
                    String mid = content.get(Content.MID);
                    checkNotNull(mid);
                    String index = content.get(Content.INDEX);
                    checkNotNull(index);
                    RTCSession.getInstance().candidate_remove(
                            topic, sdp, mid, index);
                    break;
                }
                case SESSION_CANDIDATE: {
                    String sdp = content.get(Content.SDP);
                    checkNotNull(sdp);
                    String mid = content.get(Content.MID);
                    checkNotNull(mid);
                    String index = content.get(Content.INDEX);
                    checkNotNull(index);
                    RTCSession.getInstance().candidate(topic, sdp, mid, index);
                    break;
                }
                case SESSION_CLOSE: {
                    RTCSession.getInstance().close(topic);
                    NotificationManager notificationManager = (NotificationManager)
                            context.getSystemService(Context.NOTIFICATION_SERVICE);
                    if (notificationManager != null) {
                        notificationManager.cancel(RTC_CALL_ID);
                    }
                    break;
                }
                case SESSION_ALIVE: {
                    // NOTHING to do here
                    break;
                }
            }
        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }
    }

    private static boolean shouldRun(long start, long timeout) {
        checkArgument(start > 0);
        checkArgument(timeout > 0);
        long passedTime = System.currentTimeMillis() - start;
        return passedTime < timeout;
    }

    static void connectTopic(@NonNull Context context, @NonNull String topic) {
        checkNotNull(context);
        checkNotNull(topic);

        // already listen to topic
        if (topics.contains(topic)) {
            return;
        }

        final THREADS threads = Singleton.getInstance(context).getThreads();
        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        checkNotNull(ipfs, "IPFS not valid");

        ExecutorService executor = Executors.newSingleThreadExecutor();
        topics.put(topic, executor.submit(() -> {
            try {
                ipfs.pubsubSub(new PubsubReader() {
                    @Override
                    public void receive(@NonNull PubsubInfo message) {

                        try {
                            if (!threads.isUserBlocked(message.getSenderPid())) {
                                Content content = gson.fromJson(message.getMessage(), Content.class);
                                if (content != null) {
                                    RTCSession.handleContent(context, topic, content);
                                }
                            }
                        } catch (Throwable e) {
                            Log.e(TAG, "" + e.getLocalizedMessage(), e);
                        } finally {
                            Log.e(TAG, "Receive : " + message.getMessage());
                        }

                    }

                    @NonNull
                    @Override
                    public String getTopic() {
                        return topic;
                    }
                }, true);
            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage(), e);
            }
        }));
    }

    @Nullable
    public String[] turnUris(@Nullable Addresses addresses) {
        if (addresses == null) {
            return null;
        }
        Collection<String> values = addresses.values();
        List<String> uris = new ArrayList<>();
        for (String address : values) {
            try {
                if (!address.startsWith("/ip4/127.0.0.1") && !address.startsWith("/ip6/::1/")) {
                    String[] parts = address.split("/");
                    String protocol = parts[3];
                    if (protocol.equals("udp")) { // TODO "turn" not working
                        uris.add("stun:" + parts[2] + ":" + parts[4] + "?transport=udp");
                    } else {
                        uris.add("stun:" + parts[2] + ":" + parts[4]);
                    }
                }
            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage());
            }
        }
        return uris.toArray(new String[uris.size()]);
    }

    public void emitSessionAnswer(@NonNull Context context,
                                  @NonNull String topic,
                                  @NonNull SessionDescription message) {
        checkNotNull(context);
        checkNotNull(topic);

        checkNotNull(message);
        final IPFS ipfs = Singleton.getInstance(context).getIpfs();
        if (ipfs != null) {

            try {


                HashMap<String, String> map = new HashMap<>();
                map.put(Content.EST, Event.SESSION_ANSWER.name());
                map.put(Content.SDP, message.description);
                map.put(Content.TYPE, message.type.name());

                ipfs.pubsubPub(topic, gson.toJson(map), PUBSUB_TIMEOUT);

            } catch (Throwable e) {
                Log.e(TAG, "" + e.getLocalizedMessage(), e);
            }

        }
    }

    public void busy(@NonNull String pid) {
        if (listener != null) {
            listener.busy(pid);
        }
    }

    public void accept(@NonNull String topic, @Nullable String[] ices) {
        if (listener != null) {
            listener.accept(topic, ices);
        }
    }

    public void close(@NonNull String pid) {
        if (listener != null) {
            listener.close(pid);
        }
    }

    public void setListener(@Nullable Listener listener) {
        this.listener = listener;
    }

    public void reject(@NonNull String pid) {
        if (listener != null) {
            listener.reject(pid);
        }
    }

    public void offer(@NonNull String pid, @NonNull String sdp) {
        if (listener != null) {
            listener.offer(pid, sdp);
        }
    }

    public void answer(@NonNull String pid, @NonNull String sdp, @NonNull String type) {
        if (listener != null) {
            listener.answer(pid, sdp, type);
        }
    }

    public void candidate(@NonNull String pid, @NonNull String sdp,
                          @NonNull String mid, @NonNull String index) {
        if (listener != null) {
            listener.candidate(pid, sdp, mid, index);
        }
    }

    public void candidate_remove(@NonNull String topic, @NonNull String sdp,
                                 @NonNull String mid, @NonNull String index) {
        if (listener != null) {
            listener.candidate_remove(topic, sdp, mid, index);
        }
    }

    public void timeout(String pid) {
        if (listener != null) {
            listener.timeout(pid);
        }
    }

    public void emitSessionOffer(@NonNull Context context,
                                 @NonNull String topic,
                                 @NonNull SessionDescription message) {
        checkNotNull(context);
        checkNotNull(topic);
        checkNotNull(message);


        try {
            IPFS ipfs = Singleton.getInstance(context).getIpfs();

            checkNotNull(ipfs, "IPFS not valid");

            HashMap<String, String> map = new HashMap<>();
            map.put(Content.EST, Event.SESSION_OFFER.name());
            map.put(Content.SDP, message.description);

            ipfs.pubsubPub(topic, gson.toJson(map), PUBSUB_TIMEOUT);

        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }


    }

    public void emitSessionVideoCall(@NonNull Context context, @NonNull String topic) {
        checkNotNull(context);
        checkNotNull(topic);
        try {

            IPFS ipfs = Singleton.getInstance(context).getIpfs();
            Preconditions.checkNotNull(ipfs);
            HashMap<String, String> map = new HashMap<>();
            map.put(Content.EST, Event.SESSION_VIDEO_CALL.name());

            //PeerInfo peer = ipfs.id();
            //String addresses = gson.toJson(peer.getMultiAddresses());
            //map.put(Content.ICES, addresses);

            ipfs.pubsubPub(topic, gson.toJson(map), PUBSUB_TIMEOUT);

        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }

    }


    public void emitSessionAlive(@NonNull Context context, @NonNull String topic) {
        checkNotNull(context);
        checkNotNull(topic);
        try {

            IPFS ipfs = Singleton.getInstance(context).getIpfs();
            Preconditions.checkNotNull(ipfs);
            HashMap<String, String> map = new HashMap<>();
            map.put(Content.EST, Event.SESSION_ALIVE.name());
            ipfs.pubsubPub(topic, gson.toJson(map), PUBSUB_TIMEOUT);

        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }

    }

    public void emitSessionCall(@NonNull Context context, @NonNull String topic) {
        checkNotNull(context);
        checkNotNull(topic);

        try {

            IPFS ipfs = Singleton.getInstance(context).getIpfs();
            checkNotNull(ipfs);
            HashMap<String, String> map = new HashMap<>();
            map.put(Content.EST, Event.SESSION_CALL.name());

            //PeerInfo peer = ipfs.id();
            //String addresses = gson.toJson(peer.getMultiAddresses());
            //map.put(Content.ICES, addresses);

            ipfs.pubsubPub(topic, gson.toJson(map), PUBSUB_TIMEOUT);

        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }

    }

    public void emitSessionBusy(@NonNull Context context, @NonNull String topic) {
        checkNotNull(context);
        checkNotNull(topic);

        try {

            IPFS ipfs = Singleton.getInstance(context).getIpfs();

            checkNotNull(ipfs, "IPFS not valid");
            HashMap<String, String> map = new HashMap<>();
            map.put(Content.EST, Event.SESSION_BUSY.name());
            ipfs.pubsubPub(topic, gson.toJson(map), PUBSUB_TIMEOUT);

        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }


    }

    public void emitSessionTimeout(@NonNull Context context, @NonNull String topic) {
        checkNotNull(context);
        checkNotNull(topic);

        try {

            IPFS ipfs = Singleton.getInstance(context).getIpfs();
            checkNotNull(ipfs, "IPFS not valid");
            HashMap<String, String> map = new HashMap<>();
            map.put(Content.EST, Event.SESSION_TIMEOUT.name());
            ipfs.pubsubPub(topic, gson.toJson(map), PUBSUB_TIMEOUT);

        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }


    }

    public void emitSessionReject(@NonNull Context context, @NonNull String topic) {
        checkNotNull(context);
        checkNotNull(topic);


        try {

            IPFS ipfs = Singleton.getInstance(context).getIpfs();
            checkNotNull(ipfs, "IPFS not valid");
            HashMap<String, String> map = new HashMap<>();
            map.put(Content.EST, Event.SESSION_REJECT.name());
            ipfs.pubsubPub(topic, gson.toJson(map), PUBSUB_TIMEOUT);

        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }


    }

    public void emitSessionAccept(@NonNull Context context, @NonNull String topic) {
        checkNotNull(context);
        checkNotNull(topic);

        try {

            IPFS ipfs = Singleton.getInstance(context).getIpfs();
            checkNotNull(ipfs, "IPFS not valid");
            HashMap<String, String> map = new HashMap<>();
            map.put(Content.EST, Event.SESSION_ACCEPT.name());


            //PeerInfo peer = ipfs.id();
            //String addresses = gson.toJson(peer.getMultiAddresses());
            //map.put(Content.ICES, addresses);

            ipfs.pubsubPub(topic, gson.toJson(map), PUBSUB_TIMEOUT);

        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }


    }

    public void emitSessionClose(@NonNull Context context, @NonNull String topic) {
        checkNotNull(context);
        checkNotNull(topic);

        try {

            IPFS ipfs = Singleton.getInstance(context).getIpfs();
            checkNotNull(ipfs, "IPFS not valid");
            HashMap<String, String> map = new HashMap<>();
            map.put(Content.EST, Event.SESSION_CLOSE.name());
            ipfs.pubsubPub(topic, gson.toJson(map), PUBSUB_TIMEOUT);

        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }


    }

    public void emitIceCandidatesRemove(@NonNull Context context,
                                        @NonNull String topic,
                                        @NonNull IceCandidate[] candidates) {
        checkNotNull(context);
        checkNotNull(topic);

        checkNotNull(candidates);

        try {
            IPFS ipfs = Singleton.getInstance(context).getIpfs();
            checkNotNull(ipfs, "IPFS not valid");
            for (IceCandidate candidate : candidates) {
                HashMap<String, String> map = new HashMap<>();
                map.put(Content.EST, Event.SESSION_CANDIDATE_REMOVE.name());
                map.put(Content.SDP, candidate.sdp);
                map.put(Content.MID, candidate.sdpMid);
                map.put(Content.INDEX, String.valueOf(candidate.sdpMLineIndex));

                ipfs.pubsubPub(topic, gson.toJson(map), PUBSUB_TIMEOUT);
            }

        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }


    }

    public void emitIceCandidate(@NonNull Context context,
                                 @NonNull String topic,
                                 @NonNull IceCandidate candidate) {
        checkNotNull(context);
        checkNotNull(topic);
        checkNotNull(candidate);


        try {
            IPFS ipfs = Singleton.getInstance(context).getIpfs();

            checkNotNull(ipfs, "IPFS not valid");
            HashMap<String, String> map = new HashMap<>();
            map.put(Content.EST, Event.SESSION_CANDIDATE.name());
            map.put(Content.SDP, candidate.sdp);
            map.put(Content.MID, candidate.sdpMid);
            map.put(Content.INDEX, String.valueOf(candidate.sdpMLineIndex));

            ipfs.pubsubPub(topic, gson.toJson(map), PUBSUB_TIMEOUT);

        } catch (Throwable e) {
            Log.e(TAG, "" + e.getLocalizedMessage(), e);
        }


    }

    public boolean isBusy() {
        return busy.get();
    }

    public void setBusy(boolean flag) {
        busy.set(flag);
    }

    public enum Event {
        SESSION_CALL, SESSION_VIDEO_CALL, SESSION_ACCEPT, SESSION_BUSY,
        SESSION_REJECT, SESSION_OFFER, SESSION_TIMEOUT,
        SESSION_ANSWER, SESSION_CANDIDATE, SESSION_ALIVE,
        SESSION_CLOSE, SESSION_CANDIDATE_REMOVE
    }

    public interface Listener {
        void busy(@NonNull String topic);

        void accept(@NonNull String topic, @Nullable String[] ices);

        void reject(@NonNull String topic);

        void offer(@NonNull String topic, @NonNull String sdp);

        void answer(@NonNull String topic, @NonNull String sdp, @NonNull String type);

        void candidate(@NonNull String topic, @NonNull String sdp,
                       @NonNull String mid, @NonNull String index);

        void candidate_remove(@NonNull String topic, @NonNull String sdp,
                              @NonNull String mid, @NonNull String index);

        void close(@NonNull String topic);

        void timeout(@NonNull String topic);
    }
}
