/*
 * Decompiled with CFR 0.152.
 */
package net.luminis.tls;

import at.favre.lib.hkdf.HKDF;
import at.favre.lib.hkdf.HkdfMacFactory;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.XECPublicKey;
import javax.crypto.KeyAgreement;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import net.luminis.tls.Logger;
import net.luminis.tls.TlsConstants;
import net.luminis.tls.TranscriptHash;
import net.luminis.tls.util.ByteUtils;

public class TlsState {
    private static final Charset ISO_8859_1 = Charset.forName("ISO-8859-1");
    private static String labelPrefix = "tls13 ";
    private final MessageDigest hashFunction;
    private final HKDF hkdf;
    private final byte[] emptyHash;
    private final short keyLength;
    private final short hashLength;
    private final short iv_length = (short)12;
    private boolean pskSelected;
    private PublicKey serverSharedKey;
    private PrivateKey clientPrivateKey;
    private final byte[] psk;
    private byte[] earlySecret;
    private byte[] binderKey;
    private byte[] resumptionMasterSecret;
    private byte[] serverHandshakeTrafficSecret;
    private byte[] clientEarlyTrafficSecret;
    private byte[] clientHandshakeTrafficSecret;
    private byte[] handshakeSecret;
    private byte[] clientApplicationTrafficSecret;
    private byte[] serverApplicationTrafficSecret;
    private final TranscriptHash transcriptHash;
    private byte[] sharedSecret;
    private byte[] masterSecret;

    public TlsState(TranscriptHash transcriptHash, byte[] psk, int keyLength, int hashLength) {
        this.psk = psk;
        this.transcriptHash = transcriptHash;
        this.keyLength = (short)keyLength;
        this.hashLength = (short)hashLength;
        String hashAlgorithm = "SHA-" + this.hashLength * 8;
        try {
            this.hashFunction = MessageDigest.getInstance(hashAlgorithm);
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("Missing " + hashAlgorithm + " support");
        }
        String macAlgorithm = "HmacSHA" + this.hashLength * 8;
        this.hkdf = HKDF.from((HkdfMacFactory)new HkdfMacFactory.Default(macAlgorithm, null));
        this.emptyHash = this.hashFunction.digest(new byte[0]);
        Logger.debug("Empty hash: " + ByteUtils.bytesToHex(this.emptyHash));
        if (psk == null) {
            psk = new byte[this.hashLength];
        }
        this.computeEarlySecret(psk);
    }

    public TlsState(TranscriptHash transcriptHash, int keyLength, int hashLength) {
        this(transcriptHash, null, keyLength, hashLength);
    }

    private byte[] computeEarlySecret(byte[] ikm) {
        byte[] zeroSalt = new byte[this.hashLength];
        this.earlySecret = this.hkdf.extract(zeroSalt, ikm);
        Logger.debug("Early secret: " + ByteUtils.bytesToHex(this.earlySecret));
        this.binderKey = this.hkdfExpandLabel(this.earlySecret, "res binder", this.emptyHash, this.hashLength);
        Logger.debug("Binder key: " + ByteUtils.bytesToHex(this.binderKey));
        return this.earlySecret;
    }

    public byte[] computePskBinder(byte[] partialClientHello) {
        String macAlgorithmName = "HmacSHA" + this.hashLength * 8;
        try {
            this.hashFunction.reset();
            this.hashFunction.update(partialClientHello);
            byte[] hash = this.hashFunction.digest();
            byte[] finishedKey = this.hkdfExpandLabel(this.binderKey, "finished", "", this.hashLength);
            SecretKeySpec hmacKey = new SecretKeySpec(finishedKey, macAlgorithmName);
            Mac hmacAlgorithm = Mac.getInstance(macAlgorithmName);
            hmacAlgorithm.init(hmacKey);
            hmacAlgorithm.update(hash);
            byte[] hmac = hmacAlgorithm.doFinal();
            return hmac;
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("Missing " + macAlgorithmName + " support");
        }
        catch (InvalidKeyException e) {
            throw new RuntimeException();
        }
    }

    public void computeSharedSecret() {
        try {
            KeyAgreement keyAgreement;
            if (this.serverSharedKey instanceof ECPublicKey) {
                keyAgreement = KeyAgreement.getInstance("ECDH");
            } else if (this.serverSharedKey instanceof XECPublicKey) {
                keyAgreement = KeyAgreement.getInstance("XDH");
            } else {
                throw new RuntimeException("Unsupported key type");
            }
            keyAgreement.init(this.clientPrivateKey);
            keyAgreement.doPhase(this.serverSharedKey, true);
            this.sharedSecret = keyAgreement.generateSecret();
            Logger.debug("Shared key: " + ByteUtils.bytesToHex(this.sharedSecret));
        }
        catch (InvalidKeyException | NoSuchAlgorithmException e) {
            throw new RuntimeException("Unsupported crypto: " + e);
        }
    }

    public void computeEarlyTrafficSecret() {
        byte[] clientHelloHash = this.transcriptHash.getHash(TlsConstants.HandshakeType.client_hello);
        this.clientEarlyTrafficSecret = this.hkdfExpandLabel(this.earlySecret, "c e traffic", clientHelloHash, this.hashLength);
    }

    public void computeHandshakeSecrets() {
        byte[] derivedSecret = this.hkdfExpandLabel(this.earlySecret, "derived", this.emptyHash, this.hashLength);
        Logger.debug("Derived secret: " + ByteUtils.bytesToHex(derivedSecret));
        this.handshakeSecret = this.hkdf.extract(derivedSecret, this.sharedSecret);
        Logger.debug("Handshake secret: " + ByteUtils.bytesToHex(this.handshakeSecret));
        byte[] handshakeHash = this.transcriptHash.getHash(TlsConstants.HandshakeType.server_hello);
        this.clientHandshakeTrafficSecret = this.hkdfExpandLabel(this.handshakeSecret, "c hs traffic", handshakeHash, this.hashLength);
        Logger.debug("Client handshake traffic secret: " + ByteUtils.bytesToHex(this.clientHandshakeTrafficSecret));
        this.serverHandshakeTrafficSecret = this.hkdfExpandLabel(this.handshakeSecret, "s hs traffic", handshakeHash, this.hashLength);
        Logger.debug("Server handshake traffic secret: " + ByteUtils.bytesToHex(this.serverHandshakeTrafficSecret));
        byte[] clientHandshakeKey = this.hkdfExpandLabel(this.clientHandshakeTrafficSecret, "key", "", this.keyLength);
        Logger.debug("Client handshake key: " + ByteUtils.bytesToHex(clientHandshakeKey));
        byte[] serverHandshakeKey = this.hkdfExpandLabel(this.serverHandshakeTrafficSecret, "key", "", this.keyLength);
        Logger.debug("Server handshake key: " + ByteUtils.bytesToHex(serverHandshakeKey));
        byte[] clientHandshakeIV = this.hkdfExpandLabel(this.clientHandshakeTrafficSecret, "iv", "", (short)12);
        Logger.debug("Client handshake iv: " + ByteUtils.bytesToHex(clientHandshakeIV));
        byte[] serverHandshakeIV = this.hkdfExpandLabel(this.serverHandshakeTrafficSecret, "iv", "", (short)12);
        Logger.debug("Server handshake iv: " + ByteUtils.bytesToHex(serverHandshakeIV));
    }

    public void computeApplicationSecrets() {
        this.computeApplicationSecrets(this.handshakeSecret);
    }

    void computeApplicationSecrets(byte[] handshakeSecret) {
        byte[] serverFinishedHash = this.transcriptHash.getServerHash(TlsConstants.HandshakeType.finished);
        byte[] derivedSecret = this.hkdfExpandLabel(handshakeSecret, "derived", this.emptyHash, this.hashLength);
        Logger.debug("Derived secret: " + ByteUtils.bytesToHex(derivedSecret));
        byte[] zeroKey = new byte[this.hashLength];
        this.masterSecret = this.hkdf.extract(derivedSecret, zeroKey);
        Logger.debug("Master secret: " + ByteUtils.bytesToHex(this.masterSecret));
        this.clientApplicationTrafficSecret = this.hkdfExpandLabel(this.masterSecret, "c ap traffic", serverFinishedHash, this.hashLength);
        Logger.debug("Client application traffic secret: " + ByteUtils.bytesToHex(this.clientApplicationTrafficSecret));
        this.serverApplicationTrafficSecret = this.hkdfExpandLabel(this.masterSecret, "s ap traffic", serverFinishedHash, this.hashLength);
        Logger.debug("Server application traffic secret: " + ByteUtils.bytesToHex(this.serverApplicationTrafficSecret));
        byte[] clientApplicationKey = this.hkdfExpandLabel(this.clientApplicationTrafficSecret, "key", "", this.keyLength);
        Logger.debug("Client application key: " + ByteUtils.bytesToHex(clientApplicationKey));
        byte[] serverApplicationKey = this.hkdfExpandLabel(this.serverApplicationTrafficSecret, "key", "", this.keyLength);
        Logger.debug("Server application key: " + ByteUtils.bytesToHex(serverApplicationKey));
        byte[] clientApplicationIv = this.hkdfExpandLabel(this.clientApplicationTrafficSecret, "iv", "", (short)12);
        Logger.debug("Client application iv: " + ByteUtils.bytesToHex(clientApplicationIv));
        byte[] serverApplicationIv = this.hkdfExpandLabel(this.serverApplicationTrafficSecret, "iv", "", (short)12);
        Logger.debug("Server application iv: " + ByteUtils.bytesToHex(serverApplicationIv));
    }

    public void computeResumptionMasterSecret() {
        byte[] clientFinishedHash = this.transcriptHash.getClientHash(TlsConstants.HandshakeType.finished);
        this.resumptionMasterSecret = this.hkdfExpandLabel(this.masterSecret, "res master", clientFinishedHash, this.hashLength);
        Logger.debug("Resumption master secret: " + ByteUtils.bytesToHex(this.resumptionMasterSecret));
    }

    public byte[] computePSK(byte[] ticketNonce) {
        byte[] psk = this.hkdfExpandLabel(this.resumptionMasterSecret, "resumption", ticketNonce, this.hashLength);
        return psk;
    }

    public byte[] hkdfExpandLabel(byte[] secret, String label, String context, short length) {
        return this.hkdfExpandLabel(secret, label, context.getBytes(ISO_8859_1), length);
    }

    byte[] hkdfExpandLabel(byte[] secret, String label, byte[] context, short length) {
        ByteBuffer hkdfLabel = ByteBuffer.allocate(3 + labelPrefix.length() + label.getBytes(ISO_8859_1).length + 1 + context.length);
        hkdfLabel.putShort(length);
        hkdfLabel.put((byte)(labelPrefix.length() + label.getBytes().length));
        hkdfLabel.put(labelPrefix.getBytes(ISO_8859_1));
        hkdfLabel.put(label.getBytes(ISO_8859_1));
        hkdfLabel.put((byte)context.length);
        hkdfLabel.put(context);
        return this.hkdf.expand(secret, hkdfLabel.array(), (int)length);
    }

    public short getHashLength() {
        return this.hashLength;
    }

    public byte[] getClientEarlyTrafficSecret() {
        return this.clientEarlyTrafficSecret;
    }

    public byte[] getClientHandshakeTrafficSecret() {
        return this.clientHandshakeTrafficSecret;
    }

    public byte[] getServerHandshakeTrafficSecret() {
        return this.serverHandshakeTrafficSecret;
    }

    public byte[] getClientApplicationTrafficSecret() {
        return this.clientApplicationTrafficSecret;
    }

    public byte[] getServerApplicationTrafficSecret() {
        return this.serverApplicationTrafficSecret;
    }

    public void setOwnKey(PrivateKey clientPrivateKey) {
        this.clientPrivateKey = clientPrivateKey;
    }

    public void setPskSelected(int selectedIdentity) {
        this.pskSelected = true;
    }

    public void setNoPskSelected() {
        if (this.psk != null && !this.pskSelected) {
            this.computeEarlySecret(new byte[this.hashLength]);
        }
    }

    public void setPeerKey(PublicKey serverSharedKey) {
        this.serverSharedKey = serverSharedKey;
    }
}

