/*
 * Copyright (c) 2022, Oracle and/or its affiliates.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License, version 2.0, as published by the
 * Free Software Foundation.
 *
 * This program is also distributed with certain software (including but not
 * limited to OpenSSL) that is licensed under separate terms, as designated in a
 * particular file or component or in included license documentation. The
 * authors of MySQL hereby grant you an additional permission to link the
 * program and your derivative works with the separately licensed software that
 * they have included with MySQL.
 *
 * Without limiting anything contained in the foregoing, this file, which is
 * part of MySQL Connector/J, is also subject to the Universal FOSS Exception,
 * version 1.0, a copy of which can be found at
 * http://oss.oracle.com/licenses/universal-foss-exception.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License, version 2.0,
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA
 */

package software.aws.rds.jdbc.mysql.shading.com.mysql.cj.protocol.a;

import java.sql.Timestamp;
import java.time.OffsetDateTime;
import java.util.Calendar;
import java.util.Locale;

import software.aws.rds.jdbc.mysql.shading.com.mysql.cj.BindValue;
import software.aws.rds.jdbc.mysql.shading.com.mysql.cj.Messages;
import software.aws.rds.jdbc.mysql.shading.com.mysql.cj.MysqlType;
import software.aws.rds.jdbc.mysql.shading.com.mysql.cj.exceptions.ExceptionFactory;
import software.aws.rds.jdbc.mysql.shading.com.mysql.cj.exceptions.WrongArgumentException;
import software.aws.rds.jdbc.mysql.shading.com.mysql.cj.protocol.InternalDate;
import software.aws.rds.jdbc.mysql.shading.com.mysql.cj.protocol.InternalTime;
import software.aws.rds.jdbc.mysql.shading.com.mysql.cj.protocol.InternalTimestamp;
import software.aws.rds.jdbc.mysql.shading.com.mysql.cj.protocol.Message;
import software.aws.rds.jdbc.mysql.shading.com.mysql.cj.protocol.a.NativeConstants.IntegerDataType;
import software.aws.rds.jdbc.mysql.shading.com.mysql.cj.protocol.a.NativeConstants.StringSelfDataType;
import software.aws.rds.jdbc.mysql.shading.com.mysql.cj.util.StringUtils;
import software.aws.rds.jdbc.mysql.shading.com.mysql.cj.util.TimeUtil;

public class OffsetDateTimeValueEncoder extends AbstractValueEncoder {

    @Override
    public String getString(BindValue binding) {
        OffsetDateTime odt = (OffsetDateTime) binding.getValue();
        switch (binding.getMysqlType()) {
            case NULL:
                return "null";
            case DATE:
                StringBuilder sb = new StringBuilder("'");
                sb.append(odt.atZoneSameInstant(this.serverSession.getDefaultTimeZone().toZoneId()).toLocalDate().format(TimeUtil.DATE_FORMATTER));
                sb.append("'");
                return sb.toString();
            case TIME:
                sb = new StringBuilder("'");
                sb.append(adjustLocalTime(
                        ((OffsetDateTime) binding.getValue()).atZoneSameInstant(this.serverSession.getDefaultTimeZone().toZoneId()).toLocalTime(),
                        binding.getField()).format(TimeUtil.TIME_FORMATTER_WITH_OPTIONAL_MICROS));
                sb.append("'");
                return sb.toString();
            case DATETIME:
            case TIMESTAMP:
                Timestamp x = adjustTimestamp(
                        Timestamp.valueOf(
                                ((OffsetDateTime) binding.getValue()).atZoneSameInstant(this.serverSession.getDefaultTimeZone().toZoneId()).toLocalDateTime()),
                        binding.getField(), binding.keepOrigNanos());

                StringBuffer buf = new StringBuffer();

                buf.append(TimeUtil.getSimpleDateFormat(null, "''yyyy-MM-dd HH:mm:ss",
                        binding.getMysqlType() == MysqlType.TIMESTAMP && this.preserveInstants.getValue() ? this.serverSession.getSessionTimeZone()
                                : this.serverSession.getDefaultTimeZone())
                        .format(x));

                if (this.serverSession.getCapabilities().serverSupportsFracSecs() && x.getNanos() > 0) {
                    buf.append('.');
                    buf.append(TimeUtil.formatNanos(x.getNanos(), 6));
                }
                buf.append('\'');

                return buf.toString();

            case YEAR:
                return String.valueOf(odt.atZoneSameInstant(this.serverSession.getDefaultTimeZone().toZoneId()).getYear());
            case CHAR:
            case VARCHAR:
            case TINYTEXT:
            case TEXT:
            case MEDIUMTEXT:
            case LONGTEXT:
                sb = new StringBuilder("'");
                sb.append(odt.format(this.sendFractionalSeconds.getValue() && odt.getNano() > 0 ? TimeUtil.DATETIME_FORMATTER_WITH_NANOS_WITH_OFFSET
                        : TimeUtil.DATETIME_FORMATTER_NO_FRACT_WITH_OFFSET));
                sb.append("'");
                return sb.toString();
            default:
                throw ExceptionFactory.createException(WrongArgumentException.class,
                        Messages.getString("PreparedStatement.67", new Object[] { binding.getValue().getClass().getName(), binding.getMysqlType().toString() }),
                        this.exceptionInterceptor);
        }
    }

    @Override
    public void encodeAsBinary(Message msg, BindValue binding) {
        OffsetDateTime odt = (OffsetDateTime) binding.getValue();
        NativePacketPayload intoPacket = (NativePacketPayload) msg;
        switch (binding.getMysqlType()) {
            case DATE:
                writeDate(msg, InternalDate.from(odt.atZoneSameInstant(this.serverSession.getDefaultTimeZone().toZoneId()).toLocalDate()));
                return;
            case TIME:
                writeTime(msg,
                        InternalTime.from(adjustLocalTime(
                                ((OffsetDateTime) binding.getValue()).atZoneSameInstant(this.serverSession.getDefaultTimeZone().toZoneId()).toLocalTime(),
                                binding.getField())));
                return;
            case DATETIME:
            case TIMESTAMP:
                Timestamp ts = adjustTimestamp(
                        Timestamp.valueOf(
                                ((OffsetDateTime) binding.getValue()).atZoneSameInstant(this.serverSession.getDefaultTimeZone().toZoneId()).toLocalDateTime()),
                        binding.getField(), binding.keepOrigNanos());
                Calendar calendar = Calendar
                        .getInstance(binding.getMysqlType() == MysqlType.TIMESTAMP && this.preserveInstants.getValue() ? this.serverSession.getSessionTimeZone()
                                : this.serverSession.getDefaultTimeZone(), Locale.US);
                calendar.setTime(ts);
                writeDateTime(msg, InternalTimestamp.from(calendar, ts.getNanos()));
                return;
            case YEAR:
                intoPacket.writeInteger(IntegerDataType.INT4, odt.atZoneSameInstant(this.serverSession.getDefaultTimeZone().toZoneId()).getYear());
                return;
            case CHAR:
            case VARCHAR:
            case TINYTEXT:
            case TEXT:
            case MEDIUMTEXT:
            case LONGTEXT:
                intoPacket.writeBytes(StringSelfDataType.STRING_LENENC,
                        StringUtils.getBytes(
                                odt.format(this.sendFractionalSeconds.getValue() && odt.getNano() > 0 ? TimeUtil.DATETIME_FORMATTER_WITH_NANOS_WITH_OFFSET
                                        : TimeUtil.DATETIME_FORMATTER_NO_FRACT_WITH_OFFSET),
                                this.charEncoding.getValue()));
                return;
            default:
                throw ExceptionFactory.createException(WrongArgumentException.class,
                        Messages.getString("PreparedStatement.67", new Object[] { binding.getValue().getClass().getName(), binding.getMysqlType().toString() }),
                        this.exceptionInterceptor);
        }
    }

    @Override
    public void encodeAsQueryAttribute(Message msg, BindValue binding) {
        writeDateTimeWithOffset(msg, InternalTimestamp.from((OffsetDateTime) binding.getValue()));
    }
}
