/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.loader;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringJoiner;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.logging.Logger;
import software.amazon.smithy.model.Model;
import software.amazon.smithy.model.SourceException;
import software.amazon.smithy.model.loader.ApplyMixin;
import software.amazon.smithy.model.loader.LoadOperation;
import software.amazon.smithy.model.loader.LoaderUtils;
import software.amazon.smithy.model.loader.ModelInteropTransformer;
import software.amazon.smithy.model.loader.Prelude;
import software.amazon.smithy.model.loader.ShapeModifier;
import software.amazon.smithy.model.loader.TopologicalShapeSort;
import software.amazon.smithy.model.loader.Version;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.shapes.AbstractShapeBuilder;
import software.amazon.smithy.model.shapes.MemberShape;
import software.amazon.smithy.model.shapes.Shape;
import software.amazon.smithy.model.shapes.ShapeId;
import software.amazon.smithy.model.shapes.ShapeType;
import software.amazon.smithy.model.traits.BoxTrait;
import software.amazon.smithy.model.traits.Trait;
import software.amazon.smithy.model.validation.Severity;
import software.amazon.smithy.model.validation.ValidationEvent;

final class LoaderShapeMap {
    private static final Logger LOGGER = Logger.getLogger(LoaderShapeMap.class.getName());
    private final Map<ShapeId, ShapeWrapper> shapes = new HashMap<ShapeId, ShapeWrapper>();
    private final Map<ShapeId, Shape> createdShapes = new HashMap<ShapeId, Shape>();
    private final Model preludeShapes;
    private final List<ValidationEvent> events;

    LoaderShapeMap(Model prelude, List<ValidationEvent> events) {
        this.preludeShapes = prelude;
        this.events = events;
    }

    boolean isShapePending(ShapeId id) {
        if (this.containsShapeId(id)) {
            return true;
        }
        String member = id.getMember().orElse(null);
        if (member == null) {
            return false;
        }
        ShapeId root = id.withoutMember();
        return this.containsShapeId(root) && this.shapes.get(root).hasMember(member);
    }

    boolean isRootShapeDefined(ShapeId id) {
        return this.containsPreludeShape(id) || this.containsShapeId(id) || this.createdShapes.containsKey(id);
    }

    private boolean containsPreludeShape(ShapeId id) {
        return this.preludeShapes != null && this.preludeShapes.getShapeIds().contains(id);
    }

    private boolean containsShapeId(ShapeId id) {
        return this.shapes.containsKey(id);
    }

    ShapeType getShapeType(ShapeId id) {
        if (id.hasMember()) {
            return ShapeType.MEMBER;
        }
        if (this.shapes.containsKey(id)) {
            return this.shapes.get(id).getFirst().getShapeType();
        }
        if (this.createdShapes.containsKey(id)) {
            return this.createdShapes.get(id).getType();
        }
        if (this.containsPreludeShape(id)) {
            return this.preludeShapes.expectShape(id).getType();
        }
        return null;
    }

    Version getShapeVersion(ShapeId shape) {
        ShapeId noMember = shape.withoutMember();
        if (this.shapes.containsKey(noMember)) {
            return this.shapes.get((Object)noMember).getFirst().version;
        }
        return Version.UNKNOWN;
    }

    ShapeWrapper get(ShapeId id) {
        ShapeWrapper result = this.shapes.get(id);
        if (result == null) {
            throw new IllegalArgumentException("Shape not found when loading the model: " + id);
        }
        return result;
    }

    void add(LoadOperation.DefineShape operation) {
        this.shapes.computeIfAbsent(operation.toShapeId(), id -> new ShapeWrapper()).add(operation);
    }

    void add(Shape shape, Consumer<LoadOperation> processor) {
        if (!shape.isMemberShape() && !Prelude.isPreludeShape(shape)) {
            this.createdShapes.put(shape.getId(), shape);
            if (!shape.getMixins().isEmpty()) {
                this.moveCreatedShapeToOperations(shape.getId(), processor);
            }
        }
    }

    void moveCreatedShapeToOperations(ShapeId shapeId, Consumer<LoadOperation> processor) {
        if (this.createdShapes.containsKey(shapeId)) {
            Shape shape = this.createdShapes.remove(shapeId);
            Object builder = Shape.shapeToBuilder(shape);
            LoadOperation.DefineShape operation = new LoadOperation.DefineShape(Version.UNKNOWN, (AbstractShapeBuilder<?, ?>)builder);
            for (ShapeId mixin : shape.getMixins()) {
                operation.addDependency(mixin);
                operation.addModifier(new ApplyMixin(mixin));
            }
            ((AbstractShapeBuilder)builder).clearMixins();
            for (Trait trait : shape.getIntroducedTraits().values()) {
                boolean notSynthetic;
                boolean bl = notSynthetic = !trait.isSynthetic();
                if (!notSynthetic && !trait.toShapeId().equals(BoxTrait.ID)) continue;
                processor.accept(LoadOperation.ApplyTrait.from(shape.getId(), trait));
            }
            ((AbstractShapeBuilder)builder).clearTraits();
            for (MemberShape member : shape.members()) {
                MemberShape.Builder memberBuilder = member.toBuilder();
                for (Trait trait : member.getIntroducedTraits().values()) {
                    if (trait.isSynthetic()) continue;
                    processor.accept(LoadOperation.ApplyTrait.from(member.getId(), trait));
                }
                ((MemberShape.Builder)memberBuilder.clearTraits()).clearMixins();
                operation.addMember(memberBuilder);
            }
            this.add(operation);
        } else if (shapeId.hasMember()) {
            this.moveCreatedShapeToOperations(shapeId.withoutMember(), processor);
        }
    }

    void buildShapesAndClaimMixinTraits(Model.Builder modelBuilder, Function<ShapeId, Map<ShapeId, Trait>> unclaimedTraits) {
        Function<ShapeId, Shape> createdShapeMap = id -> modelBuilder.getCurrentShapes().get(id);
        for (Shape shape : this.createdShapes.values()) {
            modelBuilder.addShapes(shape);
        }
        for (ShapeId id2 : this.sort()) {
            if (this.createdShapes.containsKey(id2)) continue;
            this.buildIntoModel(this.shapes.get(id2), modelBuilder, unclaimedTraits, createdShapeMap);
        }
    }

    private void buildIntoModel(ShapeWrapper wrapper, Model.Builder builder, Function<ShapeId, Map<ShapeId, Trait>> unclaimedTraits, Function<ShapeId, Shape> createdShapeMap) {
        Shape built = null;
        for (LoadOperation.DefineShape shape : wrapper) {
            Shape newShape;
            if (!this.validateShapeVersion(shape) || (newShape = this.buildShape(shape, unclaimedTraits, createdShapeMap)) == null || !this.validateConflicts(shape.toShapeId(), newShape, built)) continue;
            built = newShape;
        }
        if (built != null) {
            builder.addShape(built);
        }
    }

    private List<ShapeId> sort() {
        TopologicalShapeSort sorter = new TopologicalShapeSort(this.createdShapes.size() + this.shapes.size());
        for (Shape shape : this.createdShapes.values()) {
            sorter.enqueue(shape.getId(), Collections.emptyList());
        }
        for (Map.Entry entry : this.shapes.entrySet()) {
            sorter.enqueue((ShapeId)entry.getKey(), ((ShapeWrapper)entry.getValue()).dependencies());
        }
        try {
            return sorter.dequeueSortedShapes();
        }
        catch (TopologicalShapeSort.CycleException e) {
            for (ShapeId unresolved : e.getUnresolved()) {
                for (LoadOperation.DefineShape shape : this.get(unresolved)) {
                    this.emitUnresolved(shape, e.getUnresolved(), e.getResolved());
                }
            }
            return Collections.emptyList();
        }
    }

    private void emitUnresolved(LoadOperation.DefineShape shape, Set<ShapeId> unresolved, List<ShapeId> resolved) {
        ArrayList<ShapeId> notFoundShapes = new ArrayList<ShapeId>();
        ArrayList<ShapeId> missingTransitive = new ArrayList<ShapeId>();
        ArrayList<ShapeId> cycles = new ArrayList<ShapeId>();
        for (ShapeId id : shape.dependencies()) {
            if (!unresolved.contains(id)) {
                notFoundShapes.add(id);
                continue;
            }
            if (this.anyMissingTransitiveDependencies(id, resolved, unresolved, new HashSet<ShapeId>())) {
                missingTransitive.add(id);
                continue;
            }
            cycles.add(id);
        }
        StringJoiner message = new StringJoiner(" ");
        message.add("Unable to resolve mixins;");
        if (!notFoundShapes.isEmpty()) {
            message.add("attempted to mixin shapes that are not in the model: " + notFoundShapes);
        }
        if (!missingTransitive.isEmpty()) {
            message.add("unable to resolve due to missing transitive mixins: " + missingTransitive);
        }
        if (!cycles.isEmpty()) {
            message.add("cycles detected between this shape and " + cycles);
        }
        this.events.add(ValidationEvent.builder().id("Model").severity(Severity.ERROR).shapeId(shape.toShapeId()).sourceLocation(shape).message(message.toString()).build());
    }

    private boolean anyMissingTransitiveDependencies(ShapeId current, List<ShapeId> resolved, Set<ShapeId> unresolved, Set<ShapeId> visited) {
        if (resolved.contains(current)) {
            return false;
        }
        if (!unresolved.contains(current)) {
            return true;
        }
        if (visited.contains(current)) {
            visited.remove(current);
            return false;
        }
        visited.add(current);
        for (ShapeId next : this.get(current).dependencies()) {
            if (!this.anyMissingTransitiveDependencies(next, resolved, unresolved, visited)) continue;
            return true;
        }
        return false;
    }

    private boolean validateShapeVersion(LoadOperation.DefineShape operation) {
        if (!operation.version.isShapeTypeSupported(operation.getShapeType())) {
            this.events.add(ValidationEvent.builder().severity(Severity.ERROR).id("Model").shapeId(operation.toShapeId()).sourceLocation(operation).message(String.format("%s shapes cannot be used in Smithy version " + (Object)((Object)operation.version), new Object[]{operation.getShapeType()})).build());
            return false;
        }
        return true;
    }

    private boolean validateConflicts(ShapeId id, Shape built, Shape previous) {
        if (previous != null && built != null) {
            if (!previous.equals(built)) {
                StringJoiner joiner = new StringJoiner("; ");
                if (built.getType() != previous.getType()) {
                    joiner.add("Left is " + (Object)((Object)built.getType()) + ", right is " + (Object)((Object)previous.getType()));
                }
                if (!built.getMixins().equals(previous.getMixins())) {
                    joiner.add("Left mixins: " + built.getMixins() + ", right mixins: " + previous.getMixins());
                }
                if (!built.getAllTraits().equals(previous.getAllTraits())) {
                    built.getAllTraits().forEach((tid, t) -> {
                        if (!previous.hasTrait((ShapeId)tid)) {
                            joiner.add("Left has trait " + tid);
                        } else if (!previous.getAllTraits().get(tid).equals(t)) {
                            joiner.add("Left trait " + tid + " differs from right trait. " + Node.printJson(t.toNode()) + " vs " + Node.printJson(previous.getAllTraits().get(tid).toNode()));
                        }
                    });
                    previous.getAllTraits().forEach((tid, t) -> {
                        if (!built.hasTrait((ShapeId)tid)) {
                            joiner.add("Right has trait " + tid);
                        }
                    });
                }
                if (!built.getAllMembers().equals(previous.getAllMembers())) {
                    joiner.add("Members differ: " + built.getAllMembers().keySet() + " vs " + previous.getAllMembers().keySet());
                }
                this.events.add(LoaderUtils.onShapeConflict(id, built.getSourceLocation(), previous.getSourceLocation(), joiner.toString()));
                return false;
            }
            if (!LoaderUtils.isSameLocation(built, previous)) {
                LOGGER.warning(() -> "Ignoring duplicate but equivalent shape definition: " + id + " defined at " + built.getSourceLocation() + " and " + previous.getSourceLocation());
            }
        }
        return true;
    }

    private Shape buildShape(LoadOperation.DefineShape defineShape, Function<ShapeId, Map<ShapeId, Trait>> traitClaimer, Function<ShapeId, Shape> createdShapeMap) {
        try {
            AbstractShapeBuilder<?, ?> builder = defineShape.builder();
            ModelInteropTransformer.patchShapeBeforeBuilding(defineShape, builder, this.events);
            for (MemberShape.Builder memberBuilder : defineShape.memberBuilders().values()) {
                for (ShapeModifier modifier : defineShape.modifiers()) {
                    modifier.modifyMember(builder, memberBuilder, traitClaimer, createdShapeMap);
                }
                MemberShape member = this.buildMember(memberBuilder);
                if (member == null) continue;
                try {
                    builder.addMember(member);
                }
                catch (SourceException e) {
                    this.events.add(ValidationEvent.fromSourceException(e, "", builder.getId()));
                }
            }
            for (ShapeModifier modifier : defineShape.modifiers()) {
                modifier.modifyShape(builder, defineShape.memberBuilders(), traitClaimer, createdShapeMap);
                this.events.addAll(modifier.getEvents());
            }
            return (Shape)builder.build();
        }
        catch (SourceException e) {
            this.events.add(ValidationEvent.fromSourceException(e, "", defineShape.toShapeId()));
            return null;
        }
    }

    private MemberShape buildMember(MemberShape.Builder builder) {
        try {
            return builder.build();
        }
        catch (IllegalStateException e) {
            if (builder.getTarget() == null) {
                this.events.add(ValidationEvent.builder().severity(Severity.ERROR).id("Model").shapeId(builder.getId()).sourceLocation(builder).message("Member target was elided, but no bound resource or mixin contained a matching identifier or member name.").build());
                return null;
            }
            throw e;
        }
        catch (SourceException e) {
            this.events.add(ValidationEvent.fromSourceException(e, "", builder.getId()));
            return null;
        }
    }

    static final class ShapeWrapper
    implements Iterable<LoadOperation.DefineShape> {
        private final List<LoadOperation.DefineShape> shapes = new ArrayList<LoadOperation.DefineShape>(1);

        ShapeWrapper() {
        }

        @Override
        public Iterator<LoadOperation.DefineShape> iterator() {
            return this.shapes.iterator();
        }

        LoadOperation.DefineShape getFirst() {
            return this.shapes.get(0);
        }

        void add(LoadOperation.DefineShape shape) {
            this.shapes.add(shape);
        }

        boolean hasMember(String memberName) {
            for (LoadOperation.DefineShape shape : this) {
                if (!shape.hasMember(memberName)) continue;
                return true;
            }
            return false;
        }

        Set<ShapeId> dependencies() {
            if (this.shapes.size() == 1) {
                return this.getFirst().dependencies();
            }
            if (!this.hasDependencies()) {
                return Collections.emptySet();
            }
            HashSet<ShapeId> dependencies = new HashSet<ShapeId>();
            for (LoadOperation.DefineShape shape : this.shapes) {
                dependencies.addAll(shape.dependencies());
            }
            return dependencies;
        }

        private boolean hasDependencies() {
            for (LoadOperation.DefineShape shape : this.shapes) {
                if (shape.dependencies().isEmpty()) continue;
                return true;
            }
            return false;
        }
    }
}

