/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.loader;

import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import software.amazon.smithy.model.SourceException;
import software.amazon.smithy.model.SourceLocation;
import software.amazon.smithy.model.loader.LoadOperation;
import software.amazon.smithy.model.loader.LoaderShapeMap;
import software.amazon.smithy.model.loader.LoaderUtils;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.shapes.AbstractShapeBuilder;
import software.amazon.smithy.model.shapes.ShapeId;
import software.amazon.smithy.model.traits.DynamicTrait;
import software.amazon.smithy.model.traits.Trait;
import software.amazon.smithy.model.traits.TraitFactory;
import software.amazon.smithy.model.validation.Severity;
import software.amazon.smithy.model.validation.ValidationEvent;

final class LoaderTraitMap {
    private static final Logger LOGGER = Logger.getLogger(LoaderTraitMap.class.getName());
    private static final String UNRESOLVED_TRAIT_SUFFIX = ".UnresolvedTrait";
    private final TraitFactory traitFactory;
    private final Map<ShapeId, Map<ShapeId, Node>> traits = new HashMap<ShapeId, Map<ShapeId, Node>>();
    private final List<ValidationEvent> events;
    private final boolean allowUnknownTraits;
    private final Map<ShapeId, Map<ShapeId, Trait>> unclaimed = new HashMap<ShapeId, Map<ShapeId, Trait>>();

    LoaderTraitMap(TraitFactory traitFactory, List<ValidationEvent> events, boolean allowUnknownTraits) {
        this.traitFactory = traitFactory;
        this.events = events;
        this.allowUnknownTraits = allowUnknownTraits;
    }

    void applyTraitsToNonMixinsInShapeMap(LoaderShapeMap shapeMap) {
        for (Map.Entry<ShapeId, Map<ShapeId, Node>> entry : this.traits.entrySet()) {
            ShapeId target = entry.getKey();
            ShapeId root = target.withoutMember();
            boolean found = shapeMap.isShapePending(target);
            LoaderShapeMap.ShapeWrapper rootShapes = found ? shapeMap.get(root) : Collections::emptyIterator;
            for (Map.Entry<ShapeId, Node> traitEntry : entry.getValue().entrySet()) {
                ShapeId traitId = traitEntry.getKey();
                Node traitNode = traitEntry.getValue();
                Trait created = this.createTrait(target, traitId, traitNode);
                this.validateTraitIsKnown(target, traitId, created, traitNode.getSourceLocation(), shapeMap);
                if (target.hasMember()) {
                    String memberName = target.getMember().get();
                    boolean foundMember = false;
                    for (LoadOperation.DefineShape shape : rootShapes) {
                        if (!shape.hasMember(memberName)) continue;
                        foundMember = true;
                        shape.memberBuilders().get(memberName).getAllTraits();
                        this.applyTraitsToShape(shape.memberBuilders().get(memberName), created);
                    }
                    if (foundMember) continue;
                    this.unclaimed.computeIfAbsent(target.withMember(memberName), id -> new LinkedHashMap()).put(traitId, created);
                    continue;
                }
                if (found) {
                    for (LoadOperation.DefineShape shape : rootShapes) {
                        this.applyTraitsToShape(shape.builder(), created);
                    }
                    continue;
                }
                this.unclaimed.computeIfAbsent(target, id -> new LinkedHashMap()).put(traitId, created);
            }
        }
    }

    private Trait createTrait(ShapeId target, ShapeId traitId, Node traitValue) {
        try {
            return this.traitFactory.createTrait(traitId, target, traitValue).orElseGet(() -> new DynamicTrait(traitId, traitValue));
        }
        catch (SourceException e) {
            String message = String.format("Error creating trait `%s`: ", Trait.getIdiomaticTraitName(traitId));
            this.events.add(ValidationEvent.fromSourceException(e, message, target));
            return null;
        }
        catch (RuntimeException e) {
            this.events.add(ValidationEvent.builder().id("Model").severity(Severity.ERROR).shapeId(target).sourceLocation(traitValue).message(String.format("Error creating trait `%s`: %s", Trait.getIdiomaticTraitName(traitId), e.getMessage())).build());
            return null;
        }
    }

    private void validateTraitIsKnown(ShapeId target, ShapeId traitId, Trait trait, SourceLocation sourceLocation, LoaderShapeMap shapeMap) {
        if (!(shapeMap.isRootShapeDefined(traitId) || trait != null && trait.isSynthetic())) {
            Severity severity = this.allowUnknownTraits ? Severity.WARNING : Severity.ERROR;
            this.events.add(ValidationEvent.builder().id("Model.UnresolvedTrait").severity(severity).sourceLocation(sourceLocation).shapeId(target).message(String.format("Unable to resolve trait `%s`. If this is a custom trait, then it must be defined before it can be used in a model.", traitId)).build());
        }
    }

    private void applyTraitsToShape(AbstractShapeBuilder<?, ?> shape, Trait trait) {
        if (trait != null) {
            shape.addTrait(trait);
        }
    }

    Map<ShapeId, Trait> claimTraitsForShape(ShapeId id) {
        return this.unclaimed.containsKey(id) ? this.unclaimed.remove(id) : Collections.emptyMap();
    }

    void emitUnclaimedTraits() {
        for (Map.Entry<ShapeId, Map<ShapeId, Trait>> entry : this.unclaimed.entrySet()) {
            for (Map.Entry<ShapeId, Trait> traitEntry : entry.getValue().entrySet()) {
                this.events.add(ValidationEvent.builder().id("Model").severity(Severity.ERROR).sourceLocation(traitEntry.getValue()).message(String.format("Trait `%s` applied to unknown shape `%s`", Trait.getIdiomaticTraitName(traitEntry.getKey()), entry.getKey())).build());
            }
        }
    }

    void add(LoadOperation.ApplyTrait operation) {
        if (this.validateTraitVersion(operation)) {
            if (this.isAppliedToPreludeOutsidePrelude(operation)) {
                String message = String.format("Cannot apply `%s` to an immutable prelude shape defined in `smithy.api`.", operation.trait);
                this.events.add(ValidationEvent.builder().severity(Severity.ERROR).id("Model").sourceLocation(operation).shapeId(operation.target).message(message).build());
            } else {
                Map current = this.traits.computeIfAbsent(operation.target, id -> new LinkedHashMap());
                Node previous = (Node)current.get(operation.trait);
                current.put(operation.trait, this.mergeTraits(operation.target, operation.trait, previous, operation.value));
            }
        }
    }

    private boolean validateTraitVersion(LoadOperation.ApplyTrait operation) {
        ValidationEvent event = operation.version.validateVersionedTrait(operation.target, operation.trait, operation.value);
        if (event != null) {
            this.events.add(event);
        }
        return true;
    }

    private boolean isAppliedToPreludeOutsidePrelude(LoadOperation.ApplyTrait operation) {
        return !operation.namespace.equals("smithy.api") && operation.target.getNamespace().equals("smithy.api");
    }

    private Node mergeTraits(ShapeId target, ShapeId traitId, Node previous, Node updated) {
        if (previous == null) {
            return updated;
        }
        if (LoaderUtils.isSameLocation(previous, updated) && previous.equals(updated)) {
            LOGGER.finest(() -> String.format("Ignoring duplicate %s trait value on %s at same exact location", traitId, target));
            return previous;
        }
        if (previous.isArrayNode() && updated.isArrayNode()) {
            return previous.expectArrayNode().merge(updated.expectArrayNode());
        }
        if (previous.equals(updated)) {
            LOGGER.fine(() -> String.format("Ignoring duplicate %s trait value on %s", traitId, target));
            return previous;
        }
        this.events.add(ValidationEvent.builder().id("Model").severity(Severity.ERROR).sourceLocation(updated).shapeId(target).message(String.format("Conflicting `%s` trait found on shape `%s`. The previous trait was defined at `%s`, and a conflicting trait was defined at `%s`.", traitId, target, previous.getSourceLocation(), updated.getSourceLocation())).build());
        return previous;
    }
}

