/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.loader.sourcecontext;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.UncheckedIOException;
import java.net.URL;
import java.net.URLConnection;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.logging.Logger;
import software.amazon.smithy.model.FromSourceLocation;
import software.amazon.smithy.model.Model;
import software.amazon.smithy.model.SourceLocation;
import software.amazon.smithy.model.loader.sourcecontext.SourceContextLoader;
import software.amazon.smithy.model.shapes.MemberShape;
import software.amazon.smithy.model.shapes.Shape;
import software.amazon.smithy.model.validation.ValidationEvent;

final class DefaultSourceLoader
implements SourceContextLoader {
    private static final Logger LOGGER = Logger.getLogger(DefaultSourceLoader.class.getName());
    private final List<SourceContextLoader.Line> lines = new ArrayList<SourceContextLoader.Line>();
    private final Model model;
    private final int defaultCodeLines;
    private SourceLocation lastLoadedLocation;
    private Collection<SourceContextLoader.Line> lastLoadedLocationLines;

    DefaultSourceLoader(int defaultCodeLines, Model model) {
        if (defaultCodeLines < 1) {
            throw new IllegalArgumentException("Must allow at least one code hint line: " + defaultCodeLines);
        }
        this.defaultCodeLines = defaultCodeLines;
        this.model = model;
    }

    @Override
    public Collection<SourceContextLoader.Line> loadContext(FromSourceLocation source) {
        int line;
        SourceLocation location = source.getSourceLocation();
        if (location == SourceLocation.NONE) {
            return Collections.emptyList();
        }
        if (location.equals(this.lastLoadedLocation)) {
            return this.lastLoadedLocationLines;
        }
        if (this.lastLoadedLocation == null || !this.lastLoadedLocation.getFilename().equals(location.getFilename())) {
            this.loadNextFile(location);
        }
        if (!this.isValidLine(line = location.getLine())) {
            LOGGER.finer(() -> "Attempted to load context for an invalid source location: " + location);
            this.lastLoadedLocationLines = Collections.emptyList();
        } else if (this.model == null) {
            int start = Math.max(0, line - this.defaultCodeLines);
            this.lastLoadedLocationLines = this.lines.subList(start, line);
        } else {
            this.lastLoadedLocationLines = this.parseLines(source);
        }
        return this.lastLoadedLocationLines;
    }

    private void loadNextFile(SourceLocation source) {
        this.lines.clear();
        this.lastLoadedLocation = source;
        LOGGER.finer(() -> "Opening source location file for " + source);
        try (LineNumberReader reader = this.openSourceLocation(source);){
            String lineString;
            while ((lineString = reader.readLine()) != null) {
                this.lines.add(new SourceContextLoader.Line(reader.getLineNumber(), lineString));
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private LineNumberReader openSourceLocation(FromSourceLocation source) {
        try {
            SourceLocation location = source.getSourceLocation();
            String normalizedFile = location.getFilename();
            if (!location.getFilename().startsWith("file:") && !location.getFilename().startsWith("jar:")) {
                normalizedFile = "file:" + normalizedFile;
            }
            URL url = new URL(normalizedFile);
            URLConnection connection = url.openConnection();
            connection.setUseCaches(false);
            return new LineNumberReader(new InputStreamReader(connection.getInputStream(), StandardCharsets.UTF_8));
        }
        catch (IOException e) {
            throw new UncheckedIOException("Unable to load source location context for " + source, e);
        }
    }

    private boolean isValidLine(int line) {
        return --line >= 0 && line < this.lines.size();
    }

    private void addLineIfValid(int line, List<SourceContextLoader.Line> mutate) {
        if (this.isValidLine(line)) {
            mutate.add(this.lines.get(line - 1));
        }
    }

    private List<SourceContextLoader.Line> parseLines(FromSourceLocation source) {
        SourceLocation location = source.getSourceLocation();
        int line = location.getLine();
        if (source instanceof ValidationEvent) {
            ValidationEvent event = (ValidationEvent)source;
            Shape targetShape = event.getShapeId().flatMap(this.model::getShape).orElse(null);
            if (targetShape != null) {
                if (targetShape.getSourceLocation().equals(location)) {
                    source = targetShape;
                } else {
                    if (targetShape.getSourceLocation().getLine() > location.getLine()) {
                        return this.parseTraitBeforeShape(location, targetShape);
                    }
                    if (location.getFilename().endsWith(".json")) {
                        return this.parseTraitAfterShape(location, targetShape);
                    }
                    return Collections.singletonList(this.lines.get(line - 1));
                }
            }
        }
        if (source instanceof MemberShape) {
            return this.parseMemberShape(location, (MemberShape)source);
        }
        return this.parseOtherComponents(location);
    }

    private List<SourceContextLoader.Line> parseTraitBeforeShape(SourceLocation location, Shape targetShape) {
        ArrayList<SourceContextLoader.Line> result = new ArrayList<SourceContextLoader.Line>(2);
        this.addLineIfValid(location.getLine(), result);
        this.addLineIfValid(targetShape.getSourceLocation().getLine(), result);
        return result;
    }

    private List<SourceContextLoader.Line> parseTraitAfterShape(SourceLocation location, Shape targetShape) {
        ArrayList<SourceContextLoader.Line> result = new ArrayList<SourceContextLoader.Line>(2);
        this.addLineIfValid(targetShape.getSourceLocation().getLine(), result);
        this.addLineIfValid(location.getLine(), result);
        return result;
    }

    private List<SourceContextLoader.Line> parseMemberShape(SourceLocation location, MemberShape member) {
        Shape container = this.model.getShape(member.getContainer()).orElse(null);
        if (container == null) {
            LOGGER.warning(() -> "Member container not found: " + member.getId() + " -> " + member.getTarget());
        } else {
            SourceLocation containerLocation = container.getSourceLocation();
            if (containerLocation.getFilename().equals(location.getFilename()) && containerLocation.getLine() < location.getLine()) {
                ArrayList<SourceContextLoader.Line> result = new ArrayList<SourceContextLoader.Line>(2);
                this.addLineIfValid(containerLocation.getLine(), result);
                this.addLineIfValid(location.getLine(), result);
                return result;
            }
        }
        return Collections.emptyList();
    }

    private List<SourceContextLoader.Line> parseOtherComponents(SourceLocation location) {
        SourceContextLoader.Line foundLine;
        int line = location.getLine();
        int lowestPossibleLine = Math.max(0, line - 1 - this.defaultCodeLines);
        int foundStart = line - 1;
        int i = line - 1;
        while (i > lowestPossibleLine && (foundLine = this.lines.get(i)).getContent().length() > 0) {
            foundStart = i--;
        }
        return this.lines.subList(foundStart, line);
    }
}

