/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.selector;

import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.logging.Logger;
import software.amazon.smithy.model.Model;
import software.amazon.smithy.model.SourceException;
import software.amazon.smithy.model.knowledge.NeighborProviderIndex;
import software.amazon.smithy.model.neighbor.NeighborProvider;
import software.amazon.smithy.model.neighbor.Relationship;
import software.amazon.smithy.model.neighbor.RelationshipDirection;
import software.amazon.smithy.model.neighbor.RelationshipType;
import software.amazon.smithy.model.selector.Selector;
import software.amazon.smithy.model.shapes.MemberShape;
import software.amazon.smithy.model.shapes.OperationShape;
import software.amazon.smithy.model.shapes.Shape;
import software.amazon.smithy.model.shapes.ShapeId;
import software.amazon.smithy.model.shapes.StructureShape;
import software.amazon.smithy.model.shapes.ToShapeId;
import software.amazon.smithy.utils.FunctionalUtils;
import software.amazon.smithy.utils.ListUtils;

public final class PathFinder {
    private static final Logger LOGGER = Logger.getLogger(PathFinder.class.getName());
    private final Model model;
    private final NeighborProvider reverseProvider;
    private Predicate<Relationship> filter = FunctionalUtils.alwaysTrue();

    private PathFinder(Model model) {
        this.model = model;
        this.reverseProvider = NeighborProviderIndex.of(model).getReverseProvider();
    }

    public static PathFinder create(Model model) {
        return new PathFinder(model);
    }

    public void relationshipFilter(Predicate<Relationship> predicate) {
        this.filter = predicate;
    }

    public List<Path> search(ToShapeId startingShape, String targetSelector) {
        return this.search(startingShape, Selector.parse(targetSelector));
    }

    public List<Path> search(ToShapeId startingShape, Selector targetSelector) {
        Set<Shape> candidates = targetSelector.select(this.model);
        if (candidates.isEmpty()) {
            LOGGER.info(() -> "No shapes matched the PathFinder selector of `" + targetSelector + "`");
            return ListUtils.of();
        }
        LOGGER.finest(() -> candidates.size() + " shapes matched the PathFinder selector of " + targetSelector);
        return this.searchFromShapeToSet(startingShape, candidates);
    }

    private List<Path> searchFromShapeToSet(ToShapeId startingShape, Collection<Shape> candidates) {
        Shape shape = this.model.getShape(startingShape.toShapeId()).orElse(null);
        if (shape == null || candidates.isEmpty()) {
            return ListUtils.of();
        }
        return new Search(this.reverseProvider, shape, candidates, this.filter).execute();
    }

    public List<Path> search(ToShapeId startingShape, Collection<Shape> targetShapes) {
        return this.searchFromShapeToSet(startingShape, targetShapes);
    }

    public Optional<Path> createPathToInputMember(ToShapeId operationId, String memberName) {
        return this.createPathTo(operationId, memberName, RelationshipType.INPUT);
    }

    public Optional<Path> createPathToOutputMember(ToShapeId operationId, String memberName) {
        return this.createPathTo(operationId, memberName, RelationshipType.OUTPUT);
    }

    private Optional<Path> createPathTo(ToShapeId operationId, String memberName, RelationshipType rel) {
        OperationShape operation = this.model.getShape(operationId.toShapeId()).flatMap(Shape::asOperationShape).orElse(null);
        if (operation == null) {
            return Optional.empty();
        }
        ShapeId structId = rel == RelationshipType.INPUT ? operation.getInputShape() : operation.getOutputShape();
        StructureShape struct = this.model.getShape(structId).flatMap(Shape::asStructureShape).orElse(null);
        if (struct == null) {
            return Optional.empty();
        }
        MemberShape member = struct.getMember(memberName).orElse(null);
        if (member == null) {
            return Optional.empty();
        }
        Shape target = this.model.getShape(member.getTarget()).orElse(null);
        if (target == null) {
            return Optional.empty();
        }
        Path path = new Path(Relationship.create(member, RelationshipType.MEMBER_TARGET, target), null);
        path = new Path(Relationship.create(struct, RelationshipType.STRUCTURE_MEMBER, member), path);
        path = new Path(Relationship.create(operation, rel, struct), path);
        return Optional.of(path);
    }

    private static final class Search {
        private final Shape startingShape;
        private final NeighborProvider provider;
        private final Collection<Shape> candidates;
        private final List<Path> results = new ArrayList<Path>();
        private final Predicate<Relationship> filter;

        Search(NeighborProvider provider, Shape startingShape, Collection<Shape> candidates, Predicate<Relationship> filter) {
            this.startingShape = startingShape;
            this.candidates = candidates;
            this.provider = provider;
            this.filter = filter;
        }

        List<Path> execute() {
            HashSet<ShapeId> visited = new HashSet<ShapeId>();
            for (Shape candidate : this.candidates) {
                this.traverseUp(candidate, null, visited);
            }
            return this.results;
        }

        private void traverseUp(Shape current, Path path, Set<ShapeId> visited) {
            if (path != null && current.getId().equals(this.startingShape.getId())) {
                this.results.add(path);
                return;
            }
            if (visited.add(current.getId())) {
                for (Relationship relationship : this.provider.getNeighbors(current)) {
                    if (relationship.getDirection() != RelationshipDirection.DIRECTED || !this.filter.test(relationship)) continue;
                    this.traverseUp(relationship.getShape(), new Path(relationship, path), visited);
                }
                visited.remove(current.getId());
            }
        }
    }

    public static final class Path
    extends AbstractList<Relationship> {
        private final Relationship value;
        private Path next;
        private final int size;

        public Path(List<Relationship> relationships) {
            if (relationships.isEmpty()) {
                throw new IllegalArgumentException("Relationships cannot be empty!");
            }
            this.size = relationships.size();
            this.value = relationships.get(0);
            if (relationships.size() == 1) {
                this.next = null;
            } else {
                Path current = this;
                for (int i = 1; i < relationships.size(); ++i) {
                    current = current.next = new Path(relationships.get(i), null);
                }
            }
        }

        private Path(Relationship value, Path next) {
            this.value = value;
            this.next = next;
            this.size = 1 + (next == null ? 0 : next.size);
        }

        @Override
        public int size() {
            return this.size;
        }

        @Override
        public Relationship get(int index) {
            Path current = this;
            for (int i = 0; i < index; ++i) {
                current = current.next;
                if (current != null) continue;
                throw new IndexOutOfBoundsException("Invalid index " + index + "; size " + this.size());
            }
            return current.value;
        }

        @Override
        public Iterator<Relationship> iterator() {
            return new Iterator<Relationship>(){
                private Path current;
                {
                    this.current = this;
                }

                @Override
                public boolean hasNext() {
                    return this.current != null;
                }

                @Override
                public Relationship next() {
                    if (this.current == null) {
                        throw new NoSuchElementException();
                    }
                    Relationship result = this.current.value;
                    this.current = this.current.next;
                    return result;
                }
            };
        }

        public List<Shape> getShapes() {
            ArrayList<Shape> results = new ArrayList<Shape>(this.size());
            Iterator<Relationship> iterator = this.iterator();
            for (int i = 0; i < this.size(); ++i) {
                Relationship rel = iterator.next();
                results.add(rel.getShape());
                if (i != this.size() - 1) continue;
                rel.getNeighborShape().ifPresent(results::add);
            }
            return results;
        }

        public Shape getStartShape() {
            return this.value.getShape();
        }

        public Shape getEndShape() {
            Relationship last = this.tail();
            return last.getNeighborShape().orElseThrow(() -> new SourceException("Relationship points to a shape that is invalid: " + last, last.getShape()));
        }

        private Relationship tail() {
            Path current = this;
            while (current.next != null) {
                current = current.next;
            }
            return current.value;
        }

        @Override
        public String toString() {
            StringBuilder result = new StringBuilder();
            result.append("[id|").append(this.getStartShape().getId()).append("]");
            for (Relationship rel : this) {
                if (rel.getRelationshipType() == RelationshipType.MEMBER_TARGET) {
                    result.append(" > ");
                } else {
                    result.append(" -[").append(rel.getRelationshipType().getSelectorLabel().orElseGet(() -> rel.getRelationshipType().toString())).append("]-> ");
                }
                result.append("[id|").append(rel.getNeighborShapeId()).append("]");
            }
            return result.toString();
        }
    }
}

