/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.cli.dependencies;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import software.amazon.smithy.build.model.MavenRepository;
import software.amazon.smithy.cli.dependencies.DependencyResolver;
import software.amazon.smithy.cli.dependencies.DependencyResolverException;
import software.amazon.smithy.cli.dependencies.ResolvedArtifact;
import software.amazon.smithy.model.loader.ModelSyntaxException;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.node.ObjectNode;
import software.amazon.smithy.model.node.ToNode;

public final class FileCacheResolver
implements DependencyResolver {
    private static final String CURRENT_CACHE_FILE_VERSION = "1.0";
    private static final Duration SMITHY_MAVEN_TTL = Duration.parse("P1D");
    private static final Logger LOGGER = Logger.getLogger(FileCacheResolver.class.getName());
    private final DependencyResolver delegate;
    private final File location;
    private final long referenceTimeInMillis;

    public FileCacheResolver(File location, long referenceTimeInMillis, DependencyResolver delegate) {
        this.location = location;
        this.referenceTimeInMillis = referenceTimeInMillis;
        this.delegate = delegate;
    }

    @Override
    public void addRepository(MavenRepository repository) {
        this.delegate.addRepository(repository);
    }

    @Override
    public void addDependency(String coordinates) {
        this.delegate.addDependency(coordinates);
    }

    @Override
    public List<ResolvedArtifact> resolve() {
        List<ResolvedArtifact> cachedResult = this.load();
        if (!cachedResult.isEmpty()) {
            LOGGER.fine(() -> "Classpath found in cache: " + cachedResult);
            return cachedResult;
        }
        List<ResolvedArtifact> result = this.delegate.resolve();
        this.save(result);
        return result;
    }

    private List<ResolvedArtifact> load() {
        ObjectNode node;
        long cacheLastModifiedMillis = this.location.lastModified();
        long currentTimeMillis = new Date().getTime();
        long ttlMillis = SMITHY_MAVEN_TTL.toMillis();
        if (this.location.length() == 0L) {
            return Collections.emptyList();
        }
        if (!this.isCacheValid(cacheLastModifiedMillis)) {
            LOGGER.fine("Invalidating dependency cache: config is newer than the cache");
            this.invalidate();
            return Collections.emptyList();
        }
        if (currentTimeMillis - cacheLastModifiedMillis > ttlMillis) {
            LOGGER.fine(() -> "Invalidating dependency cache: Cache exceeded TTL (TTL: " + ttlMillis + ")");
            this.invalidate();
            return Collections.emptyList();
        }
        try (InputStream stream = Files.newInputStream(this.location.toPath(), new OpenOption[0]);){
            node = Node.parse((InputStream)stream, (String)this.location.toString()).expectObjectNode();
        }
        catch (IOException | ModelSyntaxException e) {
            throw new DependencyResolverException("Error loading dependency cache file from " + this.location, e);
        }
        if (!node.containsMember("version") || !CURRENT_CACHE_FILE_VERSION.equals(node.expectStringMember("version").getValue())) {
            LOGGER.fine(() -> "Invalidating dependency cache: cache file uses old version");
            this.invalidate();
            return Collections.emptyList();
        }
        ObjectNode artifactNode = node.expectObjectMember("artifacts");
        ArrayList<ResolvedArtifact> result = new ArrayList<ResolvedArtifact>(artifactNode.getStringMap().size());
        for (Map.Entry entry : artifactNode.getStringMap().entrySet()) {
            ResolvedArtifact artifact = ResolvedArtifact.fromCoordinateNode((String)entry.getKey(), (Node)entry.getValue());
            long lastModifiedOfArtifact = artifact.getLastModified();
            if (lastModifiedOfArtifact == 0L || lastModifiedOfArtifact > cacheLastModifiedMillis) {
                LOGGER.fine(() -> "Invalidating dependency cache: artifact is newer than cache: " + artifact.getPath());
                this.invalidate();
                return Collections.emptyList();
            }
            result.add(artifact);
        }
        return result;
    }

    private void save(List<ResolvedArtifact> result) {
        Path filePath = this.location.toPath();
        Path parent = filePath.getParent();
        if (parent == null) {
            throw new DependencyResolverException("Invalid classpath cache location: " + this.location);
        }
        try {
            Files.createDirectories(parent, new FileAttribute[0]);
            ObjectNode.Builder builder = Node.objectNodeBuilder();
            builder.withMember("version", CURRENT_CACHE_FILE_VERSION);
            ObjectNode.Builder artifactNodeBuilder = Node.objectNodeBuilder();
            for (ResolvedArtifact artifact : result) {
                artifactNodeBuilder.withMember(artifact.getCoordinates(), (ToNode)artifact.toNode());
            }
            builder.withMember("artifacts", (ToNode)artifactNodeBuilder.build());
            Files.write(filePath, Node.printJson((Node)builder.build()).getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        }
        catch (IOException e) {
            throw new DependencyResolverException("Unable to write classpath cache file: " + e.getMessage(), e);
        }
    }

    private boolean isCacheValid(long cacheLastModifiedMillis) {
        return this.referenceTimeInMillis <= cacheLastModifiedMillis;
    }

    private void invalidate() {
        if (this.location.exists() && !this.location.delete()) {
            LOGGER.warning("Unable to invalidate dependency cache file: " + this.location);
        }
    }
}

