/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.cli.dependencies;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import software.amazon.smithy.build.model.MavenRepository;
import software.amazon.smithy.cli.dependencies.DependencyResolver;
import software.amazon.smithy.cli.dependencies.DependencyResolverException;
import software.amazon.smithy.cli.dependencies.ResolvedArtifact;
import software.amazon.smithy.model.loader.ModelSyntaxException;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.node.ObjectNode;

public final class FileCacheResolver
implements DependencyResolver {
    private static final Logger LOGGER = Logger.getLogger(FileCacheResolver.class.getName());
    private final DependencyResolver delegate;
    private final File location;
    private final long referenceTimeInMillis;

    public FileCacheResolver(File location, long referenceTimeInMillis, DependencyResolver delegate) {
        this.location = location;
        this.referenceTimeInMillis = referenceTimeInMillis;
        this.delegate = delegate;
    }

    @Override
    public void addRepository(MavenRepository repository) {
        this.delegate.addRepository(repository);
    }

    @Override
    public void addDependency(String coordinates) {
        this.delegate.addDependency(coordinates);
    }

    @Override
    public List<ResolvedArtifact> resolve() {
        List<ResolvedArtifact> cachedResult = this.load();
        if (!cachedResult.isEmpty()) {
            LOGGER.fine(() -> "Classpath found in cache: " + cachedResult);
            return cachedResult;
        }
        List<ResolvedArtifact> result = this.delegate.resolve();
        this.save(result);
        return result;
    }

    private List<ResolvedArtifact> load() {
        ObjectNode node;
        Path filePath = this.location.toPath();
        if (!Files.exists(filePath, new LinkOption[0])) {
            return Collections.emptyList();
        }
        if (!this.isCacheValid(this.location)) {
            this.invalidate(filePath);
            return Collections.emptyList();
        }
        try {
            InputStream stream = Files.newInputStream(filePath, new OpenOption[0]);
            Object object = null;
            try {
                node = Node.parse((InputStream)stream, (String)this.location.toString()).expectObjectNode();
            }
            catch (Throwable throwable) {
                object = throwable;
                throw throwable;
            }
            finally {
                if (stream != null) {
                    if (object != null) {
                        try {
                            stream.close();
                        }
                        catch (Throwable throwable) {
                            ((Throwable)object).addSuppressed(throwable);
                        }
                    } else {
                        stream.close();
                    }
                }
            }
        }
        catch (IOException | ModelSyntaxException e) {
            throw new DependencyResolverException("Error loading dependency cache file from " + filePath, e);
        }
        ArrayList<ResolvedArtifact> result = new ArrayList<ResolvedArtifact>(node.getStringMap().size());
        for (Map.Entry entry : node.getStringMap().entrySet()) {
            Path location = Paths.get(((Node)entry.getValue()).expectStringNode().getValue(), new String[0]);
            if (this.isArtifactUpdatedSinceReferenceTime(location)) {
                this.invalidate(filePath);
                return Collections.emptyList();
            }
            result.add(ResolvedArtifact.fromCoordinates(location, (String)entry.getKey()));
        }
        return result;
    }

    private void save(List<ResolvedArtifact> result) {
        Path filePath = this.location.toPath();
        Path parent = filePath.getParent();
        if (parent == null) {
            throw new DependencyResolverException("Invalid classpath cache location: " + this.location);
        }
        try {
            Files.createDirectories(parent, new FileAttribute[0]);
            ObjectNode.Builder builder = Node.objectNodeBuilder();
            for (ResolvedArtifact artifact : result) {
                builder.withMember(artifact.getCoordinates(), artifact.getPath().toString());
            }
            ObjectNode objectNode = builder.build();
            Files.write(filePath, Node.printJson((Node)objectNode).getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        }
        catch (IOException e) {
            throw new DependencyResolverException("Unable to write classpath cache file: " + e.getMessage(), e);
        }
    }

    private boolean isCacheValid(File file) {
        return this.referenceTimeInMillis <= file.lastModified() && file.length() > 0L;
    }

    private boolean isArtifactUpdatedSinceReferenceTime(Path path) {
        File file = path.toFile();
        return !file.exists() || this.referenceTimeInMillis > 0L && file.lastModified() > this.referenceTimeInMillis;
    }

    private void invalidate(Path filePath) {
        try {
            if (Files.exists(filePath, new LinkOption[0])) {
                LOGGER.fine("Invalidating dependency cache file: " + this.location);
                Files.delete(filePath);
            }
        }
        catch (IOException e) {
            throw new DependencyResolverException("Unable to delete cache file: " + e.getMessage(), e);
        }
    }
}

