/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.qldb;

import com.amazon.ion.IonSystem;
import java.time.Duration;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.core.exception.SdkException;
import software.amazon.awssdk.services.qldbsession.QldbSessionClient;
import software.amazon.awssdk.utils.Validate;
import software.amazon.qldb.Executor;
import software.amazon.qldb.ExecutorNoReturn;
import software.amazon.qldb.QldbDriver;
import software.amazon.qldb.QldbSession;
import software.amazon.qldb.Result;
import software.amazon.qldb.RetryPolicy;
import software.amazon.qldb.RetryPolicyContext;
import software.amazon.qldb.Session;
import software.amazon.qldb.TableNameIterable;
import software.amazon.qldb.TransactionExecutor;
import software.amazon.qldb.exceptions.Errors;
import software.amazon.qldb.exceptions.ExecuteException;
import software.amazon.qldb.exceptions.QldbDriverException;

@ThreadSafe
class QldbDriverImpl
implements QldbDriver {
    static final String TABLE_NAME_QUERY = "SELECT VALUE name FROM information_schema.user_tables WHERE status = 'ACTIVE'";
    private static final Logger logger = LoggerFactory.getLogger(QldbDriver.class);
    private static final long DEFAULT_TIMEOUT_MS = 1L;
    private final String ledgerName;
    private final Semaphore poolPermits;
    private final BlockingQueue<QldbSession> pool;
    private final int readAhead;
    private final ExecutorService executorService;
    private final QldbSessionClient amazonQldbSession;
    private final RetryPolicy retryPolicy;
    private final IonSystem ionSystem;
    private final AtomicBoolean isClosed;

    protected QldbDriverImpl(String ledgerName, QldbSessionClient qldbSessionClient, RetryPolicy retryPolicy, int readAhead, int maxConcurrentTransactions, IonSystem ionSystem, ExecutorService executorService) {
        this.ledgerName = ledgerName;
        this.amazonQldbSession = qldbSessionClient;
        this.retryPolicy = retryPolicy;
        this.ionSystem = ionSystem;
        this.isClosed = new AtomicBoolean(false);
        this.readAhead = readAhead;
        this.executorService = executorService;
        this.poolPermits = new Semaphore(maxConcurrentTransactions, true);
        this.pool = new LinkedBlockingQueue<QldbSession>();
    }

    @Override
    public void close() {
        if (!this.isClosed.getAndSet(true)) {
            QldbSession curSession = (QldbSession)this.pool.poll();
            while (curSession != null) {
                curSession.close();
                curSession = (QldbSession)this.pool.poll();
            }
        }
    }

    @Override
    public void execute(ExecutorNoReturn executor) {
        this.execute(executor, this.retryPolicy);
    }

    @Override
    public void execute(ExecutorNoReturn executor, RetryPolicy retryPolicy) {
        this.execute((TransactionExecutor txn) -> {
            executor.execute(txn);
            return Boolean.TRUE;
        }, retryPolicy);
    }

    @Override
    public <T> T execute(Executor<T> executor) {
        return this.execute(executor, this.retryPolicy);
    }

    @Override
    public <T> T execute(Executor<T> executor, RetryPolicy retryPolicy) {
        Validate.paramNotNull(executor, (String)"executor");
        Validate.notNull((Object)retryPolicy, (String)"retryPolicy", (Object[])new Object[0]);
        if (this.isClosed.get()) {
            logger.error(Errors.DRIVER_CLOSED.get());
            throw QldbDriverException.create(Errors.DRIVER_CLOSED.get());
        }
        boolean replaceDeadSession = false;
        int retryAttempt = 0;
        while (true) {
            QldbSession session = null;
            try {
                session = replaceDeadSession ? this.createNewSession() : this.getSession();
                T returnedValue = session.execute(executor);
                this.releaseSession(session);
                return returnedValue;
            }
            catch (ExecuteException ee) {
                if (ee.isRetryable() && ee.isInvalidSessionException() && ++retryAttempt == 1) {
                    logger.debug("Initial session received from pool invalid. Retrying...");
                    replaceDeadSession = true;
                    continue;
                }
                if (!ee.isRetryable() || retryAttempt > retryPolicy.maxRetries()) {
                    if (ee.isSessionAlive() && session != null) {
                        this.releaseSession(session);
                    } else {
                        this.poolPermits.release();
                    }
                    throw ee.getCause();
                }
                logger.info("A recoverable error has occurred. Attempting retry #{}.", (Object)retryAttempt);
                logger.debug("Errored Transaction ID: {}. Error cause: ", (Object)ee.getTransactionId(), (Object)ee.getCause());
                boolean bl = replaceDeadSession = !ee.isSessionAlive();
                if (replaceDeadSession) {
                    logger.debug("Replacing invalid session...");
                } else {
                    logger.debug("Retrying with a different session...");
                    this.releaseSession(session);
                }
                try {
                    SdkException se = (SdkException)ee.getCause();
                    RetryPolicyContext context = new RetryPolicyContext(se, retryAttempt, ee.getTransactionId());
                    this.retrySleep(context, retryPolicy);
                }
                catch (Exception e) {
                    if (replaceDeadSession) {
                        this.poolPermits.release();
                    }
                    throw e;
                }
            }
        }
    }

    @Override
    public Iterable<String> getTableNames() {
        Result result = this.execute((TransactionExecutor txn) -> txn.execute(TABLE_NAME_QUERY), this.retryPolicy);
        return new TableNameIterable(result);
    }

    private QldbSession getSession() {
        logger.debug("Getting session. There are {} free sessions; currently available permits is: {}.", (Object)this.pool.size(), (Object)this.poolPermits.availablePermits());
        try {
            if (this.poolPermits.tryAcquire(1L, TimeUnit.MILLISECONDS)) {
                QldbSession session = (QldbSession)this.pool.poll();
                if (session == null) {
                    session = this.createNewSession();
                    logger.debug("Creating new pooled session. Session ID: {}.", (Object)session.getSessionId());
                }
                return session;
            }
            throw QldbDriverException.create(String.format(Errors.NO_SESSION_AVAILABLE.get(), 1L));
        }
        catch (InterruptedException ie) {
            Thread.currentThread().interrupt();
            throw QldbDriverException.create(Errors.GET_SESSION_INTERRUPTED.get());
        }
    }

    private QldbSession createNewSession() {
        try {
            Session session = Session.startSession(this.ledgerName, this.amazonQldbSession);
            return new QldbSession(session, this.readAhead, this.ionSystem, this.executorService);
        }
        catch (SdkException ase) {
            throw new ExecuteException((RuntimeException)((Object)ase), true, false, true, "None");
        }
    }

    private void releaseSession(QldbSession session) {
        this.pool.add(session);
        this.poolPermits.release();
        logger.debug("Session returned to pool; pool size is now: {}.", (Object)this.pool.size());
    }

    private void retrySleep(RetryPolicyContext context, RetryPolicy retryPolicy) {
        try {
            Duration backoffDelay = retryPolicy.backoffStrategy().calculateDelay(context);
            if (backoffDelay == null || backoffDelay.isNegative()) {
                backoffDelay = Duration.ofMillis(0L);
            }
            TimeUnit.MILLISECONDS.sleep(backoffDelay.toMillis());
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }
}

