/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.documentdb.jdbc.sshtunnel;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.channels.Channels;
import java.nio.channels.FileLock;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.sql.SQLException;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.commons.lang3.SystemUtils;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.documentdb.jdbc.DocumentDbConnectionProperties;
import software.amazon.documentdb.jdbc.DocumentDbMain;
import software.amazon.documentdb.jdbc.common.utilities.SqlError;
import software.amazon.documentdb.jdbc.common.utilities.SqlState;
import software.amazon.documentdb.jdbc.sshtunnel.DocumentDbMultiThreadFileChannel;
import software.amazon.documentdb.jdbc.sshtunnel.DocumentDbSshTunnelLock;

public final class DocumentDbSshTunnelServer
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(DocumentDbSshTunnelServer.class);
    private static final int SERVER_WATCHER_POLL_TIME_MS = 500;
    private static final Object MUTEX = new Object();
    private static final String DOCUMENTDB_SSH_TUNNEL_PATH = "DOCUMENTDB_SSH_TUNNEL_PATH";
    private static final String JAVA_HOME = "java.home";
    private static final String JAVA_CLASS_PATH = "java.class.path";
    private static final String CLASS_PATH_OPTION_NAME = "-cp";
    private static final String BIN_FOLDER_NAME = "bin";
    private static final String JAVA_EXECUTABLE_NAME = "java";
    private static final String SSH_TUNNEL_SERVICE_OPTION_NAME = "--ssh-tunnel-service";
    public static final int SERVICE_WAIT_TIMEOUT_SECONDS = 120;
    public static final String FILE_SCHEME = "file";
    private final AtomicInteger clientCount = new AtomicInteger(0);
    private final String sshUser;
    private final String sshHostname;
    private final String sshPrivateKeyFile;
    private final String sshPrivateKeyPassphrase;
    private final boolean sshStrictHostKeyChecking;
    private final String sshKnownHostsFile;
    private final String remoteHostname;
    private final String propertiesHashString;
    private final AtomicBoolean serverAlive = new AtomicBoolean(false);
    private ServerWatcher serverWatcher = null;
    private Thread serverWatcherThread = null;
    private volatile int serviceListeningPort = 0;

    private DocumentDbSshTunnelServer(DocumentDbSshTunnelServerBuilder builder) {
        this.sshUser = builder.sshUser;
        this.sshHostname = builder.sshHostname;
        this.sshPrivateKeyFile = builder.sshPrivateKeyFile;
        this.remoteHostname = builder.sshRemoteHostname;
        this.sshPrivateKeyPassphrase = builder.sshPrivateKeyPassphrase;
        this.sshStrictHostKeyChecking = builder.sshStrictHostKeyChecking;
        this.sshKnownHostsFile = builder.sshKnownHostsFile;
        this.propertiesHashString = DocumentDbSshTunnelLock.getHashString(this.sshUser, this.sshHostname, this.sshPrivateKeyFile, this.remoteHostname);
        LOGGER.debug("sshUser='{}' sshHostname='{}' sshPrivateKeyFile='{}' remoteHostname'{} sshPrivateKeyPassphrase='{}' sshStrictHostKeyChecking='{}' sshKnownHostsFile='{}'", new Object[]{this.sshUser, this.sshHostname, this.sshPrivateKeyFile, this.remoteHostname, this.sshPrivateKeyPassphrase, this.sshStrictHostKeyChecking, this.sshKnownHostsFile});
    }

    public int getServiceListeningPort() {
        return this.serviceListeningPort;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws Exception {
        Object object = MUTEX;
        synchronized (object) {
            this.serviceListeningPort = 0;
            this.shutdownServerWatcherThread();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addClient() throws Exception {
        Object object = MUTEX;
        synchronized (object) {
            if (this.clientCount.get() == 0) {
                this.ensureStarted();
            }
            this.clientCount.incrementAndGet();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void removeClient() throws Exception {
        Object object = MUTEX;
        synchronized (object) {
            if (this.clientCount.decrementAndGet() == 0) {
                this.close();
            }
        }
    }

    private void shutdownServerWatcherThread() throws Exception {
        if (this.serverWatcherThread.isAlive()) {
            LOGGER.info("Stopping server watcher thread.");
            this.serverWatcher.close();
            do {
                this.serverWatcherThread.join(1000L);
            } while (this.serverWatcherThread.isAlive());
            Queue<Exception> exceptions = this.serverWatcher.getExceptions();
            if (!exceptions.isEmpty()) {
                for (Exception e : exceptions) {
                    LOGGER.error(e.getMessage(), (Throwable)e);
                }
            }
            LOGGER.info("Stopped server watcher thread.");
        } else {
            LOGGER.info("Server watcher thread already stopped.");
        }
    }

    public boolean isAlive() throws Exception {
        if (this.serverWatcherThread.isAlive()) {
            return this.serverAlive.get();
        }
        Path serverLockPath = DocumentDbSshTunnelLock.getServerLockPath(this.propertiesHashString);
        try (DocumentDbMultiThreadFileChannel serverChannel = DocumentDbMultiThreadFileChannel.open(serverLockPath, StandardOpenOption.CREATE, StandardOpenOption.WRITE);){
            FileLock serverLock = serverChannel.tryLock();
            if (serverLock != null) {
                boolean bl = false;
                return bl;
            }
        }
        return true;
    }

    public static DocumentDbSshTunnelServerBuilder builder(String user, String hostname, String privateKeyFile, String remoteHostname) {
        return new DocumentDbSshTunnelServerBuilder(user, hostname, privateKeyFile, remoteHostname);
    }

    void ensureStarted() throws Exception {
        this.maybeStart();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void maybeStart() throws Exception {
        Object object = MUTEX;
        synchronized (object) {
            if (this.serviceListeningPort != 0) {
                return;
            }
            AtomicReference<Object> exception = new AtomicReference<Object>(null);
            DocumentDbSshTunnelLock.runInGlobalLock(this.propertiesHashString, () -> this.maybeStartServerHandleException(exception));
            if (exception.get() != null) {
                throw (Exception)exception.get();
            }
        }
    }

    private Exception maybeStartServerHandleException(AtomicReference<Exception> exception) {
        try {
            this.maybeStartServer();
            return null;
        }
        catch (Exception e) {
            exception.set(e);
            return e;
        }
    }

    private void maybeStartServer() throws Exception {
        Path serverLockPath = DocumentDbSshTunnelLock.getServerLockPath(this.propertiesHashString);
        try (DocumentDbMultiThreadFileChannel serverChannel = DocumentDbMultiThreadFileChannel.open(serverLockPath, StandardOpenOption.CREATE, StandardOpenOption.WRITE);){
            FileLock serverLock = serverChannel.tryLock();
            if (serverLock != null) {
                this.validateLocalSshFilesExists();
                Path startupLockPath = DocumentDbSshTunnelLock.getStartupLockPath(this.propertiesHashString);
                Path portLockPath = DocumentDbSshTunnelLock.getPortLockPath(this.propertiesHashString);
                DocumentDbSshTunnelLock.deleteStartupAndPortLockFiles(startupLockPath, portLockPath);
                if (serverLock.isValid()) {
                    serverLock.close();
                }
                Process process = this.startSshTunnelServiceProcess();
                this.waitForStartupAndReadPort(startupLockPath, process);
                Instant timeoutTime = Instant.now().plus(Duration.ofSeconds(120L));
                do {
                    if ((serverLock = serverChannel.tryLock()) == null || !serverLock.isValid()) continue;
                    serverLock.close();
                    DocumentDbSshTunnelServer.throwIfTimeout(timeoutTime, "Timeout waiting for service to acquire server lock.");
                    TimeUnit.MILLISECONDS.sleep(500L);
                } while (serverLock != null);
                this.startServerWatcherThread();
            } else {
                LOGGER.info("Server already running.");
                this.readSshPortFromFile();
                this.startServerWatcherThread();
            }
        }
    }

    private static void throwIfTimeout(Instant timeoutTime, String message) throws SQLException {
        if (Instant.now().isAfter(timeoutTime)) {
            throw SqlError.createSQLException(LOGGER, SqlState.CONNECTION_EXCEPTION, SqlError.SSH_TUNNEL_ERROR, message);
        }
    }

    private void startServerWatcherThread() {
        this.serverWatcher = new ServerWatcher(this.propertiesHashString, this.serverAlive);
        this.serverWatcherThread = new Thread(this.serverWatcher);
        this.serverWatcherThread.setDaemon(true);
        this.serverWatcherThread.start();
    }

    private void waitForStartupAndReadPort(Path startupLockPath, Process process) throws Exception {
        int pollTimeMS = 100;
        while (!Files.exists(startupLockPath, new LinkOption[0])) {
            TimeUnit.MILLISECONDS.sleep(100L);
        }
        try (DocumentDbMultiThreadFileChannel startupChannel = DocumentDbMultiThreadFileChannel.open(startupLockPath, StandardOpenOption.WRITE, StandardOpenOption.READ);){
            FileLock startupLock;
            LOGGER.info("Waiting for server to unlock Startup lock file.");
            Instant timeoutTime = Instant.now().plus(Duration.ofSeconds(120L));
            do {
                if ((startupLock = startupChannel.tryLock()) != null) continue;
                DocumentDbSshTunnelServer.throwIfProcessHasExited(process);
                DocumentDbSshTunnelServer.throwIfTimeout(timeoutTime, "Timeout waiting for service to release Startup lock.");
                TimeUnit.MILLISECONDS.sleep(100L);
            } while (startupLock == null);
            LOGGER.info("Server has unlocked Startup lock file.");
            LOGGER.info("Reading Startup lock file.");
            try (InputStream inputStream = Channels.newInputStream(startupChannel.getFileChannel());
                 InputStreamReader streamReader = new InputStreamReader(inputStream, StandardCharsets.UTF_8);
                 BufferedReader reader = new BufferedReader(streamReader);){
                String line;
                StringBuilder exceptionMessage = new StringBuilder();
                boolean isFirstLine = true;
                while ((line = reader.readLine()) != null && !DocumentDbConnectionProperties.isNullOrWhitespace(line)) {
                    if (!isFirstLine) {
                        exceptionMessage.append(System.lineSeparator());
                    } else {
                        isFirstLine = false;
                    }
                    exceptionMessage.append(line);
                }
                if (exceptionMessage.length() > 0) {
                    exceptionMessage.insert(0, "Server exception detected: '").append("'");
                    throw SqlError.createSQLException(LOGGER, SqlState.CONNECTION_EXCEPTION, SqlError.SSH_TUNNEL_ERROR, exceptionMessage.toString());
                }
                LOGGER.info("Finished reading Startup lock file.");
            }
            LOGGER.info("Reading local port number from file.");
            this.readSshPortFromFile();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void throwIfProcessHasExited(Process process) throws InterruptedException, SQLException {
        Process process2 = process;
        synchronized (process2) {
            if (process.waitFor(1L, TimeUnit.MILLISECONDS)) {
                throw SqlError.createSQLException(LOGGER, SqlState.CONNECTION_EXCEPTION, SqlError.SSH_TUNNEL_ERROR, "Service has unexpected exited.");
            }
        }
    }

    private Process startSshTunnelServiceProcess() throws IOException, SQLException, URISyntaxException {
        List<String> command = this.getSshTunnelCommand();
        ProcessBuilder builder = new ProcessBuilder(command);
        return builder.inheritIO().start();
    }

    private List<String> getSshTunnelCommand() throws SQLException, URISyntaxException {
        LinkedList<String> command = new LinkedList<String>();
        String docDbSshTunnelPathString = System.getenv(DOCUMENTDB_SSH_TUNNEL_PATH);
        if (docDbSshTunnelPathString != null) {
            Path docDbSshTunnelPath = Paths.get(docDbSshTunnelPathString, new String[0]);
            if (!Files.isExecutable(docDbSshTunnelPath) || Files.isDirectory(docDbSshTunnelPath, new LinkOption[0])) {
                throw SqlError.createSQLException(LOGGER, SqlState.CONNECTION_EXCEPTION, SqlError.SSH_TUNNEL_PATH_NOT_FOUND, docDbSshTunnelPathString);
            }
            command.add(docDbSshTunnelPath.toAbsolutePath().toString());
            command.add(SSH_TUNNEL_SERVICE_OPTION_NAME);
            command.add(this.getSshPropertiesString());
        } else {
            String className = DocumentDbMain.class.getName();
            String sshConnectionProperties = this.getSshPropertiesString();
            command.addAll(DocumentDbSshTunnelServer.getJavaCommand(className, SSH_TUNNEL_SERVICE_OPTION_NAME, sshConnectionProperties));
        }
        return command;
    }

    public static List<String> getJavaCommand(String className, String ... arguments) throws SQLException, URISyntaxException {
        String javaBinFilePath = DocumentDbSshTunnelServer.getJavaBinFilePath();
        String combinedClassPath = DocumentDbSshTunnelServer.getCombinedClassPath();
        LinkedList<String> command = new LinkedList<String>();
        command.add(javaBinFilePath);
        command.add(CLASS_PATH_OPTION_NAME);
        command.add(combinedClassPath);
        command.add(className);
        command.addAll(Arrays.asList(arguments));
        return command;
    }

    private static String getJavaBinFilePath() throws SQLException {
        String javaHome = DocumentDbSshTunnelServer.getJavaHome();
        String javaBinFilePath = Paths.get(javaHome, BIN_FOLDER_NAME, JAVA_EXECUTABLE_NAME).toString();
        boolean isOsWindows = SystemUtils.IS_OS_WINDOWS;
        Path javaBinPath = Paths.get(javaBinFilePath + (isOsWindows ? ".exe" : ""), new String[0]);
        if (!Files.exists(javaBinPath, new LinkOption[0]) || !Files.isExecutable(javaBinPath)) {
            throw SqlError.createSQLException(LOGGER, SqlState.CONNECTION_EXCEPTION, SqlError.MISSING_JAVA_BIN, javaBinPath.toString());
        }
        return javaBinFilePath;
    }

    private static String getJavaHome() throws SQLException {
        String javaHome = System.getProperty(JAVA_HOME);
        if (DocumentDbConnectionProperties.isNullOrWhitespace(javaHome) || !Files.exists(Paths.get(javaHome, new String[0]), new LinkOption[0])) {
            throw SqlError.createSQLException(LOGGER, SqlState.CONNECTION_EXCEPTION, SqlError.MISSING_JAVA_HOME, new Object[0]);
        }
        return javaHome;
    }

    private static String getCombinedClassPath() throws URISyntaxException {
        URI currentClassPathUri = DocumentDbSshTunnelServer.class.getProtectionDomain().getCodeSource().getLocation().toURI();
        String schemeSpecificPart = currentClassPathUri.getSchemeSpecificPart();
        if (currentClassPathUri.getScheme().equalsIgnoreCase(FILE_SCHEME) && !DocumentDbConnectionProperties.isNullOrWhitespace(schemeSpecificPart)) {
            String startsWithSlashExpression = "^/+";
            currentClassPathUri = new URI(currentClassPathUri.getScheme() + ":/" + schemeSpecificPart.replaceAll("^/+", ""));
        }
        String currentClassCodeSourcePath = new File(currentClassPathUri).getAbsolutePath();
        return currentClassCodeSourcePath + ";" + System.getProperty(JAVA_CLASS_PATH);
    }

    private @NonNull String getSshPropertiesString() {
        DocumentDbConnectionProperties connectionProperties = this.getConnectionProperties();
        return "jdbc:documentdb:" + connectionProperties.buildSshConnectionString();
    }

    private @NonNull DocumentDbConnectionProperties getConnectionProperties() {
        DocumentDbConnectionProperties connectionProperties = new DocumentDbConnectionProperties();
        connectionProperties.setHostname(this.remoteHostname);
        connectionProperties.setSshUser(this.sshUser);
        connectionProperties.setSshHostname(this.sshHostname);
        connectionProperties.setSshPrivateKeyFile(this.sshPrivateKeyFile);
        connectionProperties.setSshStrictHostKeyChecking(String.valueOf(this.sshStrictHostKeyChecking));
        if (this.sshPrivateKeyPassphrase != null) {
            connectionProperties.setSshPrivateKeyPassphrase(this.sshPrivateKeyPassphrase);
        }
        if (this.sshKnownHostsFile != null) {
            connectionProperties.setSshKnownHostsFile(this.sshKnownHostsFile);
        }
        return connectionProperties;
    }

    private void validateLocalSshFilesExists() throws SQLException {
        DocumentDbConnectionProperties connectionProperties = this.getConnectionProperties();
        DocumentDbSshTunnelServer.validateSshPrivateKeyFile(connectionProperties);
        DocumentDbSshTunnelServer.getSshKnownHostsFilename(connectionProperties);
    }

    static void validateSshPrivateKeyFile(DocumentDbConnectionProperties connectionProperties) throws SQLException {
        if (!connectionProperties.isSshPrivateKeyFileExists()) {
            throw SqlError.createSQLException(LOGGER, SqlState.CONNECTION_EXCEPTION, SqlError.SSH_PRIVATE_KEY_FILE_NOT_FOUND, connectionProperties.getSshPrivateKeyFile());
        }
    }

    static String getSshKnownHostsFilename(DocumentDbConnectionProperties connectionProperties) throws SQLException {
        String knowHostsFilename;
        if (!DocumentDbConnectionProperties.isNullOrWhitespace(connectionProperties.getSshKnownHostsFile())) {
            Path knownHostsPath = DocumentDbConnectionProperties.getPath(connectionProperties.getSshKnownHostsFile(), new String[0]);
            DocumentDbSshTunnelServer.validateSshKnownHostsFile(connectionProperties, knownHostsPath);
            knowHostsFilename = knownHostsPath.toString();
        } else {
            knowHostsFilename = DocumentDbConnectionProperties.getPath("~/.ssh/known_hosts", new String[0]).toString();
        }
        return knowHostsFilename;
    }

    private static void validateSshKnownHostsFile(DocumentDbConnectionProperties connectionProperties, Path knownHostsPath) throws SQLException {
        if (!Files.exists(knownHostsPath, new LinkOption[0])) {
            throw SqlError.createSQLException(LOGGER, SqlState.INVALID_PARAMETER_VALUE, SqlError.KNOWN_HOSTS_FILE_NOT_FOUND, connectionProperties.getSshKnownHostsFile());
        }
    }

    private void readSshPortFromFile() throws IOException, SQLException {
        Path portLockPath = DocumentDbSshTunnelLock.getPortLockPath(this.propertiesHashString);
        List<String> lines = Files.readAllLines(portLockPath, StandardCharsets.UTF_8);
        int port = 0;
        for (String line : lines) {
            if (!line.trim().isEmpty() && (port = Integer.parseInt(line.trim())) > 0) break;
        }
        if (port <= 0) {
            this.serviceListeningPort = 0;
            throw SqlError.createSQLException(LOGGER, SqlState.CONNECTION_EXCEPTION, SqlError.SSH_TUNNEL_ERROR, "Unable to read valid listening port for SSH Tunnel service.");
        }
        this.serviceListeningPort = port;
        LOGGER.info("SHH tunnel service listening on port: " + this.serviceListeningPort);
    }

    private static class ServerWatcher
    implements Runnable,
    AutoCloseable {
        private volatile ServerWatcherState state = ServerWatcherState.INITIALIZED;
        private final Queue<Exception> exceptions = new ConcurrentLinkedDeque<Exception>();
        private final String propertiesHashString;
        private final AtomicBoolean serverAlive;

        ServerWatcher(String propertiesHashString, AtomicBoolean serverAlive) {
            this.propertiesHashString = propertiesHashString;
            this.serverAlive = serverAlive;
        }

        public Queue<Exception> getExceptions() {
            return this.exceptions;
        }

        @Override
        public void run() {
            try {
                this.state = ServerWatcherState.RUNNING;
                do {
                    DocumentDbSshTunnelLock.runInGlobalLock(this.propertiesHashString, this::checkForServerLock);
                    if (this.state != ServerWatcherState.RUNNING) continue;
                    TimeUnit.MILLISECONDS.sleep(500L);
                } while (this.state == ServerWatcherState.RUNNING);
            }
            catch (Exception e) {
                this.exceptions.add(e);
            }
        }

        @Override
        public void close() throws Exception {
            this.state = ServerWatcherState.INTERRUPTED;
        }

        private Exception checkForServerLock() {
            Exception exception = null;
            Path serverLockPath = DocumentDbSshTunnelLock.getServerLockPath(this.propertiesHashString);
            try (DocumentDbMultiThreadFileChannel serverChannel = DocumentDbMultiThreadFileChannel.open(serverLockPath, StandardOpenOption.CREATE, StandardOpenOption.WRITE);){
                FileLock serverLock = serverChannel.tryLock();
                if (serverLock != null) {
                    this.serverAlive.compareAndSet(true, false);
                    this.state = ServerWatcherState.COMPLETED;
                } else {
                    this.serverAlive.compareAndSet(false, true);
                }
            }
            catch (Exception e) {
                exception = e;
                this.exceptions.add(e);
                this.state = ServerWatcherState.ERROR;
            }
            return exception;
        }
    }

    private static enum ServerWatcherState {
        INITIALIZED,
        RUNNING,
        INTERRUPTED,
        COMPLETED,
        ERROR;

    }

    public static class DocumentDbSshTunnelServerBuilder {
        private final String sshUser;
        private final String sshHostname;
        private final String sshPrivateKeyFile;
        private final String sshRemoteHostname;
        private String sshPrivateKeyPassphrase = null;
        private boolean sshStrictHostKeyChecking = true;
        private String sshKnownHostsFile = null;
        private static final ConcurrentMap<String, DocumentDbSshTunnelServer> SSH_TUNNEL_MAP = new ConcurrentHashMap<String, DocumentDbSshTunnelServer>();

        DocumentDbSshTunnelServerBuilder(String sshUser, String sshHostname, String sshPrivateKeyFile, String sshRemoteHostname) {
            this.sshUser = sshUser;
            this.sshHostname = sshHostname;
            this.sshPrivateKeyFile = sshPrivateKeyFile;
            this.sshRemoteHostname = sshRemoteHostname;
        }

        public DocumentDbSshTunnelServerBuilder sshPrivateKeyPassphrase(String sshPrivateKeyPassphrase) {
            this.sshPrivateKeyPassphrase = sshPrivateKeyPassphrase;
            return this;
        }

        public DocumentDbSshTunnelServerBuilder sshStrictHostKeyChecking(boolean sshStrictHostKeyChecking) {
            this.sshStrictHostKeyChecking = sshStrictHostKeyChecking;
            return this;
        }

        public DocumentDbSshTunnelServerBuilder sshKnownHostsFile(String sshKnownHostsFile) {
            this.sshKnownHostsFile = sshKnownHostsFile;
            return this;
        }

        public DocumentDbSshTunnelServer build() {
            String hashString = DocumentDbSshTunnelLock.getHashString(this.sshUser, this.sshHostname, this.sshPrivateKeyFile, this.sshRemoteHostname);
            return SSH_TUNNEL_MAP.computeIfAbsent(hashString, key -> new DocumentDbSshTunnelServer(this));
        }
    }
}

