/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.documentdb.jdbc;

import com.jcraft.jsch.HostKey;
import com.jcraft.jsch.HostKeyRepository;
import com.jcraft.jsch.JSch;
import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.Session;
import com.mongodb.MongoClientSettings;
import com.mongodb.MongoCommandException;
import com.mongodb.MongoSecurityException;
import com.mongodb.client.MongoClient;
import com.mongodb.client.MongoClients;
import com.mongodb.client.MongoDatabase;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.Statement;
import java.util.Arrays;
import java.util.concurrent.Executor;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.documentdb.jdbc.DocumentDbConnectionProperties;
import software.amazon.documentdb.jdbc.DocumentDbConnectionProperty;
import software.amazon.documentdb.jdbc.DocumentDbDatabaseMetaData;
import software.amazon.documentdb.jdbc.DocumentDbPreparedStatement;
import software.amazon.documentdb.jdbc.DocumentDbStatement;
import software.amazon.documentdb.jdbc.common.Connection;
import software.amazon.documentdb.jdbc.common.utilities.SqlError;
import software.amazon.documentdb.jdbc.common.utilities.SqlState;
import software.amazon.documentdb.jdbc.metadata.DocumentDbDatabaseSchemaMetadata;

public class DocumentDbConnection
extends Connection
implements java.sql.Connection {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)DocumentDbConnection.class.getName());
    public static final String SSH_KNOWN_HOSTS_FILE = "~/.ssh/known_hosts";
    public static final String STRICT_HOST_KEY_CHECKING = "StrictHostKeyChecking";
    public static final String HASH_KNOWN_HOSTS = "HashKnownHosts";
    public static final String SERVER_HOST_KEY = "server_host_key";
    public static final String YES = "yes";
    public static final String NO = "no";
    public static final String LOCALHOST = "localhost";
    public static final int DEFAULT_DOCUMENTDB_PORT = 27017;
    public static final int DEFAULT_SSH_PORT = 22;
    private final DocumentDbConnectionProperties connectionProperties;
    private DocumentDbDatabaseMetaData metadata;
    private DocumentDbDatabaseSchemaMetadata databaseMetadata;
    private MongoClient mongoClient = null;
    private MongoDatabase mongoDatabase = null;
    private SshPortForwardingSession session;

    DocumentDbConnection(DocumentDbConnectionProperties connectionProperties) throws SQLException {
        super(connectionProperties);
        this.connectionProperties = connectionProperties;
        if (LOGGER.isDebugEnabled()) {
            StringBuilder sb = new StringBuilder();
            sb.append("Creating connection with following properties:");
            for (String propertyName : connectionProperties.stringPropertyNames()) {
                if (DocumentDbConnectionProperty.PASSWORD.getName().equals(propertyName)) continue;
                sb.append(String.format("%n        Connection property %s=%s", propertyName, connectionProperties.get(propertyName).toString()));
            }
            LOGGER.debug(sb.toString());
        }
        this.session = DocumentDbConnection.createSshTunnel(connectionProperties);
        this.initializeClients(connectionProperties);
    }

    public static SshPortForwardingSession createSshTunnel(DocumentDbConnectionProperties connectionProperties) throws SQLException {
        if (!connectionProperties.enableSshTunnel()) {
            LOGGER.info("Internal SSH tunnel not started.");
            return null;
        }
        if (!connectionProperties.isSshPrivateKeyFileExists()) {
            throw SqlError.createSQLException(LOGGER, SqlState.CONNECTION_EXCEPTION, SqlError.SSH_PRIVATE_KEY_FILE_NOT_FOUND, connectionProperties.getSshPrivateKeyFile());
        }
        LOGGER.info("Internal SSH tunnel starting.");
        try {
            JSch jSch = new JSch();
            DocumentDbConnection.addIdentity(connectionProperties, jSch);
            Session session = DocumentDbConnection.createSession(connectionProperties, jSch);
            DocumentDbConnection.connectSession(connectionProperties, jSch, session);
            SshPortForwardingSession portForwardingSession = DocumentDbConnection.getPortForwardingSession(connectionProperties, session);
            LOGGER.info("Internal SSH tunnel started on local port '{}'.", (Object)portForwardingSession.localPort);
            return portForwardingSession;
        }
        catch (SQLException e) {
            throw e;
        }
        catch (Exception e) {
            throw new SQLException(e.getMessage(), e);
        }
    }

    public int getSshLocalPort() {
        if (this.isSshTunnelActive()) {
            return this.session.localPort;
        }
        return 0;
    }

    public boolean isSshTunnelActive() {
        return this.session != null;
    }

    @Override
    public boolean isValid(int timeout) throws SQLException {
        if (timeout < 0) {
            throw SqlError.createSQLException(LOGGER, SqlState.INVALID_PARAMETER_VALUE, SqlError.INVALID_TIMEOUT, timeout);
        }
        if (this.mongoDatabase != null) {
            try {
                int maxTimeMS = timeout + 1000;
                this.pingDatabase(maxTimeMS);
                return true;
            }
            catch (Exception e) {
                LOGGER.error(e.getMessage(), (Throwable)e);
            }
        }
        return false;
    }

    @Override
    public void doClose() {
        if (this.mongoDatabase != null) {
            this.mongoDatabase = null;
        }
        if (this.mongoClient != null) {
            this.mongoClient.close();
            this.mongoClient = null;
        }
        if (this.session != null) {
            this.session.session.disconnect();
            this.session = null;
        }
    }

    @Override
    public DatabaseMetaData getMetaData() throws SQLException {
        this.ensureDatabaseMetadata();
        return this.metadata;
    }

    public MongoClient getMongoClient() {
        return this.mongoClient;
    }

    private void ensureDatabaseMetadata() throws SQLException {
        if (this.metadata == null) {
            int version;
            if (this.connectionProperties.getRefreshSchema().booleanValue()) {
                version = -1;
                LOGGER.warn("The '{}' option is enabled and will cause a new version of the SQL schema to be generated. This can lead to poor performance. Please disable this option when it is no longer needed.", (Object)DocumentDbConnectionProperty.REFRESH_SCHEMA.getName());
            } else {
                version = 0;
            }
            this.setMetadata(version);
        }
    }

    private void setMetadata(int version) throws SQLException {
        this.databaseMetadata = DocumentDbDatabaseSchemaMetadata.get(this.connectionProperties, this.connectionProperties.getSchemaName(), version, this.getMongoClient());
        this.metadata = new DocumentDbDatabaseMetaData(this, this.databaseMetadata, this.connectionProperties);
    }

    void refreshDatabaseMetadata() throws SQLException {
        this.setMetadata(-1);
    }

    DocumentDbDatabaseSchemaMetadata getDatabaseMetadata() throws SQLException {
        this.ensureDatabaseMetadata();
        return this.databaseMetadata;
    }

    @Override
    public String getSchema() {
        return this.connectionProperties.getDatabase();
    }

    @Override
    public int getNetworkTimeout() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setNetworkTimeout(Executor executor, int milliseconds) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public Statement createStatement(int resultSetType, int resultSetConcurrency) throws SQLException {
        this.verifyOpen();
        if (resultSetType != 1003 || resultSetConcurrency != 1007) {
            throw SqlError.createSQLFeatureNotSupportedException(LOGGER, SqlError.UNSUPPORTED_RESULT_SET_TYPE, new Object[0]);
        }
        return new DocumentDbStatement(this);
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {
        this.verifyOpen();
        if (resultSetType != 1003 || resultSetConcurrency != 1007) {
            throw SqlError.createSQLFeatureNotSupportedException(LOGGER, SqlError.UNSUPPORTED_RESULT_SET_TYPE, new Object[0]);
        }
        return new DocumentDbPreparedStatement(this, sql);
    }

    @Override
    public boolean isSupportedProperty(String name) {
        return DocumentDbConnectionProperty.isSupportedProperty(name);
    }

    DocumentDbConnectionProperties getConnectionProperties() {
        return this.connectionProperties;
    }

    private void initializeClients(DocumentDbConnectionProperties connectionProperties) throws SQLException {
        MongoClientSettings settings = connectionProperties.buildMongoClientSettings(this.getSshLocalPort());
        this.mongoClient = MongoClients.create((MongoClientSettings)settings);
        this.mongoDatabase = this.mongoClient.getDatabase(connectionProperties.getDatabase());
        this.pingDatabase();
    }

    private void pingDatabase() throws SQLException {
        this.pingDatabase(0);
    }

    private void pingDatabase(int maxTimeMS) throws SQLException {
        try {
            String maxTimeMSOption = maxTimeMS > 0 ? String.format(", \"maxTimeMS\" : %d", maxTimeMS) : "";
            this.mongoDatabase.runCommand((Bson)Document.parse((String)String.format("{ \"ping\" : 1 %s }", maxTimeMSOption)));
        }
        catch (MongoSecurityException e) {
            if (e.getCode() == -4 && e.getCause() != null && e.getCause() instanceof MongoCommandException && ((MongoCommandException)e.getCause()).getCode() == 18) {
                throw SqlError.createSQLException(LOGGER, SqlState.INVALID_AUTHORIZATION_SPECIFICATION, (Exception)((Object)e), SqlError.AUTHORIZATION_ERROR, this.mongoDatabase.getName(), e.getCredential().getUserName(), e.getCredential().getSource(), e.getCredential().getMechanism());
            }
            throw SqlError.createSQLException(LOGGER, SqlState.SQL_CLIENT_UNABLE_TO_ESTABLISH_SQL_CONNECTION, (Exception)((Object)e), SqlError.SECURITY_ERROR, e.getMessage());
        }
        catch (Exception e) {
            throw new SQLException(e.getMessage(), e);
        }
    }

    private static SshPortForwardingSession getPortForwardingSession(DocumentDbConnectionProperties connectionProperties, Session session) throws JSchException {
        Pair<String, Integer> clusterHostAndPort = DocumentDbConnection.getHostAndPort(connectionProperties.getHostname(), 27017);
        int localPort = session.setPortForwardingL(LOCALHOST, 0, (String)clusterHostAndPort.getLeft(), ((Integer)clusterHostAndPort.getRight()).intValue());
        return new SshPortForwardingSession(session, localPort);
    }

    private static Pair<String, Integer> getHostAndPort(String hostname, int defaultPort) {
        int clusterPort;
        String clusterHost;
        int portSeparatorIndex = hostname.indexOf(58);
        if (portSeparatorIndex >= 0) {
            clusterHost = hostname.substring(0, portSeparatorIndex);
            clusterPort = Integer.parseInt(hostname.substring(portSeparatorIndex + 1));
        } else {
            clusterHost = hostname;
            clusterPort = defaultPort;
        }
        return new ImmutablePair((Object)clusterHost, (Object)clusterPort);
    }

    private static void connectSession(DocumentDbConnectionProperties connectionProperties, JSch jSch, Session session) throws SQLException {
        DocumentDbConnection.setSecurityConfig(connectionProperties, jSch, session);
        try {
            session.connect();
        }
        catch (JSchException e) {
            throw new SQLException(e.getMessage(), e);
        }
    }

    private static void addIdentity(DocumentDbConnectionProperties connectionProperties, JSch jSch) throws JSchException {
        String privateKeyFileName = DocumentDbConnectionProperties.getPath(connectionProperties.getSshPrivateKeyFile(), DocumentDbConnectionProperties.getSshPrivateKeyFileSearchPaths()).toString();
        LOGGER.debug("SSH private key file resolved to '{}'.", (Object)privateKeyFileName);
        String passPhrase = !DocumentDbConnectionProperties.isNullOrWhitespace(connectionProperties.getSshPrivateKeyPassphrase()) ? connectionProperties.getSshPrivateKeyPassphrase() : null;
        jSch.addIdentity(privateKeyFileName, passPhrase);
    }

    private static Session createSession(DocumentDbConnectionProperties connectionProperties, JSch jSch) throws SQLException {
        String sshUsername = connectionProperties.getSshUser();
        Pair<String, Integer> sshHostAndPort = DocumentDbConnection.getHostAndPort(connectionProperties.getSshHostname(), 22);
        DocumentDbConnection.setKnownHostsFile(connectionProperties, jSch);
        try {
            return jSch.getSession(sshUsername, (String)sshHostAndPort.getLeft(), ((Integer)sshHostAndPort.getRight()).intValue());
        }
        catch (JSchException e) {
            throw new SQLException(e.getMessage(), e);
        }
    }

    private static void setSecurityConfig(DocumentDbConnectionProperties connectionProperties, JSch jSch, Session session) {
        if (!connectionProperties.getSshStrictHostKeyChecking()) {
            session.setConfig(STRICT_HOST_KEY_CHECKING, NO);
            return;
        }
        DocumentDbConnection.setHostKeyType(connectionProperties, jSch, session);
    }

    private static void setHostKeyType(DocumentDbConnectionProperties connectionProperties, JSch jSch, Session session) {
        String hostKeyType;
        HostKeyRepository keyRepository = jSch.getHostKeyRepository();
        HostKey[] hostKeys = keyRepository.getHostKey();
        Pair<String, Integer> sshHostAndPort = DocumentDbConnection.getHostAndPort(connectionProperties.getSshHostname(), 22);
        HostKey hostKey = Arrays.stream(hostKeys).filter(hk -> hk.getHost().equals(sshHostAndPort.getLeft())).findFirst().orElse(null);
        String string = hostKeyType = hostKey != null ? hostKey.getType() : null;
        if (hostKeyType != null) {
            session.setConfig(SERVER_HOST_KEY, hostKeyType);
        }
        session.setConfig(HASH_KNOWN_HOSTS, YES);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private static void setKnownHostsFile(DocumentDbConnectionProperties connectionProperties, JSch jSch) throws SQLException {
        String knowHostsFilename;
        if (!connectionProperties.getSshStrictHostKeyChecking()) {
            return;
        }
        if (!DocumentDbConnectionProperties.isNullOrWhitespace(connectionProperties.getSshKnownHostsFile())) {
            Path knownHostsPath = DocumentDbConnectionProperties.getPath(connectionProperties.getSshKnownHostsFile(), new String[0]);
            if (!Files.exists(knownHostsPath, new LinkOption[0])) throw SqlError.createSQLException(LOGGER, SqlState.INVALID_PARAMETER_VALUE, SqlError.KNOWN_HOSTS_FILE_NOT_FOUND, connectionProperties.getSshKnownHostsFile());
            knowHostsFilename = knownHostsPath.toString();
        } else {
            knowHostsFilename = DocumentDbConnectionProperties.getPath(SSH_KNOWN_HOSTS_FILE, new String[0]).toString();
        }
        try {
            jSch.setKnownHosts(knowHostsFilename);
            return;
        }
        catch (JSchException e) {
            throw new SQLException(e.getMessage(), e);
        }
    }

    public static class SshPortForwardingSession {
        private final Session session;
        private final int localPort;

        public Session getSession() {
            return this.session;
        }

        public int getLocalPort() {
            return this.localPort;
        }

        public SshPortForwardingSession(Session session, int localPort) {
            this.session = session;
            this.localPort = localPort;
        }
    }
}

