/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.dax.cluster;

import java.io.IOException;
import java.net.InetAddress;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import software.amazon.dax.Configuration;
import software.amazon.dax.DaxAsyncClient;
import software.amazon.dax.InternalConfiguration;
import software.amazon.dax.com.amazon.dax.client.HostPort;
import software.amazon.dax.exceptions.ClientCreationException;

public class Backend {
    private static final Log LOG = LogFactory.getLog(Backend.class);
    private final InetAddress addr;
    private final int port;
    private final InternalConfiguration internalConfiguration;
    private final Configuration configuration;
    private DaxAsyncClient client;
    private ScheduledFuture<?> healthChecker;
    private ScheduledExecutorService scheduledHealthCheckExecutorService;
    private final OnHealthCheckFailure onHealthCheckFailure;

    public Backend(InetAddress addr, int port, Configuration configuration, InternalConfiguration internalConfiguration, OnHealthCheckFailure onHealthCheckFailure) throws ClientCreationException {
        this(addr, port, configuration, internalConfiguration, onHealthCheckFailure, null, Backend.newClient(configuration, internalConfiguration, addr, port));
    }

    Backend(InetAddress addr, int port, Configuration configuration, InternalConfiguration internalConfiguration, OnHealthCheckFailure onHealthCheckFailure, ScheduledExecutorService scheduledExecutorService, DaxAsyncClient client) throws ClientCreationException {
        this.addr = addr;
        this.port = port;
        this.configuration = configuration;
        this.internalConfiguration = internalConfiguration;
        this.client = client;
        this.onHealthCheckFailure = onHealthCheckFailure;
        if (configuration.enableHealthCheck()) {
            this.initHealthChecks(scheduledExecutorService);
        }
    }

    private void initHealthChecks(ScheduledExecutorService executorService) {
        if (executorService != null) {
            this.scheduledHealthCheckExecutorService = executorService;
        } else {
            this.scheduledHealthCheckExecutorService = Executors.newScheduledThreadPool(0, r -> {
                Thread t = new Thread(r);
                t.setDaemon(true);
                t.setName("DaxBackend-" + t.getId());
                return t;
            });
            if (this.scheduledHealthCheckExecutorService instanceof ThreadPoolExecutor) {
                ThreadPoolExecutor tpe = (ThreadPoolExecutor)((Object)this.scheduledHealthCheckExecutorService);
                tpe.setKeepAliveTime(this.configuration.healthCheckIntervalMillis() * 2, TimeUnit.MILLISECONDS);
                tpe.allowCoreThreadTimeOut(true);
            }
            if (this.scheduledHealthCheckExecutorService instanceof ScheduledThreadPoolExecutor) {
                ScheduledThreadPoolExecutor scheduledThreadPoolExecutor = (ScheduledThreadPoolExecutor)this.scheduledHealthCheckExecutorService;
                scheduledThreadPoolExecutor.setRemoveOnCancelPolicy(true);
            }
        }
        this.healthChecker = this.scheduledHealthCheckExecutorService.scheduleWithFixedDelay(this::healthCheck, 0L, this.configuration.healthCheckIntervalMillis(), TimeUnit.MILLISECONDS);
    }

    public DaxAsyncClient getClient() {
        return this.client;
    }

    private static DaxAsyncClient newClient(Configuration configuration, InternalConfiguration internalConfiguration, InetAddress addr, int port) throws ClientCreationException {
        try {
            return new DaxAsyncClient((Configuration)configuration.copy(c -> c.url(HostPort.url(configuration.ssl(), addr.getHostAddress(), port))), internalConfiguration);
        }
        catch (IOException ex) {
            LOG.warn((Object)("DaxAsyncClient creation failed. Exception: " + ex.getMessage()));
            throw new ClientCreationException(ex.getMessage(), ex);
        }
    }

    void purgeClient() {
        try (DaxAsyncClient oldClient = this.client;){
            this.client = Backend.newClient(this.configuration, this.internalConfiguration, this.addr, this.port);
        }
    }

    boolean purgeClientWithRetries() {
        for (int purgeRetries = 0; purgeRetries <= 2; ++purgeRetries) {
            try {
                this.purgeClient();
                LOG.warn((Object)("Dax client is purged for the server node " + this.addr));
                return true;
            }
            catch (Exception ex) {
                LOG.warn((Object)ex.getMessage());
                continue;
            }
        }
        return false;
    }

    public void close() {
        try {
            this.client.close();
        }
        finally {
            if (this.configuration.enableHealthCheck()) {
                this.cleanupHealthChecks();
            }
        }
    }

    private void cleanupHealthChecks() {
        this.healthChecker.cancel(false);
        this.scheduledHealthCheckExecutorService.shutdown();
        try {
            if (!this.scheduledHealthCheckExecutorService.awaitTermination(1000L, TimeUnit.MILLISECONDS)) {
                this.scheduledHealthCheckExecutorService.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.scheduledHealthCheckExecutorService.shutdownNow();
        }
    }

    void healthCheck() {
        int hcAttempt;
        long delay = this.configuration.healthCheckRetryInitialDelayMillis();
        for (hcAttempt = 0; hcAttempt <= this.configuration.healthCheckMaxRetries(); ++hcAttempt) {
            try {
                this.client.endpoints().get(this.configuration.healthCheckTimeoutMillis(), TimeUnit.MILLISECONDS);
                break;
            }
            catch (Exception e) {
                LOG.debug((Object)("Health-check attempt " + (hcAttempt + 1) + " failed for the node " + this.addr), (Throwable)e);
                if (hcAttempt >= this.configuration.healthCheckMaxRetries()) continue;
                try {
                    Thread.sleep(delay);
                }
                catch (InterruptedException ie) {
                    LOG.debug((Object)("Health-check thread's sleep() is interrupted after attempt " + (hcAttempt + 1) + " for the node " + this.addr), (Throwable)ie);
                }
                delay *= 2L;
                continue;
            }
        }
        if (hcAttempt > this.configuration.healthCheckMaxRetries() && this.purgeClientWithRetries()) {
            this.onHealthCheckFailure.onHealthCheckFailure();
        }
    }

    @FunctionalInterface
    public static interface OnHealthCheckFailure {
        public void onHealthCheckFailure();
    }
}

