/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.xray.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A sampling rule that services use to decide whether to instrument a request. Rule fields can match properties of the
 * service, or properties of a request. The service can ignore rules that don't match its properties.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SamplingRule implements SdkPojo, Serializable, ToCopyableBuilder<SamplingRule.Builder, SamplingRule> {
    private static final SdkField<String> RULE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RuleName").getter(getter(SamplingRule::ruleName)).setter(setter(Builder::ruleName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleName").build()).build();

    private static final SdkField<String> RULE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RuleARN").getter(getter(SamplingRule::ruleARN)).setter(setter(Builder::ruleARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleARN").build()).build();

    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceARN").getter(getter(SamplingRule::resourceARN)).setter(setter(Builder::resourceARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceARN").build()).build();

    private static final SdkField<Integer> PRIORITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Priority").getter(getter(SamplingRule::priority)).setter(setter(Builder::priority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Priority").build()).build();

    private static final SdkField<Double> FIXED_RATE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("FixedRate").getter(getter(SamplingRule::fixedRate)).setter(setter(Builder::fixedRate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FixedRate").build()).build();

    private static final SdkField<Integer> RESERVOIR_SIZE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ReservoirSize").getter(getter(SamplingRule::reservoirSize)).setter(setter(Builder::reservoirSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReservoirSize").build()).build();

    private static final SdkField<String> SERVICE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServiceName").getter(getter(SamplingRule::serviceName)).setter(setter(Builder::serviceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceName").build()).build();

    private static final SdkField<String> SERVICE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServiceType").getter(getter(SamplingRule::serviceType)).setter(setter(Builder::serviceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceType").build()).build();

    private static final SdkField<String> HOST_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Host")
            .getter(getter(SamplingRule::host)).setter(setter(Builder::host))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Host").build()).build();

    private static final SdkField<String> HTTP_METHOD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HTTPMethod").getter(getter(SamplingRule::httpMethod)).setter(setter(Builder::httpMethod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HTTPMethod").build()).build();

    private static final SdkField<String> URL_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("URLPath").getter(getter(SamplingRule::urlPath)).setter(setter(Builder::urlPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("URLPath").build()).build();

    private static final SdkField<Integer> VERSION_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Version").getter(getter(SamplingRule::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<Map<String, String>> ATTRIBUTES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Attributes")
            .getter(getter(SamplingRule::attributes))
            .setter(setter(Builder::attributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Attributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<SamplingRateBoost> SAMPLING_RATE_BOOST_FIELD = SdkField
            .<SamplingRateBoost> builder(MarshallingType.SDK_POJO).memberName("SamplingRateBoost")
            .getter(getter(SamplingRule::samplingRateBoost)).setter(setter(Builder::samplingRateBoost))
            .constructor(SamplingRateBoost::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SamplingRateBoost").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RULE_NAME_FIELD,
            RULE_ARN_FIELD, RESOURCE_ARN_FIELD, PRIORITY_FIELD, FIXED_RATE_FIELD, RESERVOIR_SIZE_FIELD, SERVICE_NAME_FIELD,
            SERVICE_TYPE_FIELD, HOST_FIELD, HTTP_METHOD_FIELD, URL_PATH_FIELD, VERSION_FIELD, ATTRIBUTES_FIELD,
            SAMPLING_RATE_BOOST_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String ruleName;

    private final String ruleARN;

    private final String resourceARN;

    private final Integer priority;

    private final Double fixedRate;

    private final Integer reservoirSize;

    private final String serviceName;

    private final String serviceType;

    private final String host;

    private final String httpMethod;

    private final String urlPath;

    private final Integer version;

    private final Map<String, String> attributes;

    private final SamplingRateBoost samplingRateBoost;

    private SamplingRule(BuilderImpl builder) {
        this.ruleName = builder.ruleName;
        this.ruleARN = builder.ruleARN;
        this.resourceARN = builder.resourceARN;
        this.priority = builder.priority;
        this.fixedRate = builder.fixedRate;
        this.reservoirSize = builder.reservoirSize;
        this.serviceName = builder.serviceName;
        this.serviceType = builder.serviceType;
        this.host = builder.host;
        this.httpMethod = builder.httpMethod;
        this.urlPath = builder.urlPath;
        this.version = builder.version;
        this.attributes = builder.attributes;
        this.samplingRateBoost = builder.samplingRateBoost;
    }

    /**
     * <p>
     * The name of the sampling rule. Specify a rule by either name or ARN, but not both.
     * </p>
     * 
     * @return The name of the sampling rule. Specify a rule by either name or ARN, but not both.
     */
    public final String ruleName() {
        return ruleName;
    }

    /**
     * <p>
     * The ARN of the sampling rule. Specify a rule by either name or ARN, but not both.
     * </p>
     * 
     * @return The ARN of the sampling rule. Specify a rule by either name or ARN, but not both.
     */
    public final String ruleARN() {
        return ruleARN;
    }

    /**
     * <p>
     * Matches the ARN of the Amazon Web Services resource on which the service runs.
     * </p>
     * 
     * @return Matches the ARN of the Amazon Web Services resource on which the service runs.
     */
    public final String resourceARN() {
        return resourceARN;
    }

    /**
     * <p>
     * The priority of the sampling rule.
     * </p>
     * 
     * @return The priority of the sampling rule.
     */
    public final Integer priority() {
        return priority;
    }

    /**
     * <p>
     * The percentage of matching requests to instrument, after the reservoir is exhausted.
     * </p>
     * 
     * @return The percentage of matching requests to instrument, after the reservoir is exhausted.
     */
    public final Double fixedRate() {
        return fixedRate;
    }

    /**
     * <p>
     * A fixed number of matching requests to instrument per second, prior to applying the fixed rate. The reservoir is
     * not used directly by services, but applies to all services using the rule collectively.
     * </p>
     * 
     * @return A fixed number of matching requests to instrument per second, prior to applying the fixed rate. The
     *         reservoir is not used directly by services, but applies to all services using the rule collectively.
     */
    public final Integer reservoirSize() {
        return reservoirSize;
    }

    /**
     * <p>
     * Matches the <code>name</code> that the service uses to identify itself in segments.
     * </p>
     * 
     * @return Matches the <code>name</code> that the service uses to identify itself in segments.
     */
    public final String serviceName() {
        return serviceName;
    }

    /**
     * <p>
     * Matches the <code>origin</code> that the service uses to identify its type in segments.
     * </p>
     * 
     * @return Matches the <code>origin</code> that the service uses to identify its type in segments.
     */
    public final String serviceType() {
        return serviceType;
    }

    /**
     * <p>
     * Matches the hostname from a request URL.
     * </p>
     * 
     * @return Matches the hostname from a request URL.
     */
    public final String host() {
        return host;
    }

    /**
     * <p>
     * Matches the HTTP method of a request.
     * </p>
     * 
     * @return Matches the HTTP method of a request.
     */
    public final String httpMethod() {
        return httpMethod;
    }

    /**
     * <p>
     * Matches the path from a request URL.
     * </p>
     * 
     * @return Matches the path from a request URL.
     */
    public final String urlPath() {
        return urlPath;
    }

    /**
     * <p>
     * The version of the sampling rule format (<code>1</code>).
     * </p>
     * 
     * @return The version of the sampling rule format (<code>1</code>).
     */
    public final Integer version() {
        return version;
    }

    /**
     * For responses, this returns true if the service returned a value for the Attributes property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAttributes() {
        return attributes != null && !(attributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Matches attributes derived from the request.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttributes} method.
     * </p>
     * 
     * @return Matches attributes derived from the request.
     */
    public final Map<String, String> attributes() {
        return attributes;
    }

    /**
     * <p>
     * Specifies the multiplier applied to the base sampling rate. This boost allows you to temporarily increase
     * sampling without changing the rule's configuration.
     * </p>
     * 
     * @return Specifies the multiplier applied to the base sampling rate. This boost allows you to temporarily increase
     *         sampling without changing the rule's configuration.
     */
    public final SamplingRateBoost samplingRateBoost() {
        return samplingRateBoost;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ruleName());
        hashCode = 31 * hashCode + Objects.hashCode(ruleARN());
        hashCode = 31 * hashCode + Objects.hashCode(resourceARN());
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(fixedRate());
        hashCode = 31 * hashCode + Objects.hashCode(reservoirSize());
        hashCode = 31 * hashCode + Objects.hashCode(serviceName());
        hashCode = 31 * hashCode + Objects.hashCode(serviceType());
        hashCode = 31 * hashCode + Objects.hashCode(host());
        hashCode = 31 * hashCode + Objects.hashCode(httpMethod());
        hashCode = 31 * hashCode + Objects.hashCode(urlPath());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributes() ? attributes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(samplingRateBoost());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SamplingRule)) {
            return false;
        }
        SamplingRule other = (SamplingRule) obj;
        return Objects.equals(ruleName(), other.ruleName()) && Objects.equals(ruleARN(), other.ruleARN())
                && Objects.equals(resourceARN(), other.resourceARN()) && Objects.equals(priority(), other.priority())
                && Objects.equals(fixedRate(), other.fixedRate()) && Objects.equals(reservoirSize(), other.reservoirSize())
                && Objects.equals(serviceName(), other.serviceName()) && Objects.equals(serviceType(), other.serviceType())
                && Objects.equals(host(), other.host()) && Objects.equals(httpMethod(), other.httpMethod())
                && Objects.equals(urlPath(), other.urlPath()) && Objects.equals(version(), other.version())
                && hasAttributes() == other.hasAttributes() && Objects.equals(attributes(), other.attributes())
                && Objects.equals(samplingRateBoost(), other.samplingRateBoost());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SamplingRule").add("RuleName", ruleName()).add("RuleARN", ruleARN())
                .add("ResourceARN", resourceARN()).add("Priority", priority()).add("FixedRate", fixedRate())
                .add("ReservoirSize", reservoirSize()).add("ServiceName", serviceName()).add("ServiceType", serviceType())
                .add("Host", host()).add("HTTPMethod", httpMethod()).add("URLPath", urlPath()).add("Version", version())
                .add("Attributes", hasAttributes() ? attributes() : null).add("SamplingRateBoost", samplingRateBoost()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RuleName":
            return Optional.ofNullable(clazz.cast(ruleName()));
        case "RuleARN":
            return Optional.ofNullable(clazz.cast(ruleARN()));
        case "ResourceARN":
            return Optional.ofNullable(clazz.cast(resourceARN()));
        case "Priority":
            return Optional.ofNullable(clazz.cast(priority()));
        case "FixedRate":
            return Optional.ofNullable(clazz.cast(fixedRate()));
        case "ReservoirSize":
            return Optional.ofNullable(clazz.cast(reservoirSize()));
        case "ServiceName":
            return Optional.ofNullable(clazz.cast(serviceName()));
        case "ServiceType":
            return Optional.ofNullable(clazz.cast(serviceType()));
        case "Host":
            return Optional.ofNullable(clazz.cast(host()));
        case "HTTPMethod":
            return Optional.ofNullable(clazz.cast(httpMethod()));
        case "URLPath":
            return Optional.ofNullable(clazz.cast(urlPath()));
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        case "Attributes":
            return Optional.ofNullable(clazz.cast(attributes()));
        case "SamplingRateBoost":
            return Optional.ofNullable(clazz.cast(samplingRateBoost()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("RuleName", RULE_NAME_FIELD);
        map.put("RuleARN", RULE_ARN_FIELD);
        map.put("ResourceARN", RESOURCE_ARN_FIELD);
        map.put("Priority", PRIORITY_FIELD);
        map.put("FixedRate", FIXED_RATE_FIELD);
        map.put("ReservoirSize", RESERVOIR_SIZE_FIELD);
        map.put("ServiceName", SERVICE_NAME_FIELD);
        map.put("ServiceType", SERVICE_TYPE_FIELD);
        map.put("Host", HOST_FIELD);
        map.put("HTTPMethod", HTTP_METHOD_FIELD);
        map.put("URLPath", URL_PATH_FIELD);
        map.put("Version", VERSION_FIELD);
        map.put("Attributes", ATTRIBUTES_FIELD);
        map.put("SamplingRateBoost", SAMPLING_RATE_BOOST_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SamplingRule, T> g) {
        return obj -> g.apply((SamplingRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SamplingRule> {
        /**
         * <p>
         * The name of the sampling rule. Specify a rule by either name or ARN, but not both.
         * </p>
         * 
         * @param ruleName
         *        The name of the sampling rule. Specify a rule by either name or ARN, but not both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleName(String ruleName);

        /**
         * <p>
         * The ARN of the sampling rule. Specify a rule by either name or ARN, but not both.
         * </p>
         * 
         * @param ruleARN
         *        The ARN of the sampling rule. Specify a rule by either name or ARN, but not both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleARN(String ruleARN);

        /**
         * <p>
         * Matches the ARN of the Amazon Web Services resource on which the service runs.
         * </p>
         * 
         * @param resourceARN
         *        Matches the ARN of the Amazon Web Services resource on which the service runs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceARN(String resourceARN);

        /**
         * <p>
         * The priority of the sampling rule.
         * </p>
         * 
         * @param priority
         *        The priority of the sampling rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(Integer priority);

        /**
         * <p>
         * The percentage of matching requests to instrument, after the reservoir is exhausted.
         * </p>
         * 
         * @param fixedRate
         *        The percentage of matching requests to instrument, after the reservoir is exhausted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fixedRate(Double fixedRate);

        /**
         * <p>
         * A fixed number of matching requests to instrument per second, prior to applying the fixed rate. The reservoir
         * is not used directly by services, but applies to all services using the rule collectively.
         * </p>
         * 
         * @param reservoirSize
         *        A fixed number of matching requests to instrument per second, prior to applying the fixed rate. The
         *        reservoir is not used directly by services, but applies to all services using the rule collectively.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reservoirSize(Integer reservoirSize);

        /**
         * <p>
         * Matches the <code>name</code> that the service uses to identify itself in segments.
         * </p>
         * 
         * @param serviceName
         *        Matches the <code>name</code> that the service uses to identify itself in segments.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceName(String serviceName);

        /**
         * <p>
         * Matches the <code>origin</code> that the service uses to identify its type in segments.
         * </p>
         * 
         * @param serviceType
         *        Matches the <code>origin</code> that the service uses to identify its type in segments.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceType(String serviceType);

        /**
         * <p>
         * Matches the hostname from a request URL.
         * </p>
         * 
         * @param host
         *        Matches the hostname from a request URL.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder host(String host);

        /**
         * <p>
         * Matches the HTTP method of a request.
         * </p>
         * 
         * @param httpMethod
         *        Matches the HTTP method of a request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder httpMethod(String httpMethod);

        /**
         * <p>
         * Matches the path from a request URL.
         * </p>
         * 
         * @param urlPath
         *        Matches the path from a request URL.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder urlPath(String urlPath);

        /**
         * <p>
         * The version of the sampling rule format (<code>1</code>).
         * </p>
         * 
         * @param version
         *        The version of the sampling rule format (<code>1</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(Integer version);

        /**
         * <p>
         * Matches attributes derived from the request.
         * </p>
         * 
         * @param attributes
         *        Matches attributes derived from the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Map<String, String> attributes);

        /**
         * <p>
         * Specifies the multiplier applied to the base sampling rate. This boost allows you to temporarily increase
         * sampling without changing the rule's configuration.
         * </p>
         * 
         * @param samplingRateBoost
         *        Specifies the multiplier applied to the base sampling rate. This boost allows you to temporarily
         *        increase sampling without changing the rule's configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder samplingRateBoost(SamplingRateBoost samplingRateBoost);

        /**
         * <p>
         * Specifies the multiplier applied to the base sampling rate. This boost allows you to temporarily increase
         * sampling without changing the rule's configuration.
         * </p>
         * This is a convenience method that creates an instance of the {@link SamplingRateBoost.Builder} avoiding the
         * need to create one manually via {@link SamplingRateBoost#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SamplingRateBoost.Builder#build()} is called immediately and its
         * result is passed to {@link #samplingRateBoost(SamplingRateBoost)}.
         * 
         * @param samplingRateBoost
         *        a consumer that will call methods on {@link SamplingRateBoost.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #samplingRateBoost(SamplingRateBoost)
         */
        default Builder samplingRateBoost(Consumer<SamplingRateBoost.Builder> samplingRateBoost) {
            return samplingRateBoost(SamplingRateBoost.builder().applyMutation(samplingRateBoost).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String ruleName;

        private String ruleARN;

        private String resourceARN;

        private Integer priority;

        private Double fixedRate;

        private Integer reservoirSize;

        private String serviceName;

        private String serviceType;

        private String host;

        private String httpMethod;

        private String urlPath;

        private Integer version;

        private Map<String, String> attributes = DefaultSdkAutoConstructMap.getInstance();

        private SamplingRateBoost samplingRateBoost;

        private BuilderImpl() {
        }

        private BuilderImpl(SamplingRule model) {
            ruleName(model.ruleName);
            ruleARN(model.ruleARN);
            resourceARN(model.resourceARN);
            priority(model.priority);
            fixedRate(model.fixedRate);
            reservoirSize(model.reservoirSize);
            serviceName(model.serviceName);
            serviceType(model.serviceType);
            host(model.host);
            httpMethod(model.httpMethod);
            urlPath(model.urlPath);
            version(model.version);
            attributes(model.attributes);
            samplingRateBoost(model.samplingRateBoost);
        }

        public final String getRuleName() {
            return ruleName;
        }

        public final void setRuleName(String ruleName) {
            this.ruleName = ruleName;
        }

        @Override
        public final Builder ruleName(String ruleName) {
            this.ruleName = ruleName;
            return this;
        }

        public final String getRuleARN() {
            return ruleARN;
        }

        public final void setRuleARN(String ruleARN) {
            this.ruleARN = ruleARN;
        }

        @Override
        public final Builder ruleARN(String ruleARN) {
            this.ruleARN = ruleARN;
            return this;
        }

        public final String getResourceARN() {
            return resourceARN;
        }

        public final void setResourceARN(String resourceARN) {
            this.resourceARN = resourceARN;
        }

        @Override
        public final Builder resourceARN(String resourceARN) {
            this.resourceARN = resourceARN;
            return this;
        }

        public final Integer getPriority() {
            return priority;
        }

        public final void setPriority(Integer priority) {
            this.priority = priority;
        }

        @Override
        public final Builder priority(Integer priority) {
            this.priority = priority;
            return this;
        }

        public final Double getFixedRate() {
            return fixedRate;
        }

        public final void setFixedRate(Double fixedRate) {
            this.fixedRate = fixedRate;
        }

        @Override
        public final Builder fixedRate(Double fixedRate) {
            this.fixedRate = fixedRate;
            return this;
        }

        public final Integer getReservoirSize() {
            return reservoirSize;
        }

        public final void setReservoirSize(Integer reservoirSize) {
            this.reservoirSize = reservoirSize;
        }

        @Override
        public final Builder reservoirSize(Integer reservoirSize) {
            this.reservoirSize = reservoirSize;
            return this;
        }

        public final String getServiceName() {
            return serviceName;
        }

        public final void setServiceName(String serviceName) {
            this.serviceName = serviceName;
        }

        @Override
        public final Builder serviceName(String serviceName) {
            this.serviceName = serviceName;
            return this;
        }

        public final String getServiceType() {
            return serviceType;
        }

        public final void setServiceType(String serviceType) {
            this.serviceType = serviceType;
        }

        @Override
        public final Builder serviceType(String serviceType) {
            this.serviceType = serviceType;
            return this;
        }

        public final String getHost() {
            return host;
        }

        public final void setHost(String host) {
            this.host = host;
        }

        @Override
        public final Builder host(String host) {
            this.host = host;
            return this;
        }

        public final String getHttpMethod() {
            return httpMethod;
        }

        public final void setHttpMethod(String httpMethod) {
            this.httpMethod = httpMethod;
        }

        @Override
        public final Builder httpMethod(String httpMethod) {
            this.httpMethod = httpMethod;
            return this;
        }

        public final String getUrlPath() {
            return urlPath;
        }

        public final void setUrlPath(String urlPath) {
            this.urlPath = urlPath;
        }

        @Override
        public final Builder urlPath(String urlPath) {
            this.urlPath = urlPath;
            return this;
        }

        public final Integer getVersion() {
            return version;
        }

        public final void setVersion(Integer version) {
            this.version = version;
        }

        @Override
        public final Builder version(Integer version) {
            this.version = version;
            return this;
        }

        public final Map<String, String> getAttributes() {
            if (attributes instanceof SdkAutoConstructMap) {
                return null;
            }
            return attributes;
        }

        public final void setAttributes(Map<String, String> attributes) {
            this.attributes = AttributeMapCopier.copy(attributes);
        }

        @Override
        public final Builder attributes(Map<String, String> attributes) {
            this.attributes = AttributeMapCopier.copy(attributes);
            return this;
        }

        public final SamplingRateBoost.Builder getSamplingRateBoost() {
            return samplingRateBoost != null ? samplingRateBoost.toBuilder() : null;
        }

        public final void setSamplingRateBoost(SamplingRateBoost.BuilderImpl samplingRateBoost) {
            this.samplingRateBoost = samplingRateBoost != null ? samplingRateBoost.build() : null;
        }

        @Override
        public final Builder samplingRateBoost(SamplingRateBoost samplingRateBoost) {
            this.samplingRateBoost = samplingRateBoost;
            return this;
        }

        @Override
        public SamplingRule build() {
            return new SamplingRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
