/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.workspacesinstances;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.workspacesinstances.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.workspacesinstances.internal.WorkspacesInstancesServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.workspacesinstances.model.AccessDeniedException;
import software.amazon.awssdk.services.workspacesinstances.model.AssociateVolumeRequest;
import software.amazon.awssdk.services.workspacesinstances.model.AssociateVolumeResponse;
import software.amazon.awssdk.services.workspacesinstances.model.ConflictException;
import software.amazon.awssdk.services.workspacesinstances.model.CreateVolumeRequest;
import software.amazon.awssdk.services.workspacesinstances.model.CreateVolumeResponse;
import software.amazon.awssdk.services.workspacesinstances.model.CreateWorkspaceInstanceRequest;
import software.amazon.awssdk.services.workspacesinstances.model.CreateWorkspaceInstanceResponse;
import software.amazon.awssdk.services.workspacesinstances.model.DeleteVolumeRequest;
import software.amazon.awssdk.services.workspacesinstances.model.DeleteVolumeResponse;
import software.amazon.awssdk.services.workspacesinstances.model.DeleteWorkspaceInstanceRequest;
import software.amazon.awssdk.services.workspacesinstances.model.DeleteWorkspaceInstanceResponse;
import software.amazon.awssdk.services.workspacesinstances.model.DisassociateVolumeRequest;
import software.amazon.awssdk.services.workspacesinstances.model.DisassociateVolumeResponse;
import software.amazon.awssdk.services.workspacesinstances.model.GetWorkspaceInstanceRequest;
import software.amazon.awssdk.services.workspacesinstances.model.GetWorkspaceInstanceResponse;
import software.amazon.awssdk.services.workspacesinstances.model.InternalServerException;
import software.amazon.awssdk.services.workspacesinstances.model.ListInstanceTypesRequest;
import software.amazon.awssdk.services.workspacesinstances.model.ListInstanceTypesResponse;
import software.amazon.awssdk.services.workspacesinstances.model.ListRegionsRequest;
import software.amazon.awssdk.services.workspacesinstances.model.ListRegionsResponse;
import software.amazon.awssdk.services.workspacesinstances.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.workspacesinstances.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.workspacesinstances.model.ListWorkspaceInstancesRequest;
import software.amazon.awssdk.services.workspacesinstances.model.ListWorkspaceInstancesResponse;
import software.amazon.awssdk.services.workspacesinstances.model.ResourceNotFoundException;
import software.amazon.awssdk.services.workspacesinstances.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.workspacesinstances.model.TagResourceRequest;
import software.amazon.awssdk.services.workspacesinstances.model.TagResourceResponse;
import software.amazon.awssdk.services.workspacesinstances.model.ThrottlingException;
import software.amazon.awssdk.services.workspacesinstances.model.UntagResourceRequest;
import software.amazon.awssdk.services.workspacesinstances.model.UntagResourceResponse;
import software.amazon.awssdk.services.workspacesinstances.model.ValidationException;
import software.amazon.awssdk.services.workspacesinstances.model.WorkspacesInstancesException;
import software.amazon.awssdk.services.workspacesinstances.transform.AssociateVolumeRequestMarshaller;
import software.amazon.awssdk.services.workspacesinstances.transform.CreateVolumeRequestMarshaller;
import software.amazon.awssdk.services.workspacesinstances.transform.CreateWorkspaceInstanceRequestMarshaller;
import software.amazon.awssdk.services.workspacesinstances.transform.DeleteVolumeRequestMarshaller;
import software.amazon.awssdk.services.workspacesinstances.transform.DeleteWorkspaceInstanceRequestMarshaller;
import software.amazon.awssdk.services.workspacesinstances.transform.DisassociateVolumeRequestMarshaller;
import software.amazon.awssdk.services.workspacesinstances.transform.GetWorkspaceInstanceRequestMarshaller;
import software.amazon.awssdk.services.workspacesinstances.transform.ListInstanceTypesRequestMarshaller;
import software.amazon.awssdk.services.workspacesinstances.transform.ListRegionsRequestMarshaller;
import software.amazon.awssdk.services.workspacesinstances.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.workspacesinstances.transform.ListWorkspaceInstancesRequestMarshaller;
import software.amazon.awssdk.services.workspacesinstances.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.workspacesinstances.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link WorkspacesInstancesClient}.
 *
 * @see WorkspacesInstancesClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultWorkspacesInstancesClient implements WorkspacesInstancesClient {
    private static final Logger log = Logger.loggerFor(DefaultWorkspacesInstancesClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultWorkspacesInstancesClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "Workspaces_Instances" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Attaches a volume to a WorkSpace Instance.
     * </p>
     *
     * @param associateVolumeRequest
     *        Specifies volume attachment parameters.
     * @return Result of the AssociateVolume operation returned by the service.
     * @throws ValidationException
     *         Indicates invalid input parameters in the request.
     * @throws AccessDeniedException
     *         Indicates insufficient permissions to perform the requested action.
     * @throws ThrottlingException
     *         Indicates the request rate has exceeded limits.
     * @throws ResourceNotFoundException
     *         Indicates the requested resource could not be found.
     * @throws InternalServerException
     *         Indicates an unexpected server-side error occurred.
     * @throws ConflictException
     *         Signals a conflict with the current state of the resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkspacesInstancesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkspacesInstancesClient.AssociateVolume
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-instances-2022-07-26/AssociateVolume"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateVolumeResponse associateVolume(AssociateVolumeRequest associateVolumeRequest) throws ValidationException,
            AccessDeniedException, ThrottlingException, ResourceNotFoundException, InternalServerException, ConflictException,
            AwsServiceException, SdkClientException, WorkspacesInstancesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateVolumeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AssociateVolumeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateVolumeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateVolumeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Workspaces Instances");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateVolume");

            return clientHandler.execute(new ClientExecutionParams<AssociateVolumeRequest, AssociateVolumeResponse>()
                    .withOperationName("AssociateVolume").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(associateVolumeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AssociateVolumeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new volume for WorkSpace Instances.
     * </p>
     *
     * @param createVolumeRequest
     *        Specifies volume creation parameters.
     * @return Result of the CreateVolume operation returned by the service.
     * @throws ValidationException
     *         Indicates invalid input parameters in the request.
     * @throws AccessDeniedException
     *         Indicates insufficient permissions to perform the requested action.
     * @throws ThrottlingException
     *         Indicates the request rate has exceeded limits.
     * @throws InternalServerException
     *         Indicates an unexpected server-side error occurred.
     * @throws ServiceQuotaExceededException
     *         Indicates that a service quota has been exceeded.
     * @throws ConflictException
     *         Signals a conflict with the current state of the resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkspacesInstancesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkspacesInstancesClient.CreateVolume
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-instances-2022-07-26/CreateVolume"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateVolumeResponse createVolume(CreateVolumeRequest createVolumeRequest) throws ValidationException,
            AccessDeniedException, ThrottlingException, InternalServerException, ServiceQuotaExceededException,
            ConflictException, AwsServiceException, SdkClientException, WorkspacesInstancesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateVolumeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateVolumeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createVolumeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVolumeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Workspaces Instances");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVolume");

            return clientHandler.execute(new ClientExecutionParams<CreateVolumeRequest, CreateVolumeResponse>()
                    .withOperationName("CreateVolume").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createVolumeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateVolumeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Launches a new WorkSpace Instance with specified configuration parameters, enabling programmatic workspace
     * deployment.
     * </p>
     *
     * @param createWorkspaceInstanceRequest
     *        Defines the configuration parameters for creating a new WorkSpaces Instance.
     * @return Result of the CreateWorkspaceInstance operation returned by the service.
     * @throws ValidationException
     *         Indicates invalid input parameters in the request.
     * @throws AccessDeniedException
     *         Indicates insufficient permissions to perform the requested action.
     * @throws ThrottlingException
     *         Indicates the request rate has exceeded limits.
     * @throws InternalServerException
     *         Indicates an unexpected server-side error occurred.
     * @throws ServiceQuotaExceededException
     *         Indicates that a service quota has been exceeded.
     * @throws ConflictException
     *         Signals a conflict with the current state of the resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkspacesInstancesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkspacesInstancesClient.CreateWorkspaceInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-instances-2022-07-26/CreateWorkspaceInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateWorkspaceInstanceResponse createWorkspaceInstance(CreateWorkspaceInstanceRequest createWorkspaceInstanceRequest)
            throws ValidationException, AccessDeniedException, ThrottlingException, InternalServerException,
            ServiceQuotaExceededException, ConflictException, AwsServiceException, SdkClientException,
            WorkspacesInstancesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateWorkspaceInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateWorkspaceInstanceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createWorkspaceInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWorkspaceInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Workspaces Instances");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorkspaceInstance");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateWorkspaceInstanceRequest, CreateWorkspaceInstanceResponse>()
                            .withOperationName("CreateWorkspaceInstance").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createWorkspaceInstanceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateWorkspaceInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a specified volume.
     * </p>
     *
     * @param deleteVolumeRequest
     *        Specifies the volume to delete.
     * @return Result of the DeleteVolume operation returned by the service.
     * @throws ValidationException
     *         Indicates invalid input parameters in the request.
     * @throws AccessDeniedException
     *         Indicates insufficient permissions to perform the requested action.
     * @throws ThrottlingException
     *         Indicates the request rate has exceeded limits.
     * @throws ResourceNotFoundException
     *         Indicates the requested resource could not be found.
     * @throws InternalServerException
     *         Indicates an unexpected server-side error occurred.
     * @throws ConflictException
     *         Signals a conflict with the current state of the resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkspacesInstancesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkspacesInstancesClient.DeleteVolume
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-instances-2022-07-26/DeleteVolume"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteVolumeResponse deleteVolume(DeleteVolumeRequest deleteVolumeRequest) throws ValidationException,
            AccessDeniedException, ThrottlingException, ResourceNotFoundException, InternalServerException, ConflictException,
            AwsServiceException, SdkClientException, WorkspacesInstancesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteVolumeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteVolumeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVolumeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVolumeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Workspaces Instances");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVolume");

            return clientHandler.execute(new ClientExecutionParams<DeleteVolumeRequest, DeleteVolumeResponse>()
                    .withOperationName("DeleteVolume").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteVolumeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteVolumeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified WorkSpace
     * </p>
     *
     * @param deleteWorkspaceInstanceRequest
     *        The WorkSpace to delete
     * @return Result of the DeleteWorkspaceInstance operation returned by the service.
     * @throws ValidationException
     *         Indicates invalid input parameters in the request.
     * @throws AccessDeniedException
     *         Indicates insufficient permissions to perform the requested action.
     * @throws ThrottlingException
     *         Indicates the request rate has exceeded limits.
     * @throws ResourceNotFoundException
     *         Indicates the requested resource could not be found.
     * @throws InternalServerException
     *         Indicates an unexpected server-side error occurred.
     * @throws ConflictException
     *         Signals a conflict with the current state of the resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkspacesInstancesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkspacesInstancesClient.DeleteWorkspaceInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-instances-2022-07-26/DeleteWorkspaceInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteWorkspaceInstanceResponse deleteWorkspaceInstance(DeleteWorkspaceInstanceRequest deleteWorkspaceInstanceRequest)
            throws ValidationException, AccessDeniedException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, ConflictException, AwsServiceException, SdkClientException, WorkspacesInstancesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteWorkspaceInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteWorkspaceInstanceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteWorkspaceInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWorkspaceInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Workspaces Instances");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorkspaceInstance");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteWorkspaceInstanceRequest, DeleteWorkspaceInstanceResponse>()
                            .withOperationName("DeleteWorkspaceInstance").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteWorkspaceInstanceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteWorkspaceInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Detaches a volume from a WorkSpace Instance.
     * </p>
     *
     * @param disassociateVolumeRequest
     *        Specifies volume detachment parameters.
     * @return Result of the DisassociateVolume operation returned by the service.
     * @throws ValidationException
     *         Indicates invalid input parameters in the request.
     * @throws AccessDeniedException
     *         Indicates insufficient permissions to perform the requested action.
     * @throws ThrottlingException
     *         Indicates the request rate has exceeded limits.
     * @throws ResourceNotFoundException
     *         Indicates the requested resource could not be found.
     * @throws InternalServerException
     *         Indicates an unexpected server-side error occurred.
     * @throws ConflictException
     *         Signals a conflict with the current state of the resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkspacesInstancesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkspacesInstancesClient.DisassociateVolume
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-instances-2022-07-26/DisassociateVolume"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateVolumeResponse disassociateVolume(DisassociateVolumeRequest disassociateVolumeRequest)
            throws ValidationException, AccessDeniedException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, ConflictException, AwsServiceException, SdkClientException, WorkspacesInstancesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateVolumeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateVolumeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateVolumeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateVolumeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Workspaces Instances");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateVolume");

            return clientHandler.execute(new ClientExecutionParams<DisassociateVolumeRequest, DisassociateVolumeResponse>()
                    .withOperationName("DisassociateVolume").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(disassociateVolumeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisassociateVolumeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves detailed information about a specific WorkSpace Instance.
     * </p>
     *
     * @param getWorkspaceInstanceRequest
     *        Identifies the WorkSpaces Instance to retrieve detailed information for.
     * @return Result of the GetWorkspaceInstance operation returned by the service.
     * @throws ValidationException
     *         Indicates invalid input parameters in the request.
     * @throws AccessDeniedException
     *         Indicates insufficient permissions to perform the requested action.
     * @throws ThrottlingException
     *         Indicates the request rate has exceeded limits.
     * @throws ResourceNotFoundException
     *         Indicates the requested resource could not be found.
     * @throws InternalServerException
     *         Indicates an unexpected server-side error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkspacesInstancesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkspacesInstancesClient.GetWorkspaceInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-instances-2022-07-26/GetWorkspaceInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetWorkspaceInstanceResponse getWorkspaceInstance(GetWorkspaceInstanceRequest getWorkspaceInstanceRequest)
            throws ValidationException, AccessDeniedException, ThrottlingException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, WorkspacesInstancesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetWorkspaceInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetWorkspaceInstanceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getWorkspaceInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWorkspaceInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Workspaces Instances");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWorkspaceInstance");

            return clientHandler.execute(new ClientExecutionParams<GetWorkspaceInstanceRequest, GetWorkspaceInstanceResponse>()
                    .withOperationName("GetWorkspaceInstance").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getWorkspaceInstanceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetWorkspaceInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of instance types supported by Amazon WorkSpaces Instances, enabling precise workspace
     * infrastructure configuration.
     * </p>
     *
     * @param listInstanceTypesRequest
     *        Defines input parameters for retrieving supported WorkSpaces Instances instance types.
     * @return Result of the ListInstanceTypes operation returned by the service.
     * @throws ValidationException
     *         Indicates invalid input parameters in the request.
     * @throws AccessDeniedException
     *         Indicates insufficient permissions to perform the requested action.
     * @throws ThrottlingException
     *         Indicates the request rate has exceeded limits.
     * @throws InternalServerException
     *         Indicates an unexpected server-side error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkspacesInstancesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkspacesInstancesClient.ListInstanceTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-instances-2022-07-26/ListInstanceTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListInstanceTypesResponse listInstanceTypes(ListInstanceTypesRequest listInstanceTypesRequest)
            throws ValidationException, AccessDeniedException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, WorkspacesInstancesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInstanceTypesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListInstanceTypesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInstanceTypesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInstanceTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Workspaces Instances");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInstanceTypes");

            return clientHandler.execute(new ClientExecutionParams<ListInstanceTypesRequest, ListInstanceTypesResponse>()
                    .withOperationName("ListInstanceTypes").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listInstanceTypesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListInstanceTypesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of AWS regions supported by Amazon WorkSpaces Instances, enabling region discovery for workspace
     * deployments.
     * </p>
     *
     * @param listRegionsRequest
     *        Defines input parameters for retrieving supported WorkSpaces Instances regions.
     * @return Result of the ListRegions operation returned by the service.
     * @throws ValidationException
     *         Indicates invalid input parameters in the request.
     * @throws AccessDeniedException
     *         Indicates insufficient permissions to perform the requested action.
     * @throws ThrottlingException
     *         Indicates the request rate has exceeded limits.
     * @throws InternalServerException
     *         Indicates an unexpected server-side error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkspacesInstancesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkspacesInstancesClient.ListRegions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-instances-2022-07-26/ListRegions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRegionsResponse listRegions(ListRegionsRequest listRegionsRequest) throws ValidationException,
            AccessDeniedException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            WorkspacesInstancesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRegionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListRegionsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRegionsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRegionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Workspaces Instances");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRegions");

            return clientHandler.execute(new ClientExecutionParams<ListRegionsRequest, ListRegionsResponse>()
                    .withOperationName("ListRegions").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listRegionsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListRegionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves tags for a WorkSpace Instance.
     * </p>
     *
     * @param listTagsForResourceRequest
     *        Specifies the WorkSpace Instance to retrieve tags for.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         Indicates invalid input parameters in the request.
     * @throws AccessDeniedException
     *         Indicates insufficient permissions to perform the requested action.
     * @throws ResourceNotFoundException
     *         Indicates the requested resource could not be found.
     * @throws ThrottlingException
     *         Indicates the request rate has exceeded limits.
     * @throws InternalServerException
     *         Indicates an unexpected server-side error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkspacesInstancesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkspacesInstancesClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-instances-2022-07-26/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, AccessDeniedException, ResourceNotFoundException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, WorkspacesInstancesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Workspaces Instances");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a collection of WorkSpaces Instances based on specified filters.
     * </p>
     *
     * @param listWorkspaceInstancesRequest
     *        Defines filters and pagination parameters for retrieving WorkSpaces Instances.
     * @return Result of the ListWorkspaceInstances operation returned by the service.
     * @throws ValidationException
     *         Indicates invalid input parameters in the request.
     * @throws AccessDeniedException
     *         Indicates insufficient permissions to perform the requested action.
     * @throws ThrottlingException
     *         Indicates the request rate has exceeded limits.
     * @throws InternalServerException
     *         Indicates an unexpected server-side error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkspacesInstancesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkspacesInstancesClient.ListWorkspaceInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-instances-2022-07-26/ListWorkspaceInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListWorkspaceInstancesResponse listWorkspaceInstances(ListWorkspaceInstancesRequest listWorkspaceInstancesRequest)
            throws ValidationException, AccessDeniedException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, WorkspacesInstancesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListWorkspaceInstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListWorkspaceInstancesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWorkspaceInstancesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkspaceInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Workspaces Instances");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkspaceInstances");

            return clientHandler
                    .execute(new ClientExecutionParams<ListWorkspaceInstancesRequest, ListWorkspaceInstancesResponse>()
                            .withOperationName("ListWorkspaceInstances").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listWorkspaceInstancesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListWorkspaceInstancesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds tags to a WorkSpace Instance.
     * </p>
     *
     * @param tagResourceRequest
     *        Specifies tags to add to a WorkSpace Instance.
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         Indicates invalid input parameters in the request.
     * @throws AccessDeniedException
     *         Indicates insufficient permissions to perform the requested action.
     * @throws ResourceNotFoundException
     *         Indicates the requested resource could not be found.
     * @throws ThrottlingException
     *         Indicates the request rate has exceeded limits.
     * @throws InternalServerException
     *         Indicates an unexpected server-side error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkspacesInstancesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkspacesInstancesClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-instances-2022-07-26/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, WorkspacesInstancesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Workspaces Instances");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from a WorkSpace Instance.
     * </p>
     *
     * @param untagResourceRequest
     *        Specifies tags to remove from a WorkSpace Instance.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         Indicates invalid input parameters in the request.
     * @throws AccessDeniedException
     *         Indicates insufficient permissions to perform the requested action.
     * @throws ResourceNotFoundException
     *         Indicates the requested resource could not be found.
     * @throws ThrottlingException
     *         Indicates the request rate has exceeded limits.
     * @throws InternalServerException
     *         Indicates an unexpected server-side error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkspacesInstancesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkspacesInstancesClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-instances-2022-07-26/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, WorkspacesInstancesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Workspaces Instances");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        WorkspacesInstancesServiceClientConfigurationBuilder serviceConfigBuilder = new WorkspacesInstancesServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(WorkspacesInstancesException::builder).protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0");
    }

    @Override
    public final WorkspacesInstancesServiceClientConfiguration serviceClientConfiguration() {
        return new WorkspacesInstancesServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
