/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.workdocs;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.workdocs.model.AbortDocumentVersionUploadRequest;
import software.amazon.awssdk.services.workdocs.model.AbortDocumentVersionUploadResponse;
import software.amazon.awssdk.services.workdocs.model.ActivateUserRequest;
import software.amazon.awssdk.services.workdocs.model.ActivateUserResponse;
import software.amazon.awssdk.services.workdocs.model.AddResourcePermissionsRequest;
import software.amazon.awssdk.services.workdocs.model.AddResourcePermissionsResponse;
import software.amazon.awssdk.services.workdocs.model.ConcurrentModificationException;
import software.amazon.awssdk.services.workdocs.model.ConflictingOperationException;
import software.amazon.awssdk.services.workdocs.model.CreateCommentRequest;
import software.amazon.awssdk.services.workdocs.model.CreateCommentResponse;
import software.amazon.awssdk.services.workdocs.model.CreateCustomMetadataRequest;
import software.amazon.awssdk.services.workdocs.model.CreateCustomMetadataResponse;
import software.amazon.awssdk.services.workdocs.model.CreateFolderRequest;
import software.amazon.awssdk.services.workdocs.model.CreateFolderResponse;
import software.amazon.awssdk.services.workdocs.model.CreateLabelsRequest;
import software.amazon.awssdk.services.workdocs.model.CreateLabelsResponse;
import software.amazon.awssdk.services.workdocs.model.CreateNotificationSubscriptionRequest;
import software.amazon.awssdk.services.workdocs.model.CreateNotificationSubscriptionResponse;
import software.amazon.awssdk.services.workdocs.model.CreateUserRequest;
import software.amazon.awssdk.services.workdocs.model.CreateUserResponse;
import software.amazon.awssdk.services.workdocs.model.CustomMetadataLimitExceededException;
import software.amazon.awssdk.services.workdocs.model.DeactivateUserRequest;
import software.amazon.awssdk.services.workdocs.model.DeactivateUserResponse;
import software.amazon.awssdk.services.workdocs.model.DeactivatingLastSystemUserException;
import software.amazon.awssdk.services.workdocs.model.DeleteCommentRequest;
import software.amazon.awssdk.services.workdocs.model.DeleteCommentResponse;
import software.amazon.awssdk.services.workdocs.model.DeleteCustomMetadataRequest;
import software.amazon.awssdk.services.workdocs.model.DeleteCustomMetadataResponse;
import software.amazon.awssdk.services.workdocs.model.DeleteDocumentRequest;
import software.amazon.awssdk.services.workdocs.model.DeleteDocumentResponse;
import software.amazon.awssdk.services.workdocs.model.DeleteFolderContentsRequest;
import software.amazon.awssdk.services.workdocs.model.DeleteFolderContentsResponse;
import software.amazon.awssdk.services.workdocs.model.DeleteFolderRequest;
import software.amazon.awssdk.services.workdocs.model.DeleteFolderResponse;
import software.amazon.awssdk.services.workdocs.model.DeleteLabelsRequest;
import software.amazon.awssdk.services.workdocs.model.DeleteLabelsResponse;
import software.amazon.awssdk.services.workdocs.model.DeleteNotificationSubscriptionRequest;
import software.amazon.awssdk.services.workdocs.model.DeleteNotificationSubscriptionResponse;
import software.amazon.awssdk.services.workdocs.model.DeleteUserRequest;
import software.amazon.awssdk.services.workdocs.model.DeleteUserResponse;
import software.amazon.awssdk.services.workdocs.model.DescribeActivitiesRequest;
import software.amazon.awssdk.services.workdocs.model.DescribeActivitiesResponse;
import software.amazon.awssdk.services.workdocs.model.DescribeCommentsRequest;
import software.amazon.awssdk.services.workdocs.model.DescribeCommentsResponse;
import software.amazon.awssdk.services.workdocs.model.DescribeDocumentVersionsRequest;
import software.amazon.awssdk.services.workdocs.model.DescribeDocumentVersionsResponse;
import software.amazon.awssdk.services.workdocs.model.DescribeFolderContentsRequest;
import software.amazon.awssdk.services.workdocs.model.DescribeFolderContentsResponse;
import software.amazon.awssdk.services.workdocs.model.DescribeGroupsRequest;
import software.amazon.awssdk.services.workdocs.model.DescribeGroupsResponse;
import software.amazon.awssdk.services.workdocs.model.DescribeNotificationSubscriptionsRequest;
import software.amazon.awssdk.services.workdocs.model.DescribeNotificationSubscriptionsResponse;
import software.amazon.awssdk.services.workdocs.model.DescribeResourcePermissionsRequest;
import software.amazon.awssdk.services.workdocs.model.DescribeResourcePermissionsResponse;
import software.amazon.awssdk.services.workdocs.model.DescribeRootFoldersRequest;
import software.amazon.awssdk.services.workdocs.model.DescribeRootFoldersResponse;
import software.amazon.awssdk.services.workdocs.model.DescribeUsersRequest;
import software.amazon.awssdk.services.workdocs.model.DescribeUsersResponse;
import software.amazon.awssdk.services.workdocs.model.DocumentLockedForCommentsException;
import software.amazon.awssdk.services.workdocs.model.DraftUploadOutOfSyncException;
import software.amazon.awssdk.services.workdocs.model.EntityAlreadyExistsException;
import software.amazon.awssdk.services.workdocs.model.EntityNotExistsException;
import software.amazon.awssdk.services.workdocs.model.FailedDependencyException;
import software.amazon.awssdk.services.workdocs.model.GetCurrentUserRequest;
import software.amazon.awssdk.services.workdocs.model.GetCurrentUserResponse;
import software.amazon.awssdk.services.workdocs.model.GetDocumentPathRequest;
import software.amazon.awssdk.services.workdocs.model.GetDocumentPathResponse;
import software.amazon.awssdk.services.workdocs.model.GetDocumentRequest;
import software.amazon.awssdk.services.workdocs.model.GetDocumentResponse;
import software.amazon.awssdk.services.workdocs.model.GetDocumentVersionRequest;
import software.amazon.awssdk.services.workdocs.model.GetDocumentVersionResponse;
import software.amazon.awssdk.services.workdocs.model.GetFolderPathRequest;
import software.amazon.awssdk.services.workdocs.model.GetFolderPathResponse;
import software.amazon.awssdk.services.workdocs.model.GetFolderRequest;
import software.amazon.awssdk.services.workdocs.model.GetFolderResponse;
import software.amazon.awssdk.services.workdocs.model.GetResourcesRequest;
import software.amazon.awssdk.services.workdocs.model.GetResourcesResponse;
import software.amazon.awssdk.services.workdocs.model.IllegalUserStateException;
import software.amazon.awssdk.services.workdocs.model.InitiateDocumentVersionUploadRequest;
import software.amazon.awssdk.services.workdocs.model.InitiateDocumentVersionUploadResponse;
import software.amazon.awssdk.services.workdocs.model.InvalidArgumentException;
import software.amazon.awssdk.services.workdocs.model.InvalidCommentOperationException;
import software.amazon.awssdk.services.workdocs.model.InvalidOperationException;
import software.amazon.awssdk.services.workdocs.model.InvalidPasswordException;
import software.amazon.awssdk.services.workdocs.model.LimitExceededException;
import software.amazon.awssdk.services.workdocs.model.ProhibitedStateException;
import software.amazon.awssdk.services.workdocs.model.RemoveAllResourcePermissionsRequest;
import software.amazon.awssdk.services.workdocs.model.RemoveAllResourcePermissionsResponse;
import software.amazon.awssdk.services.workdocs.model.RemoveResourcePermissionRequest;
import software.amazon.awssdk.services.workdocs.model.RemoveResourcePermissionResponse;
import software.amazon.awssdk.services.workdocs.model.RequestedEntityTooLargeException;
import software.amazon.awssdk.services.workdocs.model.ResourceAlreadyCheckedOutException;
import software.amazon.awssdk.services.workdocs.model.ServiceUnavailableException;
import software.amazon.awssdk.services.workdocs.model.StorageLimitExceededException;
import software.amazon.awssdk.services.workdocs.model.StorageLimitWillExceedException;
import software.amazon.awssdk.services.workdocs.model.TooManyLabelsException;
import software.amazon.awssdk.services.workdocs.model.TooManySubscriptionsException;
import software.amazon.awssdk.services.workdocs.model.UnauthorizedOperationException;
import software.amazon.awssdk.services.workdocs.model.UnauthorizedResourceAccessException;
import software.amazon.awssdk.services.workdocs.model.UpdateDocumentRequest;
import software.amazon.awssdk.services.workdocs.model.UpdateDocumentResponse;
import software.amazon.awssdk.services.workdocs.model.UpdateDocumentVersionRequest;
import software.amazon.awssdk.services.workdocs.model.UpdateDocumentVersionResponse;
import software.amazon.awssdk.services.workdocs.model.UpdateFolderRequest;
import software.amazon.awssdk.services.workdocs.model.UpdateFolderResponse;
import software.amazon.awssdk.services.workdocs.model.UpdateUserRequest;
import software.amazon.awssdk.services.workdocs.model.UpdateUserResponse;
import software.amazon.awssdk.services.workdocs.model.WorkDocsException;
import software.amazon.awssdk.services.workdocs.model.WorkDocsRequest;
import software.amazon.awssdk.services.workdocs.paginators.DescribeDocumentVersionsIterable;
import software.amazon.awssdk.services.workdocs.paginators.DescribeFolderContentsIterable;
import software.amazon.awssdk.services.workdocs.paginators.DescribeUsersIterable;
import software.amazon.awssdk.services.workdocs.transform.AbortDocumentVersionUploadRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.ActivateUserRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.AddResourcePermissionsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.CreateCommentRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.CreateCustomMetadataRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.CreateFolderRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.CreateLabelsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.CreateNotificationSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.CreateUserRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DeactivateUserRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DeleteCommentRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DeleteCustomMetadataRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DeleteDocumentRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DeleteFolderContentsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DeleteFolderRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DeleteLabelsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DeleteNotificationSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DeleteUserRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DescribeActivitiesRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DescribeCommentsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DescribeDocumentVersionsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DescribeFolderContentsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DescribeGroupsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DescribeNotificationSubscriptionsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DescribeResourcePermissionsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DescribeRootFoldersRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.DescribeUsersRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.GetCurrentUserRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.GetDocumentPathRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.GetDocumentRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.GetDocumentVersionRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.GetFolderPathRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.GetFolderRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.GetResourcesRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.InitiateDocumentVersionUploadRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.RemoveAllResourcePermissionsRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.RemoveResourcePermissionRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.UpdateDocumentRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.UpdateDocumentVersionRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.UpdateFolderRequestMarshaller;
import software.amazon.awssdk.services.workdocs.transform.UpdateUserRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link WorkDocsClient}.
 *
 * @see WorkDocsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultWorkDocsClient implements WorkDocsClient {
    private static final Logger log = Logger.loggerFor(DefaultWorkDocsClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultWorkDocsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Aborts the upload of the specified document version that was previously initiated by
     * <a>InitiateDocumentVersionUpload</a>. The client should make this call only when it no longer intends to upload
     * the document version, or fails to do so.
     * </p>
     *
     * @param abortDocumentVersionUploadRequest
     * @return Result of the AbortDocumentVersionUpload operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws ProhibitedStateException
     *         The specified document version is not in the INITIALIZED state.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.AbortDocumentVersionUpload
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/AbortDocumentVersionUpload"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AbortDocumentVersionUploadResponse abortDocumentVersionUpload(
            AbortDocumentVersionUploadRequest abortDocumentVersionUploadRequest) throws EntityNotExistsException,
            ProhibitedStateException, UnauthorizedOperationException, UnauthorizedResourceAccessException,
            FailedDependencyException, ServiceUnavailableException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AbortDocumentVersionUploadResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AbortDocumentVersionUploadResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, abortDocumentVersionUploadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AbortDocumentVersionUpload");

            return clientHandler
                    .execute(new ClientExecutionParams<AbortDocumentVersionUploadRequest, AbortDocumentVersionUploadResponse>()
                            .withOperationName("AbortDocumentVersionUpload").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(abortDocumentVersionUploadRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AbortDocumentVersionUploadRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Activates the specified user. Only active users can access Amazon WorkDocs.
     * </p>
     *
     * @param activateUserRequest
     * @return Result of the ActivateUser operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.ActivateUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/ActivateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ActivateUserResponse activateUser(ActivateUserRequest activateUserRequest) throws EntityNotExistsException,
            UnauthorizedOperationException, UnauthorizedResourceAccessException, FailedDependencyException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ActivateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ActivateUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, activateUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ActivateUser");

            return clientHandler.execute(new ClientExecutionParams<ActivateUserRequest, ActivateUserResponse>()
                    .withOperationName("ActivateUser").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(activateUserRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ActivateUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a set of permissions for the specified folder or document. The resource permissions are overwritten if
     * the principals already have different permissions.
     * </p>
     *
     * @param addResourcePermissionsRequest
     * @return Result of the AddResourcePermissions operation returned by the service.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.AddResourcePermissions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/AddResourcePermissions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AddResourcePermissionsResponse addResourcePermissions(AddResourcePermissionsRequest addResourcePermissionsRequest)
            throws UnauthorizedOperationException, UnauthorizedResourceAccessException, FailedDependencyException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddResourcePermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AddResourcePermissionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addResourcePermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddResourcePermissions");

            return clientHandler
                    .execute(new ClientExecutionParams<AddResourcePermissionsRequest, AddResourcePermissionsResponse>()
                            .withOperationName("AddResourcePermissions").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(addResourcePermissionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AddResourcePermissionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds a new comment to the specified document version.
     * </p>
     *
     * @param createCommentRequest
     * @return Result of the CreateComment operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws ProhibitedStateException
     *         The specified document version is not in the INITIALIZED state.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws DocumentLockedForCommentsException
     *         This exception is thrown when the document is locked for comments and user tries to create or delete a
     *         comment on that document.
     * @throws InvalidCommentOperationException
     *         The requested operation is not allowed on the specified comment object.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.CreateComment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/CreateComment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateCommentResponse createComment(CreateCommentRequest createCommentRequest) throws EntityNotExistsException,
            ProhibitedStateException, UnauthorizedOperationException, UnauthorizedResourceAccessException,
            FailedDependencyException, ServiceUnavailableException, DocumentLockedForCommentsException,
            InvalidCommentOperationException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateCommentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateCommentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCommentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateComment");

            return clientHandler.execute(new ClientExecutionParams<CreateCommentRequest, CreateCommentResponse>()
                    .withOperationName("CreateComment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createCommentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateCommentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds one or more custom properties to the specified resource (a folder, document, or version).
     * </p>
     *
     * @param createCustomMetadataRequest
     * @return Result of the CreateCustomMetadata operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws ProhibitedStateException
     *         The specified document version is not in the INITIALIZED state.
     * @throws CustomMetadataLimitExceededException
     *         The limit has been reached on the number of custom properties for the specified resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.CreateCustomMetadata
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/CreateCustomMetadata" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateCustomMetadataResponse createCustomMetadata(CreateCustomMetadataRequest createCustomMetadataRequest)
            throws EntityNotExistsException, UnauthorizedOperationException, UnauthorizedResourceAccessException,
            ProhibitedStateException, CustomMetadataLimitExceededException, FailedDependencyException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateCustomMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateCustomMetadataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCustomMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCustomMetadata");

            return clientHandler.execute(new ClientExecutionParams<CreateCustomMetadataRequest, CreateCustomMetadataResponse>()
                    .withOperationName("CreateCustomMetadata").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createCustomMetadataRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateCustomMetadataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a folder with the specified name and parent folder.
     * </p>
     *
     * @param createFolderRequest
     * @return Result of the CreateFolder operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws EntityAlreadyExistsException
     *         The resource already exists.
     * @throws ProhibitedStateException
     *         The specified document version is not in the INITIALIZED state.
     * @throws ConflictingOperationException
     *         Another operation is in progress on the resource that conflicts with the current operation.
     * @throws LimitExceededException
     *         The maximum of 100,000 folders under the parent folder has been exceeded.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.CreateFolder
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/CreateFolder" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateFolderResponse createFolder(CreateFolderRequest createFolderRequest) throws EntityNotExistsException,
            EntityAlreadyExistsException, ProhibitedStateException, ConflictingOperationException, LimitExceededException,
            UnauthorizedOperationException, UnauthorizedResourceAccessException, FailedDependencyException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateFolderResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateFolderResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFolderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFolder");

            return clientHandler.execute(new ClientExecutionParams<CreateFolderRequest, CreateFolderResponse>()
                    .withOperationName("CreateFolder").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createFolderRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateFolderRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds the specified list of labels to the given resource (a document or folder)
     * </p>
     *
     * @param createLabelsRequest
     * @return Result of the CreateLabels operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws TooManyLabelsException
     *         The limit has been reached on the number of labels for the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.CreateLabels
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/CreateLabels" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateLabelsResponse createLabels(CreateLabelsRequest createLabelsRequest) throws EntityNotExistsException,
            UnauthorizedOperationException, UnauthorizedResourceAccessException, FailedDependencyException,
            ServiceUnavailableException, TooManyLabelsException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateLabelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateLabelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLabelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLabels");

            return clientHandler.execute(new ClientExecutionParams<CreateLabelsRequest, CreateLabelsResponse>()
                    .withOperationName("CreateLabels").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createLabelsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateLabelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Configure Amazon WorkDocs to use Amazon SNS notifications. The endpoint receives a confirmation message, and must
     * confirm the subscription.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/workdocs/latest/developerguide/subscribe-notifications.html">Subscribe to
     * Notifications</a> in the <i>Amazon WorkDocs Developer Guide</i>.
     * </p>
     *
     * @param createNotificationSubscriptionRequest
     * @return Result of the CreateNotificationSubscription operation returned by the service.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws TooManySubscriptionsException
     *         You've reached the limit on the number of subscriptions for the WorkDocs instance.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.CreateNotificationSubscription
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/CreateNotificationSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateNotificationSubscriptionResponse createNotificationSubscription(
            CreateNotificationSubscriptionRequest createNotificationSubscriptionRequest)
            throws UnauthorizedResourceAccessException, TooManySubscriptionsException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateNotificationSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateNotificationSubscriptionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createNotificationSubscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateNotificationSubscription");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateNotificationSubscriptionRequest, CreateNotificationSubscriptionResponse>()
                            .withOperationName("CreateNotificationSubscription").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createNotificationSubscriptionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateNotificationSubscriptionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a user in a Simple AD or Microsoft AD directory. The status of a newly created user is "ACTIVE". New
     * users can access Amazon WorkDocs.
     * </p>
     *
     * @param createUserRequest
     * @return Result of the CreateUser operation returned by the service.
     * @throws EntityAlreadyExistsException
     *         The resource already exists.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.CreateUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/CreateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateUserResponse createUser(CreateUserRequest createUserRequest) throws EntityAlreadyExistsException,
            UnauthorizedOperationException, UnauthorizedResourceAccessException, FailedDependencyException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateUser");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateUserRequest, CreateUserResponse>().withOperationName("CreateUser")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createUserRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deactivates the specified user, which revokes the user's access to Amazon WorkDocs.
     * </p>
     *
     * @param deactivateUserRequest
     * @return Result of the DeactivateUser operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.DeactivateUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DeactivateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeactivateUserResponse deactivateUser(DeactivateUserRequest deactivateUserRequest) throws EntityNotExistsException,
            UnauthorizedOperationException, UnauthorizedResourceAccessException, FailedDependencyException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeactivateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeactivateUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deactivateUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeactivateUser");

            return clientHandler.execute(new ClientExecutionParams<DeactivateUserRequest, DeactivateUserResponse>()
                    .withOperationName("DeactivateUser").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deactivateUserRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeactivateUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified comment from the document version.
     * </p>
     *
     * @param deleteCommentRequest
     * @return Result of the DeleteComment operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws ProhibitedStateException
     *         The specified document version is not in the INITIALIZED state.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws DocumentLockedForCommentsException
     *         This exception is thrown when the document is locked for comments and user tries to create or delete a
     *         comment on that document.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.DeleteComment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DeleteComment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteCommentResponse deleteComment(DeleteCommentRequest deleteCommentRequest) throws EntityNotExistsException,
            ProhibitedStateException, UnauthorizedOperationException, UnauthorizedResourceAccessException,
            FailedDependencyException, ServiceUnavailableException, DocumentLockedForCommentsException, AwsServiceException,
            SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteCommentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteCommentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCommentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteComment");

            return clientHandler.execute(new ClientExecutionParams<DeleteCommentRequest, DeleteCommentResponse>()
                    .withOperationName("DeleteComment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteCommentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteCommentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes custom metadata from the specified resource.
     * </p>
     *
     * @param deleteCustomMetadataRequest
     * @return Result of the DeleteCustomMetadata operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws ProhibitedStateException
     *         The specified document version is not in the INITIALIZED state.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.DeleteCustomMetadata
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DeleteCustomMetadata" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteCustomMetadataResponse deleteCustomMetadata(DeleteCustomMetadataRequest deleteCustomMetadataRequest)
            throws EntityNotExistsException, UnauthorizedOperationException, UnauthorizedResourceAccessException,
            ProhibitedStateException, FailedDependencyException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteCustomMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteCustomMetadataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCustomMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCustomMetadata");

            return clientHandler.execute(new ClientExecutionParams<DeleteCustomMetadataRequest, DeleteCustomMetadataResponse>()
                    .withOperationName("DeleteCustomMetadata").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteCustomMetadataRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteCustomMetadataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Permanently deletes the specified document and its associated metadata.
     * </p>
     *
     * @param deleteDocumentRequest
     * @return Result of the DeleteDocument operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws ProhibitedStateException
     *         The specified document version is not in the INITIALIZED state.
     * @throws ConflictingOperationException
     *         Another operation is in progress on the resource that conflicts with the current operation.
     * @throws ConcurrentModificationException
     *         The resource hierarchy is changing.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.DeleteDocument
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DeleteDocument" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteDocumentResponse deleteDocument(DeleteDocumentRequest deleteDocumentRequest) throws EntityNotExistsException,
            ProhibitedStateException, ConflictingOperationException, ConcurrentModificationException,
            UnauthorizedOperationException, UnauthorizedResourceAccessException, FailedDependencyException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDocumentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteDocumentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDocumentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDocument");

            return clientHandler.execute(new ClientExecutionParams<DeleteDocumentRequest, DeleteDocumentResponse>()
                    .withOperationName("DeleteDocument").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteDocumentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDocumentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Permanently deletes the specified folder and its contents.
     * </p>
     *
     * @param deleteFolderRequest
     * @return Result of the DeleteFolder operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws ProhibitedStateException
     *         The specified document version is not in the INITIALIZED state.
     * @throws ConflictingOperationException
     *         Another operation is in progress on the resource that conflicts with the current operation.
     * @throws ConcurrentModificationException
     *         The resource hierarchy is changing.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.DeleteFolder
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DeleteFolder" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteFolderResponse deleteFolder(DeleteFolderRequest deleteFolderRequest) throws EntityNotExistsException,
            ProhibitedStateException, ConflictingOperationException, ConcurrentModificationException,
            UnauthorizedOperationException, UnauthorizedResourceAccessException, FailedDependencyException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteFolderResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteFolderResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFolderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFolder");

            return clientHandler.execute(new ClientExecutionParams<DeleteFolderRequest, DeleteFolderResponse>()
                    .withOperationName("DeleteFolder").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteFolderRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteFolderRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the contents of the specified folder.
     * </p>
     *
     * @param deleteFolderContentsRequest
     * @return Result of the DeleteFolderContents operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws ProhibitedStateException
     *         The specified document version is not in the INITIALIZED state.
     * @throws ConflictingOperationException
     *         Another operation is in progress on the resource that conflicts with the current operation.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.DeleteFolderContents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DeleteFolderContents" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteFolderContentsResponse deleteFolderContents(DeleteFolderContentsRequest deleteFolderContentsRequest)
            throws EntityNotExistsException, ProhibitedStateException, ConflictingOperationException,
            UnauthorizedOperationException, UnauthorizedResourceAccessException, FailedDependencyException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteFolderContentsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteFolderContentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFolderContentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFolderContents");

            return clientHandler.execute(new ClientExecutionParams<DeleteFolderContentsRequest, DeleteFolderContentsResponse>()
                    .withOperationName("DeleteFolderContents").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteFolderContentsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteFolderContentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified list of labels from a resource.
     * </p>
     *
     * @param deleteLabelsRequest
     * @return Result of the DeleteLabels operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.DeleteLabels
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DeleteLabels" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteLabelsResponse deleteLabels(DeleteLabelsRequest deleteLabelsRequest) throws EntityNotExistsException,
            UnauthorizedOperationException, UnauthorizedResourceAccessException, FailedDependencyException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteLabelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteLabelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLabelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLabels");

            return clientHandler.execute(new ClientExecutionParams<DeleteLabelsRequest, DeleteLabelsResponse>()
                    .withOperationName("DeleteLabels").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteLabelsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteLabelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified subscription from the specified organization.
     * </p>
     *
     * @param deleteNotificationSubscriptionRequest
     * @return Result of the DeleteNotificationSubscription operation returned by the service.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws ProhibitedStateException
     *         The specified document version is not in the INITIALIZED state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.DeleteNotificationSubscription
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DeleteNotificationSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteNotificationSubscriptionResponse deleteNotificationSubscription(
            DeleteNotificationSubscriptionRequest deleteNotificationSubscriptionRequest)
            throws UnauthorizedResourceAccessException, EntityNotExistsException, ServiceUnavailableException,
            ProhibitedStateException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteNotificationSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteNotificationSubscriptionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteNotificationSubscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNotificationSubscription");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteNotificationSubscriptionRequest, DeleteNotificationSubscriptionResponse>()
                            .withOperationName("DeleteNotificationSubscription").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteNotificationSubscriptionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteNotificationSubscriptionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified user from a Simple AD or Microsoft AD directory.
     * </p>
     *
     * @param deleteUserRequest
     * @return Result of the DeleteUser operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.DeleteUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DeleteUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteUserResponse deleteUser(DeleteUserRequest deleteUserRequest) throws EntityNotExistsException,
            UnauthorizedOperationException, UnauthorizedResourceAccessException, FailedDependencyException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteUser");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteUserRequest, DeleteUserResponse>().withOperationName("DeleteUser")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteUserRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the user activities in a specified time period.
     * </p>
     *
     * @param describeActivitiesRequest
     * @return Result of the DescribeActivities operation returned by the service.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws InvalidArgumentException
     *         The pagination marker or limit fields are not valid.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.DescribeActivities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DescribeActivities" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeActivitiesResponse describeActivities(DescribeActivitiesRequest describeActivitiesRequest)
            throws UnauthorizedOperationException, UnauthorizedResourceAccessException, InvalidArgumentException,
            FailedDependencyException, ServiceUnavailableException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeActivitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeActivitiesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeActivitiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeActivities");

            return clientHandler.execute(new ClientExecutionParams<DescribeActivitiesRequest, DescribeActivitiesResponse>()
                    .withOperationName("DescribeActivities").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeActivitiesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeActivitiesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List all the comments for the specified document version.
     * </p>
     *
     * @param describeCommentsRequest
     * @return Result of the DescribeComments operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws ProhibitedStateException
     *         The specified document version is not in the INITIALIZED state.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.DescribeComments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DescribeComments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeCommentsResponse describeComments(DescribeCommentsRequest describeCommentsRequest)
            throws EntityNotExistsException, ProhibitedStateException, UnauthorizedOperationException,
            UnauthorizedResourceAccessException, FailedDependencyException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeCommentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeCommentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeCommentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeComments");

            return clientHandler.execute(new ClientExecutionParams<DescribeCommentsRequest, DescribeCommentsResponse>()
                    .withOperationName("DescribeComments").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeCommentsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeCommentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the document versions for the specified document.
     * </p>
     * <p>
     * By default, only active versions are returned.
     * </p>
     *
     * @param describeDocumentVersionsRequest
     * @return Result of the DescribeDocumentVersions operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws InvalidArgumentException
     *         The pagination marker or limit fields are not valid.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws ProhibitedStateException
     *         The specified document version is not in the INITIALIZED state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.DescribeDocumentVersions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DescribeDocumentVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeDocumentVersionsResponse describeDocumentVersions(
            DescribeDocumentVersionsRequest describeDocumentVersionsRequest) throws EntityNotExistsException,
            UnauthorizedOperationException, UnauthorizedResourceAccessException, InvalidArgumentException,
            FailedDependencyException, ServiceUnavailableException, ProhibitedStateException, AwsServiceException,
            SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDocumentVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeDocumentVersionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDocumentVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDocumentVersions");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeDocumentVersionsRequest, DescribeDocumentVersionsResponse>()
                            .withOperationName("DescribeDocumentVersions").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeDocumentVersionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeDocumentVersionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the document versions for the specified document.
     * </p>
     * <p>
     * By default, only active versions are returned.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeDocumentVersions(software.amazon.awssdk.services.workdocs.model.DescribeDocumentVersionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.workdocs.paginators.DescribeDocumentVersionsIterable responses = client.describeDocumentVersionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.workdocs.paginators.DescribeDocumentVersionsIterable responses = client
     *             .describeDocumentVersionsPaginator(request);
     *     for (software.amazon.awssdk.services.workdocs.model.DescribeDocumentVersionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.workdocs.paginators.DescribeDocumentVersionsIterable responses = client.describeDocumentVersionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeDocumentVersions(software.amazon.awssdk.services.workdocs.model.DescribeDocumentVersionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeDocumentVersionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws InvalidArgumentException
     *         The pagination marker or limit fields are not valid.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws ProhibitedStateException
     *         The specified document version is not in the INITIALIZED state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.DescribeDocumentVersions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DescribeDocumentVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeDocumentVersionsIterable describeDocumentVersionsPaginator(
            DescribeDocumentVersionsRequest describeDocumentVersionsRequest) throws EntityNotExistsException,
            UnauthorizedOperationException, UnauthorizedResourceAccessException, InvalidArgumentException,
            FailedDependencyException, ServiceUnavailableException, ProhibitedStateException, AwsServiceException,
            SdkClientException, WorkDocsException {
        return new DescribeDocumentVersionsIterable(this, applyPaginatorUserAgent(describeDocumentVersionsRequest));
    }

    /**
     * <p>
     * Describes the contents of the specified folder, including its documents and subfolders.
     * </p>
     * <p>
     * By default, Amazon WorkDocs returns the first 100 active document and folder metadata items. If there are more
     * results, the response includes a marker that you can use to request the next set of results. You can also request
     * initialized documents.
     * </p>
     *
     * @param describeFolderContentsRequest
     * @return Result of the DescribeFolderContents operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws InvalidArgumentException
     *         The pagination marker or limit fields are not valid.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws ProhibitedStateException
     *         The specified document version is not in the INITIALIZED state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.DescribeFolderContents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DescribeFolderContents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeFolderContentsResponse describeFolderContents(DescribeFolderContentsRequest describeFolderContentsRequest)
            throws EntityNotExistsException, UnauthorizedResourceAccessException, InvalidArgumentException,
            FailedDependencyException, ServiceUnavailableException, ProhibitedStateException, AwsServiceException,
            SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFolderContentsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeFolderContentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFolderContentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFolderContents");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeFolderContentsRequest, DescribeFolderContentsResponse>()
                            .withOperationName("DescribeFolderContents").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeFolderContentsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeFolderContentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the contents of the specified folder, including its documents and subfolders.
     * </p>
     * <p>
     * By default, Amazon WorkDocs returns the first 100 active document and folder metadata items. If there are more
     * results, the response includes a marker that you can use to request the next set of results. You can also request
     * initialized documents.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeFolderContents(software.amazon.awssdk.services.workdocs.model.DescribeFolderContentsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.workdocs.paginators.DescribeFolderContentsIterable responses = client.describeFolderContentsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.workdocs.paginators.DescribeFolderContentsIterable responses = client
     *             .describeFolderContentsPaginator(request);
     *     for (software.amazon.awssdk.services.workdocs.model.DescribeFolderContentsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.workdocs.paginators.DescribeFolderContentsIterable responses = client.describeFolderContentsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeFolderContents(software.amazon.awssdk.services.workdocs.model.DescribeFolderContentsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeFolderContentsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws InvalidArgumentException
     *         The pagination marker or limit fields are not valid.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws ProhibitedStateException
     *         The specified document version is not in the INITIALIZED state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.DescribeFolderContents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DescribeFolderContents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeFolderContentsIterable describeFolderContentsPaginator(
            DescribeFolderContentsRequest describeFolderContentsRequest) throws EntityNotExistsException,
            UnauthorizedResourceAccessException, InvalidArgumentException, FailedDependencyException,
            ServiceUnavailableException, ProhibitedStateException, AwsServiceException, SdkClientException, WorkDocsException {
        return new DescribeFolderContentsIterable(this, applyPaginatorUserAgent(describeFolderContentsRequest));
    }

    /**
     * <p>
     * Describes the groups specified by the query. Groups are defined by the underlying Active Directory.
     * </p>
     *
     * @param describeGroupsRequest
     * @return Result of the DescribeGroups operation returned by the service.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.DescribeGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DescribeGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeGroupsResponse describeGroups(DescribeGroupsRequest describeGroupsRequest)
            throws UnauthorizedOperationException, UnauthorizedResourceAccessException, FailedDependencyException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGroups");

            return clientHandler.execute(new ClientExecutionParams<DescribeGroupsRequest, DescribeGroupsResponse>()
                    .withOperationName("DescribeGroups").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeGroupsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the specified notification subscriptions.
     * </p>
     *
     * @param describeNotificationSubscriptionsRequest
     * @return Result of the DescribeNotificationSubscriptions operation returned by the service.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.DescribeNotificationSubscriptions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DescribeNotificationSubscriptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeNotificationSubscriptionsResponse describeNotificationSubscriptions(
            DescribeNotificationSubscriptionsRequest describeNotificationSubscriptionsRequest)
            throws UnauthorizedResourceAccessException, EntityNotExistsException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeNotificationSubscriptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeNotificationSubscriptionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeNotificationSubscriptionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeNotificationSubscriptions");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeNotificationSubscriptionsRequest, DescribeNotificationSubscriptionsResponse>()
                            .withOperationName("DescribeNotificationSubscriptions").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeNotificationSubscriptionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeNotificationSubscriptionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the permissions of a specified resource.
     * </p>
     *
     * @param describeResourcePermissionsRequest
     * @return Result of the DescribeResourcePermissions operation returned by the service.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.DescribeResourcePermissions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DescribeResourcePermissions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeResourcePermissionsResponse describeResourcePermissions(
            DescribeResourcePermissionsRequest describeResourcePermissionsRequest) throws UnauthorizedOperationException,
            UnauthorizedResourceAccessException, FailedDependencyException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeResourcePermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeResourcePermissionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeResourcePermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeResourcePermissions");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeResourcePermissionsRequest, DescribeResourcePermissionsResponse>()
                            .withOperationName("DescribeResourcePermissions").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeResourcePermissionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeResourcePermissionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the current user's special folders; the <code>RootFolder</code> and the <code>RecycleBin</code>.
     * <code>RootFolder</code> is the root of user's files and folders and <code>RecycleBin</code> is the root of
     * recycled items. This is not a valid action for SigV4 (administrative API) clients.
     * </p>
     * <p>
     * This action requires an authentication token. To get an authentication token, register an application with Amazon
     * WorkDocs. For more information, see <a
     * href="https://docs.aws.amazon.com/workdocs/latest/developerguide/wd-auth-user.html">Authentication and Access
     * Control for User Applications</a> in the <i>Amazon WorkDocs Developer Guide</i>.
     * </p>
     *
     * @param describeRootFoldersRequest
     * @return Result of the DescribeRootFolders operation returned by the service.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws InvalidArgumentException
     *         The pagination marker or limit fields are not valid.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.DescribeRootFolders
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DescribeRootFolders" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeRootFoldersResponse describeRootFolders(DescribeRootFoldersRequest describeRootFoldersRequest)
            throws UnauthorizedOperationException, UnauthorizedResourceAccessException, InvalidArgumentException,
            FailedDependencyException, ServiceUnavailableException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeRootFoldersResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeRootFoldersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRootFoldersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRootFolders");

            return clientHandler.execute(new ClientExecutionParams<DescribeRootFoldersRequest, DescribeRootFoldersResponse>()
                    .withOperationName("DescribeRootFolders").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeRootFoldersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeRootFoldersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the specified users. You can describe all users or filter the results (for example, by status or
     * organization).
     * </p>
     * <p>
     * By default, Amazon WorkDocs returns the first 24 active or pending users. If there are more results, the response
     * includes a marker that you can use to request the next set of results.
     * </p>
     *
     * @param describeUsersRequest
     * @return Result of the DescribeUsers operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws InvalidArgumentException
     *         The pagination marker or limit fields are not valid.
     * @throws RequestedEntityTooLargeException
     *         The response is too large to return. The request must include a filter to reduce the size of the
     *         response.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.DescribeUsers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DescribeUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeUsersResponse describeUsers(DescribeUsersRequest describeUsersRequest) throws EntityNotExistsException,
            UnauthorizedOperationException, UnauthorizedResourceAccessException, FailedDependencyException,
            ServiceUnavailableException, InvalidArgumentException, RequestedEntityTooLargeException, AwsServiceException,
            SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeUsersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeUsers");

            return clientHandler.execute(new ClientExecutionParams<DescribeUsersRequest, DescribeUsersResponse>()
                    .withOperationName("DescribeUsers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeUsersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeUsersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the specified users. You can describe all users or filter the results (for example, by status or
     * organization).
     * </p>
     * <p>
     * By default, Amazon WorkDocs returns the first 24 active or pending users. If there are more results, the response
     * includes a marker that you can use to request the next set of results.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #describeUsers(software.amazon.awssdk.services.workdocs.model.DescribeUsersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.workdocs.paginators.DescribeUsersIterable responses = client.describeUsersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.workdocs.paginators.DescribeUsersIterable responses = client.describeUsersPaginator(request);
     *     for (software.amazon.awssdk.services.workdocs.model.DescribeUsersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.workdocs.paginators.DescribeUsersIterable responses = client.describeUsersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeUsers(software.amazon.awssdk.services.workdocs.model.DescribeUsersRequest)} operation.</b>
     * </p>
     *
     * @param describeUsersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws InvalidArgumentException
     *         The pagination marker or limit fields are not valid.
     * @throws RequestedEntityTooLargeException
     *         The response is too large to return. The request must include a filter to reduce the size of the
     *         response.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.DescribeUsers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/DescribeUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeUsersIterable describeUsersPaginator(DescribeUsersRequest describeUsersRequest)
            throws EntityNotExistsException, UnauthorizedOperationException, UnauthorizedResourceAccessException,
            FailedDependencyException, ServiceUnavailableException, InvalidArgumentException, RequestedEntityTooLargeException,
            AwsServiceException, SdkClientException, WorkDocsException {
        return new DescribeUsersIterable(this, applyPaginatorUserAgent(describeUsersRequest));
    }

    /**
     * <p>
     * Retrieves details of the current user for whom the authentication token was generated. This is not a valid action
     * for SigV4 (administrative API) clients.
     * </p>
     * <p>
     * This action requires an authentication token. To get an authentication token, register an application with Amazon
     * WorkDocs. For more information, see <a
     * href="https://docs.aws.amazon.com/workdocs/latest/developerguide/wd-auth-user.html">Authentication and Access
     * Control for User Applications</a> in the <i>Amazon WorkDocs Developer Guide</i>.
     * </p>
     *
     * @param getCurrentUserRequest
     * @return Result of the GetCurrentUser operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.GetCurrentUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/GetCurrentUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetCurrentUserResponse getCurrentUser(GetCurrentUserRequest getCurrentUserRequest) throws EntityNotExistsException,
            UnauthorizedOperationException, UnauthorizedResourceAccessException, FailedDependencyException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCurrentUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetCurrentUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCurrentUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCurrentUser");

            return clientHandler.execute(new ClientExecutionParams<GetCurrentUserRequest, GetCurrentUserResponse>()
                    .withOperationName("GetCurrentUser").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getCurrentUserRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetCurrentUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves details of a document.
     * </p>
     *
     * @param getDocumentRequest
     * @return Result of the GetDocument operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws InvalidArgumentException
     *         The pagination marker or limit fields are not valid.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws InvalidPasswordException
     *         The password is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.GetDocument
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/GetDocument" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetDocumentResponse getDocument(GetDocumentRequest getDocumentRequest) throws EntityNotExistsException,
            UnauthorizedOperationException, UnauthorizedResourceAccessException, InvalidArgumentException,
            FailedDependencyException, ServiceUnavailableException, InvalidPasswordException, AwsServiceException,
            SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDocumentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetDocumentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDocumentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDocument");

            return clientHandler.execute(new ClientExecutionParams<GetDocumentRequest, GetDocumentResponse>()
                    .withOperationName("GetDocument").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDocumentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDocumentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the path information (the hierarchy from the root folder) for the requested document.
     * </p>
     * <p>
     * By default, Amazon WorkDocs returns a maximum of 100 levels upwards from the requested document and only includes
     * the IDs of the parent folders in the path. You can limit the maximum number of levels. You can also request the
     * names of the parent folders.
     * </p>
     *
     * @param getDocumentPathRequest
     * @return Result of the GetDocumentPath operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.GetDocumentPath
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/GetDocumentPath" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetDocumentPathResponse getDocumentPath(GetDocumentPathRequest getDocumentPathRequest)
            throws EntityNotExistsException, UnauthorizedOperationException, UnauthorizedResourceAccessException,
            FailedDependencyException, ServiceUnavailableException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDocumentPathResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetDocumentPathResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDocumentPathRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDocumentPath");

            return clientHandler.execute(new ClientExecutionParams<GetDocumentPathRequest, GetDocumentPathResponse>()
                    .withOperationName("GetDocumentPath").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDocumentPathRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDocumentPathRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves version metadata for the specified document.
     * </p>
     *
     * @param getDocumentVersionRequest
     * @return Result of the GetDocumentVersion operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws ProhibitedStateException
     *         The specified document version is not in the INITIALIZED state.
     * @throws InvalidPasswordException
     *         The password is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.GetDocumentVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/GetDocumentVersion" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetDocumentVersionResponse getDocumentVersion(GetDocumentVersionRequest getDocumentVersionRequest)
            throws EntityNotExistsException, UnauthorizedOperationException, UnauthorizedResourceAccessException,
            FailedDependencyException, ServiceUnavailableException, ProhibitedStateException, InvalidPasswordException,
            AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDocumentVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDocumentVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDocumentVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDocumentVersion");

            return clientHandler.execute(new ClientExecutionParams<GetDocumentVersionRequest, GetDocumentVersionResponse>()
                    .withOperationName("GetDocumentVersion").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDocumentVersionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDocumentVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the metadata of the specified folder.
     * </p>
     *
     * @param getFolderRequest
     * @return Result of the GetFolder operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws InvalidArgumentException
     *         The pagination marker or limit fields are not valid.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws ProhibitedStateException
     *         The specified document version is not in the INITIALIZED state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.GetFolder
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/GetFolder" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetFolderResponse getFolder(GetFolderRequest getFolderRequest) throws EntityNotExistsException,
            UnauthorizedOperationException, UnauthorizedResourceAccessException, InvalidArgumentException,
            FailedDependencyException, ServiceUnavailableException, ProhibitedStateException, AwsServiceException,
            SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetFolderResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetFolderResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFolderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFolder");

            return clientHandler.execute(new ClientExecutionParams<GetFolderRequest, GetFolderResponse>()
                    .withOperationName("GetFolder").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getFolderRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetFolderRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the path information (the hierarchy from the root folder) for the specified folder.
     * </p>
     * <p>
     * By default, Amazon WorkDocs returns a maximum of 100 levels upwards from the requested folder and only includes
     * the IDs of the parent folders in the path. You can limit the maximum number of levels. You can also request the
     * parent folder names.
     * </p>
     *
     * @param getFolderPathRequest
     * @return Result of the GetFolderPath operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.GetFolderPath
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/GetFolderPath" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetFolderPathResponse getFolderPath(GetFolderPathRequest getFolderPathRequest) throws EntityNotExistsException,
            UnauthorizedOperationException, UnauthorizedResourceAccessException, FailedDependencyException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetFolderPathResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetFolderPathResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFolderPathRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFolderPath");

            return clientHandler.execute(new ClientExecutionParams<GetFolderPathRequest, GetFolderPathResponse>()
                    .withOperationName("GetFolderPath").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getFolderPathRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetFolderPathRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a collection of resources, including folders and documents. The only <code>CollectionType</code>
     * supported is <code>SHARED_WITH_ME</code>.
     * </p>
     *
     * @param getResourcesRequest
     * @return Result of the GetResources operation returned by the service.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws InvalidArgumentException
     *         The pagination marker or limit fields are not valid.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.GetResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/GetResources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetResourcesResponse getResources(GetResourcesRequest getResourcesRequest) throws UnauthorizedResourceAccessException,
            UnauthorizedOperationException, InvalidArgumentException, FailedDependencyException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResources");

            return clientHandler.execute(new ClientExecutionParams<GetResourcesRequest, GetResourcesResponse>()
                    .withOperationName("GetResources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new document object and version object.
     * </p>
     * <p>
     * The client specifies the parent folder ID and name of the document to upload. The ID is optionally specified when
     * creating a new version of an existing document. This is the first step to upload a document. Next, upload the
     * document to the URL returned from the call, and then call <a>UpdateDocumentVersion</a>.
     * </p>
     * <p>
     * To cancel the document upload, call <a>AbortDocumentVersionUpload</a>.
     * </p>
     *
     * @param initiateDocumentVersionUploadRequest
     * @return Result of the InitiateDocumentVersionUpload operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws EntityAlreadyExistsException
     *         The resource already exists.
     * @throws StorageLimitExceededException
     *         The storage limit has been exceeded.
     * @throws StorageLimitWillExceedException
     *         The storage limit will be exceeded.
     * @throws ProhibitedStateException
     *         The specified document version is not in the INITIALIZED state.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws DraftUploadOutOfSyncException
     *         This exception is thrown when a valid checkout ID is not presented on document version upload calls for a
     *         document that has been checked out from Web client.
     * @throws ResourceAlreadyCheckedOutException
     *         The resource is already checked out.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.InitiateDocumentVersionUpload
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/InitiateDocumentVersionUpload"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public InitiateDocumentVersionUploadResponse initiateDocumentVersionUpload(
            InitiateDocumentVersionUploadRequest initiateDocumentVersionUploadRequest) throws EntityNotExistsException,
            EntityAlreadyExistsException, StorageLimitExceededException, StorageLimitWillExceedException,
            ProhibitedStateException, UnauthorizedOperationException, UnauthorizedResourceAccessException,
            FailedDependencyException, ServiceUnavailableException, DraftUploadOutOfSyncException,
            ResourceAlreadyCheckedOutException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<InitiateDocumentVersionUploadResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, InitiateDocumentVersionUploadResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                initiateDocumentVersionUploadRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InitiateDocumentVersionUpload");

            return clientHandler
                    .execute(new ClientExecutionParams<InitiateDocumentVersionUploadRequest, InitiateDocumentVersionUploadResponse>()
                            .withOperationName("InitiateDocumentVersionUpload").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(initiateDocumentVersionUploadRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new InitiateDocumentVersionUploadRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes all the permissions from the specified resource.
     * </p>
     *
     * @param removeAllResourcePermissionsRequest
     * @return Result of the RemoveAllResourcePermissions operation returned by the service.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.RemoveAllResourcePermissions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/RemoveAllResourcePermissions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RemoveAllResourcePermissionsResponse removeAllResourcePermissions(
            RemoveAllResourcePermissionsRequest removeAllResourcePermissionsRequest) throws UnauthorizedOperationException,
            UnauthorizedResourceAccessException, FailedDependencyException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveAllResourcePermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RemoveAllResourcePermissionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeAllResourcePermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveAllResourcePermissions");

            return clientHandler
                    .execute(new ClientExecutionParams<RemoveAllResourcePermissionsRequest, RemoveAllResourcePermissionsResponse>()
                            .withOperationName("RemoveAllResourcePermissions").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(removeAllResourcePermissionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RemoveAllResourcePermissionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the permission for the specified principal from the specified resource.
     * </p>
     *
     * @param removeResourcePermissionRequest
     * @return Result of the RemoveResourcePermission operation returned by the service.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.RemoveResourcePermission
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/RemoveResourcePermission"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RemoveResourcePermissionResponse removeResourcePermission(
            RemoveResourcePermissionRequest removeResourcePermissionRequest) throws UnauthorizedOperationException,
            UnauthorizedResourceAccessException, FailedDependencyException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveResourcePermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RemoveResourcePermissionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeResourcePermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveResourcePermission");

            return clientHandler
                    .execute(new ClientExecutionParams<RemoveResourcePermissionRequest, RemoveResourcePermissionResponse>()
                            .withOperationName("RemoveResourcePermission").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(removeResourcePermissionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RemoveResourcePermissionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the specified attributes of a document. The user must have access to both the document and its parent
     * folder, if applicable.
     * </p>
     *
     * @param updateDocumentRequest
     * @return Result of the UpdateDocument operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws EntityAlreadyExistsException
     *         The resource already exists.
     * @throws LimitExceededException
     *         The maximum of 100,000 folders under the parent folder has been exceeded.
     * @throws ProhibitedStateException
     *         The specified document version is not in the INITIALIZED state.
     * @throws ConflictingOperationException
     *         Another operation is in progress on the resource that conflicts with the current operation.
     * @throws ConcurrentModificationException
     *         The resource hierarchy is changing.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.UpdateDocument
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/UpdateDocument" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateDocumentResponse updateDocument(UpdateDocumentRequest updateDocumentRequest) throws EntityNotExistsException,
            EntityAlreadyExistsException, LimitExceededException, ProhibitedStateException, ConflictingOperationException,
            ConcurrentModificationException, UnauthorizedOperationException, UnauthorizedResourceAccessException,
            FailedDependencyException, ServiceUnavailableException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDocumentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateDocumentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDocumentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDocument");

            return clientHandler.execute(new ClientExecutionParams<UpdateDocumentRequest, UpdateDocumentResponse>()
                    .withOperationName("UpdateDocument").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateDocumentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateDocumentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Changes the status of the document version to ACTIVE.
     * </p>
     * <p>
     * Amazon WorkDocs also sets its document container to ACTIVE. This is the last step in a document upload, after the
     * client uploads the document to an S3-presigned URL returned by <a>InitiateDocumentVersionUpload</a>.
     * </p>
     *
     * @param updateDocumentVersionRequest
     * @return Result of the UpdateDocumentVersion operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws ProhibitedStateException
     *         The specified document version is not in the INITIALIZED state.
     * @throws ConcurrentModificationException
     *         The resource hierarchy is changing.
     * @throws InvalidOperationException
     *         The operation is invalid.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.UpdateDocumentVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/UpdateDocumentVersion" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateDocumentVersionResponse updateDocumentVersion(UpdateDocumentVersionRequest updateDocumentVersionRequest)
            throws EntityNotExistsException, ProhibitedStateException, ConcurrentModificationException,
            InvalidOperationException, UnauthorizedOperationException, UnauthorizedResourceAccessException,
            FailedDependencyException, ServiceUnavailableException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDocumentVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDocumentVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDocumentVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDocumentVersion");

            return clientHandler.execute(new ClientExecutionParams<UpdateDocumentVersionRequest, UpdateDocumentVersionResponse>()
                    .withOperationName("UpdateDocumentVersion").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateDocumentVersionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateDocumentVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the specified attributes of the specified folder. The user must have access to both the folder and its
     * parent folder, if applicable.
     * </p>
     *
     * @param updateFolderRequest
     * @return Result of the UpdateFolder operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws EntityAlreadyExistsException
     *         The resource already exists.
     * @throws ProhibitedStateException
     *         The specified document version is not in the INITIALIZED state.
     * @throws ConflictingOperationException
     *         Another operation is in progress on the resource that conflicts with the current operation.
     * @throws ConcurrentModificationException
     *         The resource hierarchy is changing.
     * @throws LimitExceededException
     *         The maximum of 100,000 folders under the parent folder has been exceeded.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.UpdateFolder
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/UpdateFolder" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateFolderResponse updateFolder(UpdateFolderRequest updateFolderRequest) throws EntityNotExistsException,
            EntityAlreadyExistsException, ProhibitedStateException, ConflictingOperationException,
            ConcurrentModificationException, LimitExceededException, UnauthorizedOperationException,
            UnauthorizedResourceAccessException, FailedDependencyException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateFolderResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateFolderResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFolderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFolder");

            return clientHandler.execute(new ClientExecutionParams<UpdateFolderRequest, UpdateFolderResponse>()
                    .withOperationName("UpdateFolder").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateFolderRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateFolderRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the specified attributes of the specified user, and grants or revokes administrative privileges to the
     * Amazon WorkDocs site.
     * </p>
     *
     * @param updateUserRequest
     * @return Result of the UpdateUser operation returned by the service.
     * @throws EntityNotExistsException
     *         The resource does not exist.
     * @throws UnauthorizedOperationException
     *         The operation is not permitted.
     * @throws UnauthorizedResourceAccessException
     *         The caller does not have access to perform the action on the resource.
     * @throws IllegalUserStateException
     *         The user is undergoing transfer of ownership.
     * @throws FailedDependencyException
     *         The AWS Directory Service cannot reach an on-premises instance. Or a dependency under the control of the
     *         organization is failing, such as a connected Active Directory.
     * @throws ServiceUnavailableException
     *         One or more of the dependencies is unavailable.
     * @throws DeactivatingLastSystemUserException
     *         The last user in the organization is being deactivated.
     * @throws InvalidArgumentException
     *         The pagination marker or limit fields are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkDocsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkDocsClient.UpdateUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workdocs-2016-05-01/UpdateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateUserResponse updateUser(UpdateUserRequest updateUserRequest) throws EntityNotExistsException,
            UnauthorizedOperationException, UnauthorizedResourceAccessException, IllegalUserStateException,
            FailedDependencyException, ServiceUnavailableException, DeactivatingLastSystemUserException,
            InvalidArgumentException, AwsServiceException, SdkClientException, WorkDocsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkDocs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateUser");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateUserRequest, UpdateUserResponse>().withOperationName("UpdateUser")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateUserRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(WorkDocsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FailedDependencyException")
                                .exceptionBuilderSupplier(FailedDependencyException::builder).httpStatusCode(424).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidOperationException")
                                .exceptionBuilderSupplier(InvalidOperationException::builder).httpStatusCode(405).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CustomMetadataLimitExceededException")
                                .exceptionBuilderSupplier(CustomMetadataLimitExceededException::builder).httpStatusCode(429)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedResourceAccessException")
                                .exceptionBuilderSupplier(UnauthorizedResourceAccessException::builder).httpStatusCode(404)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManySubscriptionsException")
                                .exceptionBuilderSupplier(TooManySubscriptionsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EntityAlreadyExistsException")
                                .exceptionBuilderSupplier(EntityAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EntityNotExistsException")
                                .exceptionBuilderSupplier(EntityNotExistsException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DocumentLockedForCommentsException")
                                .exceptionBuilderSupplier(DocumentLockedForCommentsException::builder).httpStatusCode(409)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StorageLimitExceededException")
                                .exceptionBuilderSupplier(StorageLimitExceededException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StorageLimitWillExceedException")
                                .exceptionBuilderSupplier(StorageLimitWillExceedException::builder).httpStatusCode(413).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ProhibitedStateException")
                                .exceptionBuilderSupplier(ProhibitedStateException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IllegalUserStateException")
                                .exceptionBuilderSupplier(IllegalUserStateException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArgumentException")
                                .exceptionBuilderSupplier(InvalidArgumentException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictingOperationException")
                                .exceptionBuilderSupplier(ConflictingOperationException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidCommentOperationException")
                                .exceptionBuilderSupplier(InvalidCommentOperationException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPasswordException")
                                .exceptionBuilderSupplier(InvalidPasswordException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedOperationException")
                                .exceptionBuilderSupplier(UnauthorizedOperationException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyLabelsException")
                                .exceptionBuilderSupplier(TooManyLabelsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DraftUploadOutOfSyncException")
                                .exceptionBuilderSupplier(DraftUploadOutOfSyncException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyCheckedOutException")
                                .exceptionBuilderSupplier(ResourceAlreadyCheckedOutException::builder).httpStatusCode(409)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestedEntityTooLargeException")
                                .exceptionBuilderSupplier(RequestedEntityTooLargeException::builder).httpStatusCode(413).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DeactivatingLastSystemUserException")
                                .exceptionBuilderSupplier(DeactivatingLastSystemUserException::builder).httpStatusCode(409)
                                .build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends WorkDocsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
