/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wellarchitected;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.wellarchitected.model.AccessDeniedException;
import software.amazon.awssdk.services.wellarchitected.model.AssociateLensesRequest;
import software.amazon.awssdk.services.wellarchitected.model.AssociateLensesResponse;
import software.amazon.awssdk.services.wellarchitected.model.ConflictException;
import software.amazon.awssdk.services.wellarchitected.model.CreateMilestoneRequest;
import software.amazon.awssdk.services.wellarchitected.model.CreateMilestoneResponse;
import software.amazon.awssdk.services.wellarchitected.model.CreateWorkloadRequest;
import software.amazon.awssdk.services.wellarchitected.model.CreateWorkloadResponse;
import software.amazon.awssdk.services.wellarchitected.model.CreateWorkloadShareRequest;
import software.amazon.awssdk.services.wellarchitected.model.CreateWorkloadShareResponse;
import software.amazon.awssdk.services.wellarchitected.model.DeleteWorkloadRequest;
import software.amazon.awssdk.services.wellarchitected.model.DeleteWorkloadResponse;
import software.amazon.awssdk.services.wellarchitected.model.DeleteWorkloadShareRequest;
import software.amazon.awssdk.services.wellarchitected.model.DeleteWorkloadShareResponse;
import software.amazon.awssdk.services.wellarchitected.model.DisassociateLensesRequest;
import software.amazon.awssdk.services.wellarchitected.model.DisassociateLensesResponse;
import software.amazon.awssdk.services.wellarchitected.model.GetAnswerRequest;
import software.amazon.awssdk.services.wellarchitected.model.GetAnswerResponse;
import software.amazon.awssdk.services.wellarchitected.model.GetLensReviewReportRequest;
import software.amazon.awssdk.services.wellarchitected.model.GetLensReviewReportResponse;
import software.amazon.awssdk.services.wellarchitected.model.GetLensReviewRequest;
import software.amazon.awssdk.services.wellarchitected.model.GetLensReviewResponse;
import software.amazon.awssdk.services.wellarchitected.model.GetLensVersionDifferenceRequest;
import software.amazon.awssdk.services.wellarchitected.model.GetLensVersionDifferenceResponse;
import software.amazon.awssdk.services.wellarchitected.model.GetMilestoneRequest;
import software.amazon.awssdk.services.wellarchitected.model.GetMilestoneResponse;
import software.amazon.awssdk.services.wellarchitected.model.GetWorkloadRequest;
import software.amazon.awssdk.services.wellarchitected.model.GetWorkloadResponse;
import software.amazon.awssdk.services.wellarchitected.model.InternalServerException;
import software.amazon.awssdk.services.wellarchitected.model.ListAnswersRequest;
import software.amazon.awssdk.services.wellarchitected.model.ListAnswersResponse;
import software.amazon.awssdk.services.wellarchitected.model.ListLensReviewImprovementsRequest;
import software.amazon.awssdk.services.wellarchitected.model.ListLensReviewImprovementsResponse;
import software.amazon.awssdk.services.wellarchitected.model.ListLensReviewsRequest;
import software.amazon.awssdk.services.wellarchitected.model.ListLensReviewsResponse;
import software.amazon.awssdk.services.wellarchitected.model.ListLensesRequest;
import software.amazon.awssdk.services.wellarchitected.model.ListLensesResponse;
import software.amazon.awssdk.services.wellarchitected.model.ListMilestonesRequest;
import software.amazon.awssdk.services.wellarchitected.model.ListMilestonesResponse;
import software.amazon.awssdk.services.wellarchitected.model.ListNotificationsRequest;
import software.amazon.awssdk.services.wellarchitected.model.ListNotificationsResponse;
import software.amazon.awssdk.services.wellarchitected.model.ListShareInvitationsRequest;
import software.amazon.awssdk.services.wellarchitected.model.ListShareInvitationsResponse;
import software.amazon.awssdk.services.wellarchitected.model.ListWorkloadSharesRequest;
import software.amazon.awssdk.services.wellarchitected.model.ListWorkloadSharesResponse;
import software.amazon.awssdk.services.wellarchitected.model.ListWorkloadsRequest;
import software.amazon.awssdk.services.wellarchitected.model.ListWorkloadsResponse;
import software.amazon.awssdk.services.wellarchitected.model.ResourceNotFoundException;
import software.amazon.awssdk.services.wellarchitected.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.wellarchitected.model.ThrottlingException;
import software.amazon.awssdk.services.wellarchitected.model.UpdateAnswerRequest;
import software.amazon.awssdk.services.wellarchitected.model.UpdateAnswerResponse;
import software.amazon.awssdk.services.wellarchitected.model.UpdateLensReviewRequest;
import software.amazon.awssdk.services.wellarchitected.model.UpdateLensReviewResponse;
import software.amazon.awssdk.services.wellarchitected.model.UpdateShareInvitationRequest;
import software.amazon.awssdk.services.wellarchitected.model.UpdateShareInvitationResponse;
import software.amazon.awssdk.services.wellarchitected.model.UpdateWorkloadRequest;
import software.amazon.awssdk.services.wellarchitected.model.UpdateWorkloadResponse;
import software.amazon.awssdk.services.wellarchitected.model.UpdateWorkloadShareRequest;
import software.amazon.awssdk.services.wellarchitected.model.UpdateWorkloadShareResponse;
import software.amazon.awssdk.services.wellarchitected.model.UpgradeLensReviewRequest;
import software.amazon.awssdk.services.wellarchitected.model.UpgradeLensReviewResponse;
import software.amazon.awssdk.services.wellarchitected.model.ValidationException;
import software.amazon.awssdk.services.wellarchitected.model.WellArchitectedException;
import software.amazon.awssdk.services.wellarchitected.model.WellArchitectedRequest;
import software.amazon.awssdk.services.wellarchitected.paginators.ListAnswersIterable;
import software.amazon.awssdk.services.wellarchitected.paginators.ListLensReviewImprovementsIterable;
import software.amazon.awssdk.services.wellarchitected.paginators.ListLensReviewsIterable;
import software.amazon.awssdk.services.wellarchitected.paginators.ListLensesIterable;
import software.amazon.awssdk.services.wellarchitected.paginators.ListMilestonesIterable;
import software.amazon.awssdk.services.wellarchitected.paginators.ListNotificationsIterable;
import software.amazon.awssdk.services.wellarchitected.paginators.ListShareInvitationsIterable;
import software.amazon.awssdk.services.wellarchitected.paginators.ListWorkloadSharesIterable;
import software.amazon.awssdk.services.wellarchitected.paginators.ListWorkloadsIterable;
import software.amazon.awssdk.services.wellarchitected.transform.AssociateLensesRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.CreateMilestoneRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.CreateWorkloadRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.CreateWorkloadShareRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.DeleteWorkloadRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.DeleteWorkloadShareRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.DisassociateLensesRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.GetAnswerRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.GetLensReviewReportRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.GetLensReviewRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.GetLensVersionDifferenceRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.GetMilestoneRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.GetWorkloadRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.ListAnswersRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.ListLensReviewImprovementsRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.ListLensReviewsRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.ListLensesRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.ListMilestonesRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.ListNotificationsRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.ListShareInvitationsRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.ListWorkloadSharesRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.ListWorkloadsRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.UpdateAnswerRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.UpdateLensReviewRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.UpdateShareInvitationRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.UpdateWorkloadRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.UpdateWorkloadShareRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.UpgradeLensReviewRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link WellArchitectedClient}.
 *
 * @see WellArchitectedClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultWellArchitectedClient implements WellArchitectedClient {
    private static final Logger log = Logger.loggerFor(DefaultWellArchitectedClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultWellArchitectedClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Associate a lens to a workload.
     * </p>
     *
     * @param associateLensesRequest
     *        Input to associate lens reviews.
     * @return Result of the AssociateLenses operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws ConflictException
     *         The resource already exists.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.AssociateLenses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/AssociateLenses"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateLensesResponse associateLenses(AssociateLensesRequest associateLensesRequest) throws ValidationException,
            ResourceNotFoundException, ConflictException, InternalServerException, AccessDeniedException, ThrottlingException,
            AwsServiceException, SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateLensesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AssociateLensesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateLensesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateLenses");

            return clientHandler.execute(new ClientExecutionParams<AssociateLensesRequest, AssociateLensesResponse>()
                    .withOperationName("AssociateLenses").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(associateLensesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AssociateLensesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Create a milestone for an existing workload.
     * </p>
     *
     * @param createMilestoneRequest
     *        Input for milestone creation.
     * @return Result of the CreateMilestone operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws ConflictException
     *         The resource already exists.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws ServiceQuotaExceededException
     *         The user has reached their resource quota.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.CreateMilestone
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/CreateMilestone"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateMilestoneResponse createMilestone(CreateMilestoneRequest createMilestoneRequest) throws ValidationException,
            ConflictException, InternalServerException, ResourceNotFoundException, ServiceQuotaExceededException,
            AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateMilestoneResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateMilestoneResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMilestoneRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMilestone");

            return clientHandler.execute(new ClientExecutionParams<CreateMilestoneRequest, CreateMilestoneResponse>()
                    .withOperationName("CreateMilestone").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createMilestoneRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateMilestoneRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Create a new workload.
     * </p>
     * <p>
     * The owner of a workload can share the workload with other AWS accounts and IAM users in the same AWS Region. Only
     * the owner of a workload can delete it.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/wellarchitected/latest/userguide/define-workload.html">Defining a Workload</a>
     * in the <i>AWS Well-Architected Tool User Guide</i>.
     * </p>
     *
     * @param createWorkloadRequest
     *        Input for workload creation.
     * @return Result of the CreateWorkload operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws ConflictException
     *         The resource already exists.
     * @throws ServiceQuotaExceededException
     *         The user has reached their resource quota.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.CreateWorkload
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/CreateWorkload" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateWorkloadResponse createWorkload(CreateWorkloadRequest createWorkloadRequest) throws ValidationException,
            ConflictException, ServiceQuotaExceededException, InternalServerException, AccessDeniedException,
            ThrottlingException, AwsServiceException, SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateWorkloadResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateWorkloadResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWorkloadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorkload");

            return clientHandler.execute(new ClientExecutionParams<CreateWorkloadRequest, CreateWorkloadResponse>()
                    .withOperationName("CreateWorkload").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createWorkloadRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateWorkloadRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Create a workload share.
     * </p>
     * <p>
     * The owner of a workload can share it with other AWS accounts and IAM users in the same AWS Region. Shared access
     * to a workload is not removed until the workload invitation is deleted.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/wellarchitected/latest/userguide/workloads-sharing.html">Sharing a Workload</a>
     * in the <i>AWS Well-Architected Tool User Guide</i>.
     * </p>
     *
     * @param createWorkloadShareRequest
     *        Input for Create Workload Share
     * @return Result of the CreateWorkloadShare operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws ConflictException
     *         The resource already exists.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws ServiceQuotaExceededException
     *         The user has reached their resource quota.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.CreateWorkloadShare
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/CreateWorkloadShare"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateWorkloadShareResponse createWorkloadShare(CreateWorkloadShareRequest createWorkloadShareRequest)
            throws ValidationException, ConflictException, InternalServerException, ResourceNotFoundException,
            ServiceQuotaExceededException, AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException,
            WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateWorkloadShareResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateWorkloadShareResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWorkloadShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorkloadShare");

            return clientHandler.execute(new ClientExecutionParams<CreateWorkloadShareRequest, CreateWorkloadShareResponse>()
                    .withOperationName("CreateWorkloadShare").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createWorkloadShareRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateWorkloadShareRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete an existing workload.
     * </p>
     *
     * @param deleteWorkloadRequest
     *        Input for workload deletion.
     * @return Result of the DeleteWorkload operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws ConflictException
     *         The resource already exists.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.DeleteWorkload
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/DeleteWorkload" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteWorkloadResponse deleteWorkload(DeleteWorkloadRequest deleteWorkloadRequest) throws ValidationException,
            ResourceNotFoundException, ConflictException, InternalServerException, AccessDeniedException, ThrottlingException,
            AwsServiceException, SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteWorkloadResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteWorkloadResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWorkloadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorkload");

            return clientHandler.execute(new ClientExecutionParams<DeleteWorkloadRequest, DeleteWorkloadResponse>()
                    .withOperationName("DeleteWorkload").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteWorkloadRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteWorkloadRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete a workload share.
     * </p>
     *
     * @param deleteWorkloadShareRequest
     *        Input for Delete Workload Share
     * @return Result of the DeleteWorkloadShare operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws ConflictException
     *         The resource already exists.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.DeleteWorkloadShare
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/DeleteWorkloadShare"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteWorkloadShareResponse deleteWorkloadShare(DeleteWorkloadShareRequest deleteWorkloadShareRequest)
            throws ValidationException, InternalServerException, ResourceNotFoundException, ConflictException,
            AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteWorkloadShareResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteWorkloadShareResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWorkloadShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorkloadShare");

            return clientHandler.execute(new ClientExecutionParams<DeleteWorkloadShareRequest, DeleteWorkloadShareResponse>()
                    .withOperationName("DeleteWorkloadShare").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteWorkloadShareRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteWorkloadShareRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disassociate a lens from a workload.
     * </p>
     * <note>
     * <p>
     * The AWS Well-Architected Framework lens (<code>wellarchitected</code>) cannot be removed from a workload.
     * </p>
     * </note>
     *
     * @param disassociateLensesRequest
     *        Input to disassociate lens reviews.
     * @return Result of the DisassociateLenses operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws ConflictException
     *         The resource already exists.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.DisassociateLenses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/DisassociateLenses"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateLensesResponse disassociateLenses(DisassociateLensesRequest disassociateLensesRequest)
            throws ValidationException, ResourceNotFoundException, ConflictException, InternalServerException,
            AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateLensesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateLensesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateLensesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateLenses");

            return clientHandler.execute(new ClientExecutionParams<DisassociateLensesRequest, DisassociateLensesResponse>()
                    .withOperationName("DisassociateLenses").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(disassociateLensesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisassociateLensesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get lens review.
     * </p>
     *
     * @param getAnswerRequest
     *        Input to get answer.
     * @return Result of the GetAnswer operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.GetAnswer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/GetAnswer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAnswerResponse getAnswer(GetAnswerRequest getAnswerRequest) throws ValidationException, ResourceNotFoundException,
            InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException,
            WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAnswerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAnswerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAnswerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAnswer");

            return clientHandler.execute(new ClientExecutionParams<GetAnswerRequest, GetAnswerResponse>()
                    .withOperationName("GetAnswer").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getAnswerRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetAnswerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get lens review.
     * </p>
     *
     * @param getLensReviewRequest
     *        Input to get lens review.
     * @return Result of the GetLensReview operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.GetLensReview
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/GetLensReview" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetLensReviewResponse getLensReview(GetLensReviewRequest getLensReviewRequest) throws ValidationException,
            ResourceNotFoundException, InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetLensReviewResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetLensReviewResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLensReviewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLensReview");

            return clientHandler.execute(new ClientExecutionParams<GetLensReviewRequest, GetLensReviewResponse>()
                    .withOperationName("GetLensReview").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getLensReviewRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetLensReviewRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get lens review report.
     * </p>
     *
     * @param getLensReviewReportRequest
     *        Input to get lens review report.
     * @return Result of the GetLensReviewReport operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.GetLensReviewReport
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/GetLensReviewReport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetLensReviewReportResponse getLensReviewReport(GetLensReviewReportRequest getLensReviewReportRequest)
            throws ValidationException, ResourceNotFoundException, InternalServerException, AccessDeniedException,
            ThrottlingException, AwsServiceException, SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetLensReviewReportResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetLensReviewReportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLensReviewReportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLensReviewReport");

            return clientHandler.execute(new ClientExecutionParams<GetLensReviewReportRequest, GetLensReviewReportResponse>()
                    .withOperationName("GetLensReviewReport").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getLensReviewReportRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetLensReviewReportRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get lens version differences.
     * </p>
     *
     * @param getLensVersionDifferenceRequest
     * @return Result of the GetLensVersionDifference operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.GetLensVersionDifference
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/GetLensVersionDifference"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetLensVersionDifferenceResponse getLensVersionDifference(
            GetLensVersionDifferenceRequest getLensVersionDifferenceRequest) throws ValidationException,
            ResourceNotFoundException, InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetLensVersionDifferenceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetLensVersionDifferenceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLensVersionDifferenceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLensVersionDifference");

            return clientHandler
                    .execute(new ClientExecutionParams<GetLensVersionDifferenceRequest, GetLensVersionDifferenceResponse>()
                            .withOperationName("GetLensVersionDifference").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getLensVersionDifferenceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetLensVersionDifferenceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get a milestone for an existing workload.
     * </p>
     *
     * @param getMilestoneRequest
     *        Input to get a milestone.
     * @return Result of the GetMilestone operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.GetMilestone
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/GetMilestone" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetMilestoneResponse getMilestone(GetMilestoneRequest getMilestoneRequest) throws ValidationException,
            ResourceNotFoundException, InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMilestoneResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMilestoneResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMilestoneRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMilestone");

            return clientHandler.execute(new ClientExecutionParams<GetMilestoneRequest, GetMilestoneResponse>()
                    .withOperationName("GetMilestone").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getMilestoneRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMilestoneRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get an existing workload.
     * </p>
     *
     * @param getWorkloadRequest
     *        Input to get a workload.
     * @return Result of the GetWorkload operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.GetWorkload
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/GetWorkload" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetWorkloadResponse getWorkload(GetWorkloadRequest getWorkloadRequest) throws ValidationException,
            ResourceNotFoundException, InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetWorkloadResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetWorkloadResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWorkloadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWorkload");

            return clientHandler.execute(new ClientExecutionParams<GetWorkloadRequest, GetWorkloadResponse>()
                    .withOperationName("GetWorkload").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getWorkloadRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetWorkloadRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List of answers.
     * </p>
     *
     * @param listAnswersRequest
     *        Input to list answers.
     * @return Result of the ListAnswers operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.ListAnswers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListAnswers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListAnswersResponse listAnswers(ListAnswersRequest listAnswersRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAnswersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAnswersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAnswersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAnswers");

            return clientHandler.execute(new ClientExecutionParams<ListAnswersRequest, ListAnswersResponse>()
                    .withOperationName("ListAnswers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listAnswersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAnswersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List of answers.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAnswers(software.amazon.awssdk.services.wellarchitected.model.ListAnswersRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListAnswersIterable responses = client.listAnswersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.wellarchitected.paginators.ListAnswersIterable responses = client
     *             .listAnswersPaginator(request);
     *     for (software.amazon.awssdk.services.wellarchitected.model.ListAnswersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListAnswersIterable responses = client.listAnswersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAnswers(software.amazon.awssdk.services.wellarchitected.model.ListAnswersRequest)} operation.</b>
     * </p>
     *
     * @param listAnswersRequest
     *        Input to list answers.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.ListAnswers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListAnswers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListAnswersIterable listAnswersPaginator(ListAnswersRequest listAnswersRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, WellArchitectedException {
        return new ListAnswersIterable(this, applyPaginatorUserAgent(listAnswersRequest));
    }

    /**
     * <p>
     * List lens review improvements.
     * </p>
     *
     * @param listLensReviewImprovementsRequest
     *        Input to list lens review improvements.
     * @return Result of the ListLensReviewImprovements operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.ListLensReviewImprovements
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListLensReviewImprovements"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListLensReviewImprovementsResponse listLensReviewImprovements(
            ListLensReviewImprovementsRequest listLensReviewImprovementsRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListLensReviewImprovementsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListLensReviewImprovementsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLensReviewImprovementsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLensReviewImprovements");

            return clientHandler
                    .execute(new ClientExecutionParams<ListLensReviewImprovementsRequest, ListLensReviewImprovementsResponse>()
                            .withOperationName("ListLensReviewImprovements").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listLensReviewImprovementsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListLensReviewImprovementsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List lens review improvements.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listLensReviewImprovements(software.amazon.awssdk.services.wellarchitected.model.ListLensReviewImprovementsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListLensReviewImprovementsIterable responses = client.listLensReviewImprovementsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.wellarchitected.paginators.ListLensReviewImprovementsIterable responses = client
     *             .listLensReviewImprovementsPaginator(request);
     *     for (software.amazon.awssdk.services.wellarchitected.model.ListLensReviewImprovementsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListLensReviewImprovementsIterable responses = client.listLensReviewImprovementsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLensReviewImprovements(software.amazon.awssdk.services.wellarchitected.model.ListLensReviewImprovementsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listLensReviewImprovementsRequest
     *        Input to list lens review improvements.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.ListLensReviewImprovements
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListLensReviewImprovements"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListLensReviewImprovementsIterable listLensReviewImprovementsPaginator(
            ListLensReviewImprovementsRequest listLensReviewImprovementsRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, WellArchitectedException {
        return new ListLensReviewImprovementsIterable(this, applyPaginatorUserAgent(listLensReviewImprovementsRequest));
    }

    /**
     * <p>
     * List lens reviews.
     * </p>
     *
     * @param listLensReviewsRequest
     *        Input to list lens reviews.
     * @return Result of the ListLensReviews operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.ListLensReviews
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListLensReviews"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListLensReviewsResponse listLensReviews(ListLensReviewsRequest listLensReviewsRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListLensReviewsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListLensReviewsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLensReviewsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLensReviews");

            return clientHandler.execute(new ClientExecutionParams<ListLensReviewsRequest, ListLensReviewsResponse>()
                    .withOperationName("ListLensReviews").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listLensReviewsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListLensReviewsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List lens reviews.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listLensReviews(software.amazon.awssdk.services.wellarchitected.model.ListLensReviewsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListLensReviewsIterable responses = client.listLensReviewsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.wellarchitected.paginators.ListLensReviewsIterable responses = client
     *             .listLensReviewsPaginator(request);
     *     for (software.amazon.awssdk.services.wellarchitected.model.ListLensReviewsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListLensReviewsIterable responses = client.listLensReviewsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLensReviews(software.amazon.awssdk.services.wellarchitected.model.ListLensReviewsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listLensReviewsRequest
     *        Input to list lens reviews.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.ListLensReviews
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListLensReviews"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListLensReviewsIterable listLensReviewsPaginator(ListLensReviewsRequest listLensReviewsRequest)
            throws ValidationException, InternalServerException, ResourceNotFoundException, AccessDeniedException,
            ThrottlingException, AwsServiceException, SdkClientException, WellArchitectedException {
        return new ListLensReviewsIterable(this, applyPaginatorUserAgent(listLensReviewsRequest));
    }

    /**
     * <p>
     * List the available lenses.
     * </p>
     *
     * @param listLensesRequest
     *        Input to list lenses.
     * @return Result of the ListLenses operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.ListLenses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListLenses" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListLensesResponse listLenses(ListLensesRequest listLensesRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException,
            WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListLensesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListLensesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLensesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLenses");

            return clientHandler
                    .execute(new ClientExecutionParams<ListLensesRequest, ListLensesResponse>().withOperationName("ListLenses")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listLensesRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListLensesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List the available lenses.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listLenses(software.amazon.awssdk.services.wellarchitected.model.ListLensesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListLensesIterable responses = client.listLensesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.wellarchitected.paginators.ListLensesIterable responses = client.listLensesPaginator(request);
     *     for (software.amazon.awssdk.services.wellarchitected.model.ListLensesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListLensesIterable responses = client.listLensesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLenses(software.amazon.awssdk.services.wellarchitected.model.ListLensesRequest)} operation.</b>
     * </p>
     *
     * @param listLensesRequest
     *        Input to list lenses.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.ListLenses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListLenses" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListLensesIterable listLensesPaginator(ListLensesRequest listLensesRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException,
            WellArchitectedException {
        return new ListLensesIterable(this, applyPaginatorUserAgent(listLensesRequest));
    }

    /**
     * <p>
     * List all milestones for an existing workload.
     * </p>
     *
     * @param listMilestonesRequest
     *        Input to list all milestones for a workload.
     * @return Result of the ListMilestones operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.ListMilestones
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListMilestones" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListMilestonesResponse listMilestones(ListMilestonesRequest listMilestonesRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListMilestonesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListMilestonesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMilestonesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMilestones");

            return clientHandler.execute(new ClientExecutionParams<ListMilestonesRequest, ListMilestonesResponse>()
                    .withOperationName("ListMilestones").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listMilestonesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListMilestonesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List all milestones for an existing workload.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMilestones(software.amazon.awssdk.services.wellarchitected.model.ListMilestonesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListMilestonesIterable responses = client.listMilestonesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.wellarchitected.paginators.ListMilestonesIterable responses = client
     *             .listMilestonesPaginator(request);
     *     for (software.amazon.awssdk.services.wellarchitected.model.ListMilestonesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListMilestonesIterable responses = client.listMilestonesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMilestones(software.amazon.awssdk.services.wellarchitected.model.ListMilestonesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listMilestonesRequest
     *        Input to list all milestones for a workload.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.ListMilestones
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListMilestones" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListMilestonesIterable listMilestonesPaginator(ListMilestonesRequest listMilestonesRequest)
            throws ValidationException, InternalServerException, ResourceNotFoundException, AccessDeniedException,
            ThrottlingException, AwsServiceException, SdkClientException, WellArchitectedException {
        return new ListMilestonesIterable(this, applyPaginatorUserAgent(listMilestonesRequest));
    }

    /**
     * <p>
     * List lens notifications.
     * </p>
     *
     * @param listNotificationsRequest
     * @return Result of the ListNotifications operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.ListNotifications
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListNotifications"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListNotificationsResponse listNotifications(ListNotificationsRequest listNotificationsRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListNotificationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListNotificationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNotificationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNotifications");

            return clientHandler.execute(new ClientExecutionParams<ListNotificationsRequest, ListNotificationsResponse>()
                    .withOperationName("ListNotifications").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listNotificationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListNotificationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List lens notifications.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listNotifications(software.amazon.awssdk.services.wellarchitected.model.ListNotificationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListNotificationsIterable responses = client.listNotificationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.wellarchitected.paginators.ListNotificationsIterable responses = client
     *             .listNotificationsPaginator(request);
     *     for (software.amazon.awssdk.services.wellarchitected.model.ListNotificationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListNotificationsIterable responses = client.listNotificationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNotifications(software.amazon.awssdk.services.wellarchitected.model.ListNotificationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listNotificationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.ListNotifications
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListNotifications"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListNotificationsIterable listNotificationsPaginator(ListNotificationsRequest listNotificationsRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, WellArchitectedException {
        return new ListNotificationsIterable(this, applyPaginatorUserAgent(listNotificationsRequest));
    }

    /**
     * <p>
     * List the workload invitations.
     * </p>
     *
     * @param listShareInvitationsRequest
     *        Input for List Share Invitations
     * @return Result of the ListShareInvitations operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.ListShareInvitations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListShareInvitations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListShareInvitationsResponse listShareInvitations(ListShareInvitationsRequest listShareInvitationsRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListShareInvitationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListShareInvitationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listShareInvitationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListShareInvitations");

            return clientHandler.execute(new ClientExecutionParams<ListShareInvitationsRequest, ListShareInvitationsResponse>()
                    .withOperationName("ListShareInvitations").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listShareInvitationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListShareInvitationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List the workload invitations.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listShareInvitations(software.amazon.awssdk.services.wellarchitected.model.ListShareInvitationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListShareInvitationsIterable responses = client.listShareInvitationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.wellarchitected.paginators.ListShareInvitationsIterable responses = client
     *             .listShareInvitationsPaginator(request);
     *     for (software.amazon.awssdk.services.wellarchitected.model.ListShareInvitationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListShareInvitationsIterable responses = client.listShareInvitationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listShareInvitations(software.amazon.awssdk.services.wellarchitected.model.ListShareInvitationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listShareInvitationsRequest
     *        Input for List Share Invitations
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.ListShareInvitations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListShareInvitations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListShareInvitationsIterable listShareInvitationsPaginator(ListShareInvitationsRequest listShareInvitationsRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, WellArchitectedException {
        return new ListShareInvitationsIterable(this, applyPaginatorUserAgent(listShareInvitationsRequest));
    }

    /**
     * <p>
     * List the workload shares associated with the workload.
     * </p>
     *
     * @param listWorkloadSharesRequest
     *        Input for List Workload Share
     * @return Result of the ListWorkloadShares operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.ListWorkloadShares
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListWorkloadShares"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListWorkloadSharesResponse listWorkloadShares(ListWorkloadSharesRequest listWorkloadSharesRequest)
            throws ValidationException, InternalServerException, ResourceNotFoundException, AccessDeniedException,
            ThrottlingException, AwsServiceException, SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListWorkloadSharesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListWorkloadSharesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkloadSharesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkloadShares");

            return clientHandler.execute(new ClientExecutionParams<ListWorkloadSharesRequest, ListWorkloadSharesResponse>()
                    .withOperationName("ListWorkloadShares").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listWorkloadSharesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListWorkloadSharesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List the workload shares associated with the workload.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listWorkloadShares(software.amazon.awssdk.services.wellarchitected.model.ListWorkloadSharesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListWorkloadSharesIterable responses = client.listWorkloadSharesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.wellarchitected.paginators.ListWorkloadSharesIterable responses = client
     *             .listWorkloadSharesPaginator(request);
     *     for (software.amazon.awssdk.services.wellarchitected.model.ListWorkloadSharesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListWorkloadSharesIterable responses = client.listWorkloadSharesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listWorkloadShares(software.amazon.awssdk.services.wellarchitected.model.ListWorkloadSharesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listWorkloadSharesRequest
     *        Input for List Workload Share
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.ListWorkloadShares
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListWorkloadShares"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListWorkloadSharesIterable listWorkloadSharesPaginator(ListWorkloadSharesRequest listWorkloadSharesRequest)
            throws ValidationException, InternalServerException, ResourceNotFoundException, AccessDeniedException,
            ThrottlingException, AwsServiceException, SdkClientException, WellArchitectedException {
        return new ListWorkloadSharesIterable(this, applyPaginatorUserAgent(listWorkloadSharesRequest));
    }

    /**
     * <p>
     * List workloads. Paginated.
     * </p>
     *
     * @param listWorkloadsRequest
     *        Input to list all workloads.
     * @return Result of the ListWorkloads operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.ListWorkloads
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListWorkloads" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListWorkloadsResponse listWorkloads(ListWorkloadsRequest listWorkloadsRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException,
            WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListWorkloadsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListWorkloadsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkloadsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkloads");

            return clientHandler.execute(new ClientExecutionParams<ListWorkloadsRequest, ListWorkloadsResponse>()
                    .withOperationName("ListWorkloads").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listWorkloadsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListWorkloadsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List workloads. Paginated.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listWorkloads(software.amazon.awssdk.services.wellarchitected.model.ListWorkloadsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListWorkloadsIterable responses = client.listWorkloadsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.wellarchitected.paginators.ListWorkloadsIterable responses = client
     *             .listWorkloadsPaginator(request);
     *     for (software.amazon.awssdk.services.wellarchitected.model.ListWorkloadsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListWorkloadsIterable responses = client.listWorkloadsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listWorkloads(software.amazon.awssdk.services.wellarchitected.model.ListWorkloadsRequest)} operation.</b>
     * </p>
     *
     * @param listWorkloadsRequest
     *        Input to list all workloads.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.ListWorkloads
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListWorkloads" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListWorkloadsIterable listWorkloadsPaginator(ListWorkloadsRequest listWorkloadsRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException,
            WellArchitectedException {
        return new ListWorkloadsIterable(this, applyPaginatorUserAgent(listWorkloadsRequest));
    }

    /**
     * <p>
     * Update the answer.
     * </p>
     *
     * @param updateAnswerRequest
     *        Input to update answer.
     * @return Result of the UpdateAnswer operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws ConflictException
     *         The resource already exists.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.UpdateAnswer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/UpdateAnswer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateAnswerResponse updateAnswer(UpdateAnswerRequest updateAnswerRequest) throws ValidationException,
            ResourceNotFoundException, ConflictException, InternalServerException, AccessDeniedException, ThrottlingException,
            AwsServiceException, SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateAnswerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateAnswerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAnswerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAnswer");

            return clientHandler.execute(new ClientExecutionParams<UpdateAnswerRequest, UpdateAnswerResponse>()
                    .withOperationName("UpdateAnswer").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateAnswerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateAnswerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update lens review.
     * </p>
     *
     * @param updateLensReviewRequest
     *        Input for update lens review.
     * @return Result of the UpdateLensReview operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws ConflictException
     *         The resource already exists.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.UpdateLensReview
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/UpdateLensReview"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateLensReviewResponse updateLensReview(UpdateLensReviewRequest updateLensReviewRequest) throws ValidationException,
            ResourceNotFoundException, ConflictException, InternalServerException, AccessDeniedException, ThrottlingException,
            AwsServiceException, SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateLensReviewResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateLensReviewResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLensReviewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLensReview");

            return clientHandler.execute(new ClientExecutionParams<UpdateLensReviewRequest, UpdateLensReviewResponse>()
                    .withOperationName("UpdateLensReview").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateLensReviewRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateLensReviewRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update a workload invitation.
     * </p>
     *
     * @param updateShareInvitationRequest
     *        Input for Update Share Invitation
     * @return Result of the UpdateShareInvitation operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws ConflictException
     *         The resource already exists.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.UpdateShareInvitation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/UpdateShareInvitation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateShareInvitationResponse updateShareInvitation(UpdateShareInvitationRequest updateShareInvitationRequest)
            throws ValidationException, InternalServerException, ResourceNotFoundException, ConflictException,
            AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateShareInvitationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateShareInvitationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateShareInvitationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateShareInvitation");

            return clientHandler.execute(new ClientExecutionParams<UpdateShareInvitationRequest, UpdateShareInvitationResponse>()
                    .withOperationName("UpdateShareInvitation").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateShareInvitationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateShareInvitationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update an existing workload.
     * </p>
     *
     * @param updateWorkloadRequest
     *        Input to update a workload.
     * @return Result of the UpdateWorkload operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws ConflictException
     *         The resource already exists.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.UpdateWorkload
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/UpdateWorkload" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateWorkloadResponse updateWorkload(UpdateWorkloadRequest updateWorkloadRequest) throws ValidationException,
            ResourceNotFoundException, ConflictException, InternalServerException, AccessDeniedException, ThrottlingException,
            AwsServiceException, SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateWorkloadResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateWorkloadResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateWorkloadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWorkload");

            return clientHandler.execute(new ClientExecutionParams<UpdateWorkloadRequest, UpdateWorkloadResponse>()
                    .withOperationName("UpdateWorkload").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateWorkloadRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateWorkloadRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update a workload share.
     * </p>
     *
     * @param updateWorkloadShareRequest
     *        Input for Update Workload Share
     * @return Result of the UpdateWorkloadShare operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws ConflictException
     *         The resource already exists.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.UpdateWorkloadShare
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/UpdateWorkloadShare"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateWorkloadShareResponse updateWorkloadShare(UpdateWorkloadShareRequest updateWorkloadShareRequest)
            throws ValidationException, InternalServerException, ResourceNotFoundException, ConflictException,
            AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateWorkloadShareResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateWorkloadShareResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateWorkloadShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWorkloadShare");

            return clientHandler.execute(new ClientExecutionParams<UpdateWorkloadShareRequest, UpdateWorkloadShareResponse>()
                    .withOperationName("UpdateWorkloadShare").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateWorkloadShareRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateWorkloadShareRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Upgrade lens review.
     * </p>
     *
     * @param upgradeLensReviewRequest
     * @return Result of the UpgradeLensReview operation returned by the service.
     * @throws ValidationException
     *         The user input is not valid.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws ConflictException
     *         The resource already exists.
     * @throws InternalServerException
     *         There is a problem with the AWS Well-Architected Tool API service.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WellArchitectedException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WellArchitectedClient.UpgradeLensReview
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/UpgradeLensReview"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpgradeLensReviewResponse upgradeLensReview(UpgradeLensReviewRequest upgradeLensReviewRequest)
            throws ValidationException, ResourceNotFoundException, ConflictException, InternalServerException,
            AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException, WellArchitectedException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpgradeLensReviewResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpgradeLensReviewResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, upgradeLensReviewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpgradeLensReview");

            return clientHandler.execute(new ClientExecutionParams<UpgradeLensReviewRequest, UpgradeLensReviewResponse>()
                    .withOperationName("UpgradeLensReview").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(upgradeLensReviewRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpgradeLensReviewRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(WellArchitectedException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends WellArchitectedRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
