/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies that WAF should run a <code>Challenge</code> check against the request to verify that the request is coming
 * from a legitimate client session:
 * </p>
 * <ul>
 * <li>
 * <p>
 * If the request includes a valid, unexpired challenge token, WAF applies any custom request handling and labels that
 * you've configured and then allows the web request inspection to proceed to the next rule, similar to a
 * <code>CountAction</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * If the request doesn't include a valid, unexpired challenge token, WAF discontinues the web ACL evaluation of the
 * request and blocks it from going to its intended destination.
 * </p>
 * <p>
 * WAF then generates a challenge response that it sends back to the client, which includes the following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * The header <code>x-amzn-waf-action</code> with a value of <code>challenge</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * The HTTP status code <code>202 Request Accepted</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * If the request contains an <code>Accept</code> header with a value of <code>text/html</code>, the response includes a
 * JavaScript page interstitial with a challenge script.
 * </p>
 * </li>
 * </ul>
 * <p>
 * Challenges run silent browser interrogations in the background, and don't generally affect the end user experience.
 * </p>
 * <p>
 * A challenge enforces token acquisition using an interstitial JavaScript challenge that inspects the client session
 * for legitimate behavior. The challenge blocks bots or at least increases the cost of operating sophisticated bots.
 * </p>
 * <p>
 * After the client session successfully responds to the challenge, it receives a new token from WAF, which the
 * challenge script uses to resubmit the original request.
 * </p>
 * </li>
 * </ul>
 * <p>
 * You can configure the expiration time in the <code>ChallengeConfig</code> <code>ImmunityTimeProperty</code> setting
 * at the rule and web ACL level. The rule setting overrides the web ACL setting.
 * </p>
 * <p>
 * This action option is available for rules. It isn't available for web ACL default actions.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ChallengeAction implements SdkPojo, Serializable, ToCopyableBuilder<ChallengeAction.Builder, ChallengeAction> {
    private static final SdkField<CustomRequestHandling> CUSTOM_REQUEST_HANDLING_FIELD = SdkField
            .<CustomRequestHandling> builder(MarshallingType.SDK_POJO).memberName("CustomRequestHandling")
            .getter(getter(ChallengeAction::customRequestHandling)).setter(setter(Builder::customRequestHandling))
            .constructor(CustomRequestHandling::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomRequestHandling").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections
            .unmodifiableList(Arrays.asList(CUSTOM_REQUEST_HANDLING_FIELD));

    private static final long serialVersionUID = 1L;

    private final CustomRequestHandling customRequestHandling;

    private ChallengeAction(BuilderImpl builder) {
        this.customRequestHandling = builder.customRequestHandling;
    }

    /**
     * <p>
     * Defines custom handling for the web request, used when the challenge inspection determines that the request's
     * token is valid and unexpired.
     * </p>
     * <p>
     * For information about customizing web requests and responses, see <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-custom-request-response.html">Customizing web
     * requests and responses in WAF</a> in the <i>WAF Developer Guide</i>.
     * </p>
     * 
     * @return Defines custom handling for the web request, used when the challenge inspection determines that the
     *         request's token is valid and unexpired.</p>
     *         <p>
     *         For information about customizing web requests and responses, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-custom-request-response.html">Customizing
     *         web requests and responses in WAF</a> in the <i>WAF Developer Guide</i>.
     */
    public final CustomRequestHandling customRequestHandling() {
        return customRequestHandling;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(customRequestHandling());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ChallengeAction)) {
            return false;
        }
        ChallengeAction other = (ChallengeAction) obj;
        return Objects.equals(customRequestHandling(), other.customRequestHandling());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ChallengeAction").add("CustomRequestHandling", customRequestHandling()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CustomRequestHandling":
            return Optional.ofNullable(clazz.cast(customRequestHandling()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ChallengeAction, T> g) {
        return obj -> g.apply((ChallengeAction) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ChallengeAction> {
        /**
         * <p>
         * Defines custom handling for the web request, used when the challenge inspection determines that the request's
         * token is valid and unexpired.
         * </p>
         * <p>
         * For information about customizing web requests and responses, see <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-custom-request-response.html">Customizing web
         * requests and responses in WAF</a> in the <i>WAF Developer Guide</i>.
         * </p>
         * 
         * @param customRequestHandling
         *        Defines custom handling for the web request, used when the challenge inspection determines that the
         *        request's token is valid and unexpired.</p>
         *        <p>
         *        For information about customizing web requests and responses, see <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-custom-request-response.html"
         *        >Customizing web requests and responses in WAF</a> in the <i>WAF Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customRequestHandling(CustomRequestHandling customRequestHandling);

        /**
         * <p>
         * Defines custom handling for the web request, used when the challenge inspection determines that the request's
         * token is valid and unexpired.
         * </p>
         * <p>
         * For information about customizing web requests and responses, see <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-custom-request-response.html">Customizing web
         * requests and responses in WAF</a> in the <i>WAF Developer Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link CustomRequestHandling.Builder} avoiding
         * the need to create one manually via {@link CustomRequestHandling#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CustomRequestHandling.Builder#build()} is called immediately and
         * its result is passed to {@link #customRequestHandling(CustomRequestHandling)}.
         * 
         * @param customRequestHandling
         *        a consumer that will call methods on {@link CustomRequestHandling.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #customRequestHandling(CustomRequestHandling)
         */
        default Builder customRequestHandling(Consumer<CustomRequestHandling.Builder> customRequestHandling) {
            return customRequestHandling(CustomRequestHandling.builder().applyMutation(customRequestHandling).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private CustomRequestHandling customRequestHandling;

        private BuilderImpl() {
        }

        private BuilderImpl(ChallengeAction model) {
            customRequestHandling(model.customRequestHandling);
        }

        public final CustomRequestHandling.Builder getCustomRequestHandling() {
            return customRequestHandling != null ? customRequestHandling.toBuilder() : null;
        }

        public final void setCustomRequestHandling(CustomRequestHandling.BuilderImpl customRequestHandling) {
            this.customRequestHandling = customRequestHandling != null ? customRequestHandling.build() : null;
        }

        @Override
        public final Builder customRequestHandling(CustomRequestHandling customRequestHandling) {
            this.customRequestHandling = customRequestHandling;
            return this;
        }

        @Override
        public ChallengeAction build() {
            return new ChallengeAction(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
