/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <note>
 * <p>
 * This is the latest version of <b>AWS WAF</b>, named AWS WAFV2, released in November, 2019. For information, including
 * how to migrate your AWS WAF resources from the prior release, see the <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">AWS WAF Developer Guide</a>.
 * </p>
 * </note>
 * <p>
 * A rule statement that compares a number of bytes against the size of a request component, using a comparison
 * operator, such as greater than (&gt;) or less than (&lt;). For example, you can use a size constraint statement to
 * look for query strings that are longer than 100 bytes.
 * </p>
 * <p>
 * If you configure AWS WAF to inspect the request body, AWS WAF inspects only the first 8192 bytes (8 KB). If the
 * request body for your web requests never exceeds 8192 bytes, you can create a size constraint condition and block
 * requests that have a request body greater than 8192 bytes.
 * </p>
 * <p>
 * If you choose URI for the value of Part of the request to filter on, the slash (/) in the URI counts as one
 * character. For example, the URI <code>/logo.jpg</code> is nine characters long.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SizeConstraintStatement implements SdkPojo, Serializable,
        ToCopyableBuilder<SizeConstraintStatement.Builder, SizeConstraintStatement> {
    private static final SdkField<FieldToMatch> FIELD_TO_MATCH_FIELD = SdkField.<FieldToMatch> builder(MarshallingType.SDK_POJO)
            .getter(getter(SizeConstraintStatement::fieldToMatch)).setter(setter(Builder::fieldToMatch))
            .constructor(FieldToMatch::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FieldToMatch").build()).build();

    private static final SdkField<String> COMPARISON_OPERATOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SizeConstraintStatement::comparisonOperatorAsString)).setter(setter(Builder::comparisonOperator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComparisonOperator").build())
            .build();

    private static final SdkField<Long> SIZE_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .getter(getter(SizeConstraintStatement::size)).setter(setter(Builder::size))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Size").build()).build();

    private static final SdkField<List<TextTransformation>> TEXT_TRANSFORMATIONS_FIELD = SdkField
            .<List<TextTransformation>> builder(MarshallingType.LIST)
            .getter(getter(SizeConstraintStatement::textTransformations))
            .setter(setter(Builder::textTransformations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TextTransformations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TextTransformation> builder(MarshallingType.SDK_POJO)
                                            .constructor(TextTransformation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FIELD_TO_MATCH_FIELD,
            COMPARISON_OPERATOR_FIELD, SIZE_FIELD, TEXT_TRANSFORMATIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final FieldToMatch fieldToMatch;

    private final String comparisonOperator;

    private final Long size;

    private final List<TextTransformation> textTransformations;

    private SizeConstraintStatement(BuilderImpl builder) {
        this.fieldToMatch = builder.fieldToMatch;
        this.comparisonOperator = builder.comparisonOperator;
        this.size = builder.size;
        this.textTransformations = builder.textTransformations;
    }

    /**
     * <p>
     * The part of a web request that you want AWS WAF to inspect. For more information, see <a>FieldToMatch</a>.
     * </p>
     * 
     * @return The part of a web request that you want AWS WAF to inspect. For more information, see
     *         <a>FieldToMatch</a>.
     */
    public FieldToMatch fieldToMatch() {
        return fieldToMatch;
    }

    /**
     * <p>
     * The operator to use to compare the request part to the size setting.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #comparisonOperator} will return {@link ComparisonOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #comparisonOperatorAsString}.
     * </p>
     * 
     * @return The operator to use to compare the request part to the size setting.
     * @see ComparisonOperator
     */
    public ComparisonOperator comparisonOperator() {
        return ComparisonOperator.fromValue(comparisonOperator);
    }

    /**
     * <p>
     * The operator to use to compare the request part to the size setting.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #comparisonOperator} will return {@link ComparisonOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #comparisonOperatorAsString}.
     * </p>
     * 
     * @return The operator to use to compare the request part to the size setting.
     * @see ComparisonOperator
     */
    public String comparisonOperatorAsString() {
        return comparisonOperator;
    }

    /**
     * <p>
     * The size, in byte, to compare to the request part, after any transformations.
     * </p>
     * 
     * @return The size, in byte, to compare to the request part, after any transformations.
     */
    public Long size() {
        return size;
    }

    /**
     * Returns true if the TextTransformations property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasTextTransformations() {
        return textTransformations != null && !(textTransformations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Text transformations eliminate some of the unusual formatting that attackers use in web requests in an effort to
     * bypass detection. If you specify one or more transformations in a rule statement, AWS WAF performs all
     * transformations on the content of the request component identified by <code>FieldToMatch</code>, starting from
     * the lowest priority setting, before inspecting the content for a match.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTextTransformations()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Text transformations eliminate some of the unusual formatting that attackers use in web requests in an
     *         effort to bypass detection. If you specify one or more transformations in a rule statement, AWS WAF
     *         performs all transformations on the content of the request component identified by
     *         <code>FieldToMatch</code>, starting from the lowest priority setting, before inspecting the content for a
     *         match.
     */
    public List<TextTransformation> textTransformations() {
        return textTransformations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(fieldToMatch());
        hashCode = 31 * hashCode + Objects.hashCode(comparisonOperatorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(size());
        hashCode = 31 * hashCode + Objects.hashCode(textTransformations());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SizeConstraintStatement)) {
            return false;
        }
        SizeConstraintStatement other = (SizeConstraintStatement) obj;
        return Objects.equals(fieldToMatch(), other.fieldToMatch())
                && Objects.equals(comparisonOperatorAsString(), other.comparisonOperatorAsString())
                && Objects.equals(size(), other.size()) && Objects.equals(textTransformations(), other.textTransformations());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("SizeConstraintStatement").add("FieldToMatch", fieldToMatch())
                .add("ComparisonOperator", comparisonOperatorAsString()).add("Size", size())
                .add("TextTransformations", textTransformations()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FieldToMatch":
            return Optional.ofNullable(clazz.cast(fieldToMatch()));
        case "ComparisonOperator":
            return Optional.ofNullable(clazz.cast(comparisonOperatorAsString()));
        case "Size":
            return Optional.ofNullable(clazz.cast(size()));
        case "TextTransformations":
            return Optional.ofNullable(clazz.cast(textTransformations()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SizeConstraintStatement, T> g) {
        return obj -> g.apply((SizeConstraintStatement) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SizeConstraintStatement> {
        /**
         * <p>
         * The part of a web request that you want AWS WAF to inspect. For more information, see <a>FieldToMatch</a>.
         * </p>
         * 
         * @param fieldToMatch
         *        The part of a web request that you want AWS WAF to inspect. For more information, see
         *        <a>FieldToMatch</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fieldToMatch(FieldToMatch fieldToMatch);

        /**
         * <p>
         * The part of a web request that you want AWS WAF to inspect. For more information, see <a>FieldToMatch</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link FieldToMatch.Builder} avoiding the need to
         * create one manually via {@link FieldToMatch#builder()}.
         *
         * When the {@link Consumer} completes, {@link FieldToMatch.Builder#build()} is called immediately and its
         * result is passed to {@link #fieldToMatch(FieldToMatch)}.
         * 
         * @param fieldToMatch
         *        a consumer that will call methods on {@link FieldToMatch.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #fieldToMatch(FieldToMatch)
         */
        default Builder fieldToMatch(Consumer<FieldToMatch.Builder> fieldToMatch) {
            return fieldToMatch(FieldToMatch.builder().applyMutation(fieldToMatch).build());
        }

        /**
         * <p>
         * The operator to use to compare the request part to the size setting.
         * </p>
         * 
         * @param comparisonOperator
         *        The operator to use to compare the request part to the size setting.
         * @see ComparisonOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperator
         */
        Builder comparisonOperator(String comparisonOperator);

        /**
         * <p>
         * The operator to use to compare the request part to the size setting.
         * </p>
         * 
         * @param comparisonOperator
         *        The operator to use to compare the request part to the size setting.
         * @see ComparisonOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperator
         */
        Builder comparisonOperator(ComparisonOperator comparisonOperator);

        /**
         * <p>
         * The size, in byte, to compare to the request part, after any transformations.
         * </p>
         * 
         * @param size
         *        The size, in byte, to compare to the request part, after any transformations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder size(Long size);

        /**
         * <p>
         * Text transformations eliminate some of the unusual formatting that attackers use in web requests in an effort
         * to bypass detection. If you specify one or more transformations in a rule statement, AWS WAF performs all
         * transformations on the content of the request component identified by <code>FieldToMatch</code>, starting
         * from the lowest priority setting, before inspecting the content for a match.
         * </p>
         * 
         * @param textTransformations
         *        Text transformations eliminate some of the unusual formatting that attackers use in web requests in an
         *        effort to bypass detection. If you specify one or more transformations in a rule statement, AWS WAF
         *        performs all transformations on the content of the request component identified by
         *        <code>FieldToMatch</code>, starting from the lowest priority setting, before inspecting the content
         *        for a match.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder textTransformations(Collection<TextTransformation> textTransformations);

        /**
         * <p>
         * Text transformations eliminate some of the unusual formatting that attackers use in web requests in an effort
         * to bypass detection. If you specify one or more transformations in a rule statement, AWS WAF performs all
         * transformations on the content of the request component identified by <code>FieldToMatch</code>, starting
         * from the lowest priority setting, before inspecting the content for a match.
         * </p>
         * 
         * @param textTransformations
         *        Text transformations eliminate some of the unusual formatting that attackers use in web requests in an
         *        effort to bypass detection. If you specify one or more transformations in a rule statement, AWS WAF
         *        performs all transformations on the content of the request component identified by
         *        <code>FieldToMatch</code>, starting from the lowest priority setting, before inspecting the content
         *        for a match.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder textTransformations(TextTransformation... textTransformations);

        /**
         * <p>
         * Text transformations eliminate some of the unusual formatting that attackers use in web requests in an effort
         * to bypass detection. If you specify one or more transformations in a rule statement, AWS WAF performs all
         * transformations on the content of the request component identified by <code>FieldToMatch</code>, starting
         * from the lowest priority setting, before inspecting the content for a match.
         * </p>
         * This is a convenience that creates an instance of the {@link List<TextTransformation>.Builder} avoiding the
         * need to create one manually via {@link List<TextTransformation>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<TextTransformation>.Builder#build()} is called immediately
         * and its result is passed to {@link #textTransformations(List<TextTransformation>)}.
         * 
         * @param textTransformations
         *        a consumer that will call methods on {@link List<TextTransformation>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #textTransformations(List<TextTransformation>)
         */
        Builder textTransformations(Consumer<TextTransformation.Builder>... textTransformations);
    }

    static final class BuilderImpl implements Builder {
        private FieldToMatch fieldToMatch;

        private String comparisonOperator;

        private Long size;

        private List<TextTransformation> textTransformations = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(SizeConstraintStatement model) {
            fieldToMatch(model.fieldToMatch);
            comparisonOperator(model.comparisonOperator);
            size(model.size);
            textTransformations(model.textTransformations);
        }

        public final FieldToMatch.Builder getFieldToMatch() {
            return fieldToMatch != null ? fieldToMatch.toBuilder() : null;
        }

        @Override
        public final Builder fieldToMatch(FieldToMatch fieldToMatch) {
            this.fieldToMatch = fieldToMatch;
            return this;
        }

        public final void setFieldToMatch(FieldToMatch.BuilderImpl fieldToMatch) {
            this.fieldToMatch = fieldToMatch != null ? fieldToMatch.build() : null;
        }

        public final String getComparisonOperator() {
            return comparisonOperator;
        }

        @Override
        public final Builder comparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
            return this;
        }

        @Override
        public final Builder comparisonOperator(ComparisonOperator comparisonOperator) {
            this.comparisonOperator(comparisonOperator == null ? null : comparisonOperator.toString());
            return this;
        }

        public final void setComparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
        }

        public final Long getSize() {
            return size;
        }

        @Override
        public final Builder size(Long size) {
            this.size = size;
            return this;
        }

        public final void setSize(Long size) {
            this.size = size;
        }

        public final Collection<TextTransformation.Builder> getTextTransformations() {
            return textTransformations != null ? textTransformations.stream().map(TextTransformation::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder textTransformations(Collection<TextTransformation> textTransformations) {
            this.textTransformations = TextTransformationsCopier.copy(textTransformations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder textTransformations(TextTransformation... textTransformations) {
            textTransformations(Arrays.asList(textTransformations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder textTransformations(Consumer<TextTransformation.Builder>... textTransformations) {
            textTransformations(Stream.of(textTransformations).map(c -> TextTransformation.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setTextTransformations(Collection<TextTransformation.BuilderImpl> textTransformations) {
            this.textTransformations = TextTransformationsCopier.copyFromBuilder(textTransformations);
        }

        @Override
        public SizeConstraintStatement build() {
            return new SizeConstraintStatement(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
