/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <note>
 * <p>
 * This is the latest version of <b>AWS WAF</b>, named AWS WAFV2, released in November, 2019. For information, including
 * how to migrate your AWS WAF resources from the prior release, see the <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">AWS WAF Developer Guide</a>.
 * </p>
 * </note>
 * <p>
 * A logical rule statement used to combine other rule statements with OR logic. You provide more than one
 * <a>Statement</a> within the <code>OrStatement</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OrStatement implements SdkPojo, Serializable, ToCopyableBuilder<OrStatement.Builder, OrStatement> {
    private static final SdkField<List<Statement>> STATEMENTS_FIELD = SdkField
            .<List<Statement>> builder(MarshallingType.LIST)
            .getter(getter(OrStatement::statements))
            .setter(setter(Builder::statements))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Statements").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Statement> builder(MarshallingType.SDK_POJO)
                                            .constructor(Statement::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATEMENTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<Statement> statements;

    private OrStatement(BuilderImpl builder) {
        this.statements = builder.statements;
    }

    /**
     * Returns true if the Statements property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasStatements() {
        return statements != null && !(statements instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The statements to combine with OR logic. You can use any statements that can be nested.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasStatements()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The statements to combine with OR logic. You can use any statements that can be nested.
     */
    public List<Statement> statements() {
        return statements;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(statements());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OrStatement)) {
            return false;
        }
        OrStatement other = (OrStatement) obj;
        return Objects.equals(statements(), other.statements());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("OrStatement").add("Statements", statements()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Statements":
            return Optional.ofNullable(clazz.cast(statements()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<OrStatement, T> g) {
        return obj -> g.apply((OrStatement) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OrStatement> {
        /**
         * <p>
         * The statements to combine with OR logic. You can use any statements that can be nested.
         * </p>
         * 
         * @param statements
         *        The statements to combine with OR logic. You can use any statements that can be nested.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statements(Collection<Statement> statements);

        /**
         * <p>
         * The statements to combine with OR logic. You can use any statements that can be nested.
         * </p>
         * 
         * @param statements
         *        The statements to combine with OR logic. You can use any statements that can be nested.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statements(Statement... statements);

        /**
         * <p>
         * The statements to combine with OR logic. You can use any statements that can be nested.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Statement>.Builder} avoiding the need to
         * create one manually via {@link List<Statement>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Statement>.Builder#build()} is called immediately and its
         * result is passed to {@link #statements(List<Statement>)}.
         * 
         * @param statements
         *        a consumer that will call methods on {@link List<Statement>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #statements(List<Statement>)
         */
        Builder statements(Consumer<Statement.Builder>... statements);
    }

    static final class BuilderImpl implements Builder {
        private List<Statement> statements = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(OrStatement model) {
            statements(model.statements);
        }

        public final Collection<Statement.Builder> getStatements() {
            return statements != null ? statements.stream().map(Statement::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder statements(Collection<Statement> statements) {
            this.statements = StatementsCopier.copy(statements);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statements(Statement... statements) {
            statements(Arrays.asList(statements));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statements(Consumer<Statement.Builder>... statements) {
            statements(Stream.of(statements).map(c -> Statement.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setStatements(Collection<Statement.BuilderImpl> statements) {
            this.statements = StatementsCopier.copyFromBuilder(statements);
        }

        @Override
        public OrStatement build() {
            return new OrStatement(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
