/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <note>
 * <p>
 * This is the latest version of <b>AWS WAF</b>, named AWS WAFV2, released in November, 2019. For information, including
 * how to migrate your AWS WAF resources from the prior release, see the <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">AWS WAF Developer Guide</a>.
 * </p>
 * </note>
 * <p>
 * Text transformations eliminate some of the unusual formatting that attackers use in web requests in an effort to
 * bypass detection.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TextTransformation implements SdkPojo, Serializable,
        ToCopyableBuilder<TextTransformation.Builder, TextTransformation> {
    private static final SdkField<Integer> PRIORITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(TextTransformation::priority)).setter(setter(Builder::priority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Priority").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(TextTransformation::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PRIORITY_FIELD, TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer priority;

    private final String type;

    private TextTransformation(BuilderImpl builder) {
        this.priority = builder.priority;
        this.type = builder.type;
    }

    /**
     * <p>
     * Sets the relative processing order for multiple transformations that are defined for a rule statement. AWS WAF
     * processes all transformations, from lowest priority to highest, before inspecting the transformed content. The
     * priorities don't need to be consecutive, but they must all be different.
     * </p>
     * 
     * @return Sets the relative processing order for multiple transformations that are defined for a rule statement.
     *         AWS WAF processes all transformations, from lowest priority to highest, before inspecting the transformed
     *         content. The priorities don't need to be consecutive, but they must all be different.
     */
    public Integer priority() {
        return priority;
    }

    /**
     * <p>
     * You can specify the following transformation types:
     * </p>
     * <p>
     * <b>CMD_LINE</b>
     * </p>
     * <p>
     * When you're concerned that attackers are injecting an operating system command line command and using unusual
     * formatting to disguise some or all of the command, use this option to perform the following transformations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Delete the following characters: \ " ' ^
     * </p>
     * </li>
     * <li>
     * <p>
     * Delete spaces before the following characters: / (
     * </p>
     * </li>
     * <li>
     * <p>
     * Replace the following characters with a space: , ;
     * </p>
     * </li>
     * <li>
     * <p>
     * Replace multiple spaces with one space
     * </p>
     * </li>
     * <li>
     * <p>
     * Convert uppercase letters (A-Z) to lowercase (a-z)
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>COMPRESS_WHITE_SPACE</b>
     * </p>
     * <p>
     * Use this option to replace the following characters with a space character (decimal 32):
     * </p>
     * <ul>
     * <li>
     * <p>
     * \f, formfeed, decimal 12
     * </p>
     * </li>
     * <li>
     * <p>
     * \t, tab, decimal 9
     * </p>
     * </li>
     * <li>
     * <p>
     * \n, newline, decimal 10
     * </p>
     * </li>
     * <li>
     * <p>
     * \r, carriage return, decimal 13
     * </p>
     * </li>
     * <li>
     * <p>
     * \v, vertical tab, decimal 11
     * </p>
     * </li>
     * <li>
     * <p>
     * non-breaking space, decimal 160
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
     * </p>
     * <p>
     * <b>HTML_ENTITY_DECODE</b>
     * </p>
     * <p>
     * Use this option to replace HTML-encoded characters with unencoded characters. <code>HTML_ENTITY_DECODE</code>
     * performs the following operations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Replaces <code>(ampersand)quot;</code> with <code>"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)lt;</code> with a "less than" symbol
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
     * corresponding characters
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
     * corresponding characters
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>LOWERCASE</b>
     * </p>
     * <p>
     * Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
     * </p>
     * <p>
     * <b>URL_DECODE</b>
     * </p>
     * <p>
     * Use this option to decode a URL-encoded value.
     * </p>
     * <p>
     * <b>NONE</b>
     * </p>
     * <p>
     * Specify <code>NONE</code> if you don't want any text transformations.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link TextTransformationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return You can specify the following transformation types:</p>
     *         <p>
     *         <b>CMD_LINE</b>
     *         </p>
     *         <p>
     *         When you're concerned that attackers are injecting an operating system command line command and using
     *         unusual formatting to disguise some or all of the command, use this option to perform the following
     *         transformations:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Delete the following characters: \ " ' ^
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delete spaces before the following characters: / (
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replace the following characters with a space: , ;
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replace multiple spaces with one space
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Convert uppercase letters (A-Z) to lowercase (a-z)
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>COMPRESS_WHITE_SPACE</b>
     *         </p>
     *         <p>
     *         Use this option to replace the following characters with a space character (decimal 32):
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         \f, formfeed, decimal 12
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \t, tab, decimal 9
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \n, newline, decimal 10
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \r, carriage return, decimal 13
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \v, vertical tab, decimal 11
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         non-breaking space, decimal 160
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
     *         </p>
     *         <p>
     *         <b>HTML_ENTITY_DECODE</b>
     *         </p>
     *         <p>
     *         Use this option to replace HTML-encoded characters with unencoded characters.
     *         <code>HTML_ENTITY_DECODE</code> performs the following operations:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)quot;</code> with <code>"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)lt;</code> with a "less than" symbol
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
     *         corresponding characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
     *         corresponding characters
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>LOWERCASE</b>
     *         </p>
     *         <p>
     *         Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
     *         </p>
     *         <p>
     *         <b>URL_DECODE</b>
     *         </p>
     *         <p>
     *         Use this option to decode a URL-encoded value.
     *         </p>
     *         <p>
     *         <b>NONE</b>
     *         </p>
     *         <p>
     *         Specify <code>NONE</code> if you don't want any text transformations.
     * @see TextTransformationType
     */
    public TextTransformationType type() {
        return TextTransformationType.fromValue(type);
    }

    /**
     * <p>
     * You can specify the following transformation types:
     * </p>
     * <p>
     * <b>CMD_LINE</b>
     * </p>
     * <p>
     * When you're concerned that attackers are injecting an operating system command line command and using unusual
     * formatting to disguise some or all of the command, use this option to perform the following transformations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Delete the following characters: \ " ' ^
     * </p>
     * </li>
     * <li>
     * <p>
     * Delete spaces before the following characters: / (
     * </p>
     * </li>
     * <li>
     * <p>
     * Replace the following characters with a space: , ;
     * </p>
     * </li>
     * <li>
     * <p>
     * Replace multiple spaces with one space
     * </p>
     * </li>
     * <li>
     * <p>
     * Convert uppercase letters (A-Z) to lowercase (a-z)
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>COMPRESS_WHITE_SPACE</b>
     * </p>
     * <p>
     * Use this option to replace the following characters with a space character (decimal 32):
     * </p>
     * <ul>
     * <li>
     * <p>
     * \f, formfeed, decimal 12
     * </p>
     * </li>
     * <li>
     * <p>
     * \t, tab, decimal 9
     * </p>
     * </li>
     * <li>
     * <p>
     * \n, newline, decimal 10
     * </p>
     * </li>
     * <li>
     * <p>
     * \r, carriage return, decimal 13
     * </p>
     * </li>
     * <li>
     * <p>
     * \v, vertical tab, decimal 11
     * </p>
     * </li>
     * <li>
     * <p>
     * non-breaking space, decimal 160
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
     * </p>
     * <p>
     * <b>HTML_ENTITY_DECODE</b>
     * </p>
     * <p>
     * Use this option to replace HTML-encoded characters with unencoded characters. <code>HTML_ENTITY_DECODE</code>
     * performs the following operations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Replaces <code>(ampersand)quot;</code> with <code>"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)lt;</code> with a "less than" symbol
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
     * corresponding characters
     * </p>
     * </li>
     * <li>
     * <p>
     * Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
     * corresponding characters
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>LOWERCASE</b>
     * </p>
     * <p>
     * Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
     * </p>
     * <p>
     * <b>URL_DECODE</b>
     * </p>
     * <p>
     * Use this option to decode a URL-encoded value.
     * </p>
     * <p>
     * <b>NONE</b>
     * </p>
     * <p>
     * Specify <code>NONE</code> if you don't want any text transformations.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link TextTransformationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return You can specify the following transformation types:</p>
     *         <p>
     *         <b>CMD_LINE</b>
     *         </p>
     *         <p>
     *         When you're concerned that attackers are injecting an operating system command line command and using
     *         unusual formatting to disguise some or all of the command, use this option to perform the following
     *         transformations:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Delete the following characters: \ " ' ^
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Delete spaces before the following characters: / (
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replace the following characters with a space: , ;
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replace multiple spaces with one space
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Convert uppercase letters (A-Z) to lowercase (a-z)
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>COMPRESS_WHITE_SPACE</b>
     *         </p>
     *         <p>
     *         Use this option to replace the following characters with a space character (decimal 32):
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         \f, formfeed, decimal 12
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \t, tab, decimal 9
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \n, newline, decimal 10
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \r, carriage return, decimal 13
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         \v, vertical tab, decimal 11
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         non-breaking space, decimal 160
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
     *         </p>
     *         <p>
     *         <b>HTML_ENTITY_DECODE</b>
     *         </p>
     *         <p>
     *         Use this option to replace HTML-encoded characters with unencoded characters.
     *         <code>HTML_ENTITY_DECODE</code> performs the following operations:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)quot;</code> with <code>"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)lt;</code> with a "less than" symbol
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
     *         corresponding characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
     *         corresponding characters
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>LOWERCASE</b>
     *         </p>
     *         <p>
     *         Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
     *         </p>
     *         <p>
     *         <b>URL_DECODE</b>
     *         </p>
     *         <p>
     *         Use this option to decode a URL-encoded value.
     *         </p>
     *         <p>
     *         <b>NONE</b>
     *         </p>
     *         <p>
     *         Specify <code>NONE</code> if you don't want any text transformations.
     * @see TextTransformationType
     */
    public String typeAsString() {
        return type;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TextTransformation)) {
            return false;
        }
        TextTransformation other = (TextTransformation) obj;
        return Objects.equals(priority(), other.priority()) && Objects.equals(typeAsString(), other.typeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("TextTransformation").add("Priority", priority()).add("Type", typeAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Priority":
            return Optional.ofNullable(clazz.cast(priority()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TextTransformation, T> g) {
        return obj -> g.apply((TextTransformation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TextTransformation> {
        /**
         * <p>
         * Sets the relative processing order for multiple transformations that are defined for a rule statement. AWS
         * WAF processes all transformations, from lowest priority to highest, before inspecting the transformed
         * content. The priorities don't need to be consecutive, but they must all be different.
         * </p>
         * 
         * @param priority
         *        Sets the relative processing order for multiple transformations that are defined for a rule statement.
         *        AWS WAF processes all transformations, from lowest priority to highest, before inspecting the
         *        transformed content. The priorities don't need to be consecutive, but they must all be different.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(Integer priority);

        /**
         * <p>
         * You can specify the following transformation types:
         * </p>
         * <p>
         * <b>CMD_LINE</b>
         * </p>
         * <p>
         * When you're concerned that attackers are injecting an operating system command line command and using unusual
         * formatting to disguise some or all of the command, use this option to perform the following transformations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Delete the following characters: \ " ' ^
         * </p>
         * </li>
         * <li>
         * <p>
         * Delete spaces before the following characters: / (
         * </p>
         * </li>
         * <li>
         * <p>
         * Replace the following characters with a space: , ;
         * </p>
         * </li>
         * <li>
         * <p>
         * Replace multiple spaces with one space
         * </p>
         * </li>
         * <li>
         * <p>
         * Convert uppercase letters (A-Z) to lowercase (a-z)
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>COMPRESS_WHITE_SPACE</b>
         * </p>
         * <p>
         * Use this option to replace the following characters with a space character (decimal 32):
         * </p>
         * <ul>
         * <li>
         * <p>
         * \f, formfeed, decimal 12
         * </p>
         * </li>
         * <li>
         * <p>
         * \t, tab, decimal 9
         * </p>
         * </li>
         * <li>
         * <p>
         * \n, newline, decimal 10
         * </p>
         * </li>
         * <li>
         * <p>
         * \r, carriage return, decimal 13
         * </p>
         * </li>
         * <li>
         * <p>
         * \v, vertical tab, decimal 11
         * </p>
         * </li>
         * <li>
         * <p>
         * non-breaking space, decimal 160
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
         * </p>
         * <p>
         * <b>HTML_ENTITY_DECODE</b>
         * </p>
         * <p>
         * Use this option to replace HTML-encoded characters with unencoded characters. <code>HTML_ENTITY_DECODE</code>
         * performs the following operations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Replaces <code>(ampersand)quot;</code> with <code>"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)lt;</code> with a "less than" symbol
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
         * corresponding characters
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
         * corresponding characters
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>LOWERCASE</b>
         * </p>
         * <p>
         * Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
         * </p>
         * <p>
         * <b>URL_DECODE</b>
         * </p>
         * <p>
         * Use this option to decode a URL-encoded value.
         * </p>
         * <p>
         * <b>NONE</b>
         * </p>
         * <p>
         * Specify <code>NONE</code> if you don't want any text transformations.
         * </p>
         * 
         * @param type
         *        You can specify the following transformation types:</p>
         *        <p>
         *        <b>CMD_LINE</b>
         *        </p>
         *        <p>
         *        When you're concerned that attackers are injecting an operating system command line command and using
         *        unusual formatting to disguise some or all of the command, use this option to perform the following
         *        transformations:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Delete the following characters: \ " ' ^
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Delete spaces before the following characters: / (
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replace the following characters with a space: , ;
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replace multiple spaces with one space
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Convert uppercase letters (A-Z) to lowercase (a-z)
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>COMPRESS_WHITE_SPACE</b>
         *        </p>
         *        <p>
         *        Use this option to replace the following characters with a space character (decimal 32):
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        \f, formfeed, decimal 12
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \t, tab, decimal 9
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \n, newline, decimal 10
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \r, carriage return, decimal 13
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \v, vertical tab, decimal 11
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        non-breaking space, decimal 160
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
         *        </p>
         *        <p>
         *        <b>HTML_ENTITY_DECODE</b>
         *        </p>
         *        <p>
         *        Use this option to replace HTML-encoded characters with unencoded characters.
         *        <code>HTML_ENTITY_DECODE</code> performs the following operations:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)quot;</code> with <code>"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)lt;</code> with a "less than" symbol
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with
         *        the corresponding characters
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
         *        corresponding characters
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>LOWERCASE</b>
         *        </p>
         *        <p>
         *        Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
         *        </p>
         *        <p>
         *        <b>URL_DECODE</b>
         *        </p>
         *        <p>
         *        Use this option to decode a URL-encoded value.
         *        </p>
         *        <p>
         *        <b>NONE</b>
         *        </p>
         *        <p>
         *        Specify <code>NONE</code> if you don't want any text transformations.
         * @see TextTransformationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TextTransformationType
         */
        Builder type(String type);

        /**
         * <p>
         * You can specify the following transformation types:
         * </p>
         * <p>
         * <b>CMD_LINE</b>
         * </p>
         * <p>
         * When you're concerned that attackers are injecting an operating system command line command and using unusual
         * formatting to disguise some or all of the command, use this option to perform the following transformations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Delete the following characters: \ " ' ^
         * </p>
         * </li>
         * <li>
         * <p>
         * Delete spaces before the following characters: / (
         * </p>
         * </li>
         * <li>
         * <p>
         * Replace the following characters with a space: , ;
         * </p>
         * </li>
         * <li>
         * <p>
         * Replace multiple spaces with one space
         * </p>
         * </li>
         * <li>
         * <p>
         * Convert uppercase letters (A-Z) to lowercase (a-z)
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>COMPRESS_WHITE_SPACE</b>
         * </p>
         * <p>
         * Use this option to replace the following characters with a space character (decimal 32):
         * </p>
         * <ul>
         * <li>
         * <p>
         * \f, formfeed, decimal 12
         * </p>
         * </li>
         * <li>
         * <p>
         * \t, tab, decimal 9
         * </p>
         * </li>
         * <li>
         * <p>
         * \n, newline, decimal 10
         * </p>
         * </li>
         * <li>
         * <p>
         * \r, carriage return, decimal 13
         * </p>
         * </li>
         * <li>
         * <p>
         * \v, vertical tab, decimal 11
         * </p>
         * </li>
         * <li>
         * <p>
         * non-breaking space, decimal 160
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
         * </p>
         * <p>
         * <b>HTML_ENTITY_DECODE</b>
         * </p>
         * <p>
         * Use this option to replace HTML-encoded characters with unencoded characters. <code>HTML_ENTITY_DECODE</code>
         * performs the following operations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Replaces <code>(ampersand)quot;</code> with <code>"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)lt;</code> with a "less than" symbol
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with the
         * corresponding characters
         * </p>
         * </li>
         * <li>
         * <p>
         * Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
         * corresponding characters
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>LOWERCASE</b>
         * </p>
         * <p>
         * Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
         * </p>
         * <p>
         * <b>URL_DECODE</b>
         * </p>
         * <p>
         * Use this option to decode a URL-encoded value.
         * </p>
         * <p>
         * <b>NONE</b>
         * </p>
         * <p>
         * Specify <code>NONE</code> if you don't want any text transformations.
         * </p>
         * 
         * @param type
         *        You can specify the following transformation types:</p>
         *        <p>
         *        <b>CMD_LINE</b>
         *        </p>
         *        <p>
         *        When you're concerned that attackers are injecting an operating system command line command and using
         *        unusual formatting to disguise some or all of the command, use this option to perform the following
         *        transformations:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Delete the following characters: \ " ' ^
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Delete spaces before the following characters: / (
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replace the following characters with a space: , ;
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replace multiple spaces with one space
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Convert uppercase letters (A-Z) to lowercase (a-z)
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>COMPRESS_WHITE_SPACE</b>
         *        </p>
         *        <p>
         *        Use this option to replace the following characters with a space character (decimal 32):
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        \f, formfeed, decimal 12
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \t, tab, decimal 9
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \n, newline, decimal 10
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \r, carriage return, decimal 13
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        \v, vertical tab, decimal 11
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        non-breaking space, decimal 160
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>COMPRESS_WHITE_SPACE</code> also replaces multiple spaces with one space.
         *        </p>
         *        <p>
         *        <b>HTML_ENTITY_DECODE</b>
         *        </p>
         *        <p>
         *        Use this option to replace HTML-encoded characters with unencoded characters.
         *        <code>HTML_ENTITY_DECODE</code> performs the following operations:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)quot;</code> with <code>"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)nbsp;</code> with a non-breaking space, decimal 160
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)lt;</code> with a "less than" symbol
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces <code>(ampersand)gt;</code> with <code>&gt;</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces characters that are represented in hexadecimal format, <code>(ampersand)#xhhhh;</code>, with
         *        the corresponding characters
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Replaces characters that are represented in decimal format, <code>(ampersand)#nnnn;</code>, with the
         *        corresponding characters
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>LOWERCASE</b>
         *        </p>
         *        <p>
         *        Use this option to convert uppercase letters (A-Z) to lowercase (a-z).
         *        </p>
         *        <p>
         *        <b>URL_DECODE</b>
         *        </p>
         *        <p>
         *        Use this option to decode a URL-encoded value.
         *        </p>
         *        <p>
         *        <b>NONE</b>
         *        </p>
         *        <p>
         *        Specify <code>NONE</code> if you don't want any text transformations.
         * @see TextTransformationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TextTransformationType
         */
        Builder type(TextTransformationType type);
    }

    static final class BuilderImpl implements Builder {
        private Integer priority;

        private String type;

        private BuilderImpl() {
        }

        private BuilderImpl(TextTransformation model) {
            priority(model.priority);
            type(model.type);
        }

        public final Integer getPriority() {
            return priority;
        }

        @Override
        public final Builder priority(Integer priority) {
            this.priority = priority;
            return this;
        }

        public final void setPriority(Integer priority) {
            this.priority = priority;
        }

        public final String getTypeAsString() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(TextTransformationType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public TextTransformation build() {
            return new TextTransformation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
