/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <note>
 * <p>
 * This is the latest version of <b>AWS WAF</b>, named AWS WAFV2, released in November, 2019. For information, including
 * how to migrate your AWS WAF resources from the prior release, see the <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">AWS WAF Developer Guide</a>.
 * </p>
 * </note>
 * <p>
 * A rate-based rule tracks the rate of requests for each originating IP address, and triggers the rule action when the
 * rate exceeds a limit that you specify on the number of requests in any 5-minute time span. You can use this to put a
 * temporary block on requests from an IP address that is sending excessive requests.
 * </p>
 * <p>
 * When the rule action triggers, AWS WAF blocks additional requests from the IP address until the request rate falls
 * below the limit.
 * </p>
 * <p>
 * You can optionally nest another statement inside the rate-based statement, to narrow the scope of the rule so that it
 * only counts requests that match the nested statement. For example, based on recent requests that you have seen from
 * an attacker, you might create a rate-based rule with a nested AND rule statement that contains the following nested
 * statements:
 * </p>
 * <ul>
 * <li>
 * <p>
 * An IP match statement with an IP set that specified the address 192.0.2.44.
 * </p>
 * </li>
 * <li>
 * <p>
 * A string match statement that searches in the User-Agent header for the string BadBot.
 * </p>
 * </li>
 * </ul>
 * <p>
 * In this rate-based rule, you also define a rate limit. For this example, the rate limit is 1,000. Requests that meet
 * both of the conditions in the statements are counted. If the count exceeds 1,000 requests per five minutes, the rule
 * action triggers. Requests that do not meet both conditions are not counted towards the rate limit and are not
 * affected by this rule.
 * </p>
 * <p>
 * You cannot nest a <code>RateBasedStatement</code>, for example for use inside a <code>NotStatement</code> or
 * <code>OrStatement</code>. It can only be referenced as a top-level statement within a rule.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RateBasedStatement implements SdkPojo, Serializable,
        ToCopyableBuilder<RateBasedStatement.Builder, RateBasedStatement> {
    private static final SdkField<Long> LIMIT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .getter(getter(RateBasedStatement::limit)).setter(setter(Builder::limit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Limit").build()).build();

    private static final SdkField<String> AGGREGATE_KEY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(RateBasedStatement::aggregateKeyTypeAsString)).setter(setter(Builder::aggregateKeyType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AggregateKeyType").build()).build();

    private static final SdkField<Statement> SCOPE_DOWN_STATEMENT_FIELD = SdkField.<Statement> builder(MarshallingType.SDK_POJO)
            .getter(getter(RateBasedStatement::scopeDownStatement)).setter(setter(Builder::scopeDownStatement))
            .constructor(Statement::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScopeDownStatement").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LIMIT_FIELD,
            AGGREGATE_KEY_TYPE_FIELD, SCOPE_DOWN_STATEMENT_FIELD));

    private static final long serialVersionUID = 1L;

    private final Long limit;

    private final String aggregateKeyType;

    private final Statement scopeDownStatement;

    private RateBasedStatement(BuilderImpl builder) {
        this.limit = builder.limit;
        this.aggregateKeyType = builder.aggregateKeyType;
        this.scopeDownStatement = builder.scopeDownStatement;
    }

    /**
     * <p>
     * The limit on requests per 5-minute period for a single originating IP address. If the statement includes a
     * <code>ScopDownStatement</code>, this limit is applied only to the requests that match the statement.
     * </p>
     * 
     * @return The limit on requests per 5-minute period for a single originating IP address. If the statement includes
     *         a <code>ScopDownStatement</code>, this limit is applied only to the requests that match the statement.
     */
    public Long limit() {
        return limit;
    }

    /**
     * <p>
     * Setting that indicates how to aggregate the request counts. Currently, you must set this to <code>IP</code>. The
     * request counts are aggregated on IP addresses.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #aggregateKeyType}
     * will return {@link RateBasedStatementAggregateKeyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #aggregateKeyTypeAsString}.
     * </p>
     * 
     * @return Setting that indicates how to aggregate the request counts. Currently, you must set this to
     *         <code>IP</code>. The request counts are aggregated on IP addresses.
     * @see RateBasedStatementAggregateKeyType
     */
    public RateBasedStatementAggregateKeyType aggregateKeyType() {
        return RateBasedStatementAggregateKeyType.fromValue(aggregateKeyType);
    }

    /**
     * <p>
     * Setting that indicates how to aggregate the request counts. Currently, you must set this to <code>IP</code>. The
     * request counts are aggregated on IP addresses.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #aggregateKeyType}
     * will return {@link RateBasedStatementAggregateKeyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #aggregateKeyTypeAsString}.
     * </p>
     * 
     * @return Setting that indicates how to aggregate the request counts. Currently, you must set this to
     *         <code>IP</code>. The request counts are aggregated on IP addresses.
     * @see RateBasedStatementAggregateKeyType
     */
    public String aggregateKeyTypeAsString() {
        return aggregateKeyType;
    }

    /**
     * <p>
     * An optional nested statement that narrows the scope of the rate-based statement to matching web requests. This
     * can be any nestable statement, and you can nest statements at any level below this scope-down statement.
     * </p>
     * 
     * @return An optional nested statement that narrows the scope of the rate-based statement to matching web requests.
     *         This can be any nestable statement, and you can nest statements at any level below this scope-down
     *         statement.
     */
    public Statement scopeDownStatement() {
        return scopeDownStatement;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(limit());
        hashCode = 31 * hashCode + Objects.hashCode(aggregateKeyTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(scopeDownStatement());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RateBasedStatement)) {
            return false;
        }
        RateBasedStatement other = (RateBasedStatement) obj;
        return Objects.equals(limit(), other.limit())
                && Objects.equals(aggregateKeyTypeAsString(), other.aggregateKeyTypeAsString())
                && Objects.equals(scopeDownStatement(), other.scopeDownStatement());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("RateBasedStatement").add("Limit", limit()).add("AggregateKeyType", aggregateKeyTypeAsString())
                .add("ScopeDownStatement", scopeDownStatement()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Limit":
            return Optional.ofNullable(clazz.cast(limit()));
        case "AggregateKeyType":
            return Optional.ofNullable(clazz.cast(aggregateKeyTypeAsString()));
        case "ScopeDownStatement":
            return Optional.ofNullable(clazz.cast(scopeDownStatement()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RateBasedStatement, T> g) {
        return obj -> g.apply((RateBasedStatement) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RateBasedStatement> {
        /**
         * <p>
         * The limit on requests per 5-minute period for a single originating IP address. If the statement includes a
         * <code>ScopDownStatement</code>, this limit is applied only to the requests that match the statement.
         * </p>
         * 
         * @param limit
         *        The limit on requests per 5-minute period for a single originating IP address. If the statement
         *        includes a <code>ScopDownStatement</code>, this limit is applied only to the requests that match the
         *        statement.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder limit(Long limit);

        /**
         * <p>
         * Setting that indicates how to aggregate the request counts. Currently, you must set this to <code>IP</code>.
         * The request counts are aggregated on IP addresses.
         * </p>
         * 
         * @param aggregateKeyType
         *        Setting that indicates how to aggregate the request counts. Currently, you must set this to
         *        <code>IP</code>. The request counts are aggregated on IP addresses.
         * @see RateBasedStatementAggregateKeyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RateBasedStatementAggregateKeyType
         */
        Builder aggregateKeyType(String aggregateKeyType);

        /**
         * <p>
         * Setting that indicates how to aggregate the request counts. Currently, you must set this to <code>IP</code>.
         * The request counts are aggregated on IP addresses.
         * </p>
         * 
         * @param aggregateKeyType
         *        Setting that indicates how to aggregate the request counts. Currently, you must set this to
         *        <code>IP</code>. The request counts are aggregated on IP addresses.
         * @see RateBasedStatementAggregateKeyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RateBasedStatementAggregateKeyType
         */
        Builder aggregateKeyType(RateBasedStatementAggregateKeyType aggregateKeyType);

        /**
         * <p>
         * An optional nested statement that narrows the scope of the rate-based statement to matching web requests.
         * This can be any nestable statement, and you can nest statements at any level below this scope-down statement.
         * </p>
         * 
         * @param scopeDownStatement
         *        An optional nested statement that narrows the scope of the rate-based statement to matching web
         *        requests. This can be any nestable statement, and you can nest statements at any level below this
         *        scope-down statement.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scopeDownStatement(Statement scopeDownStatement);

        /**
         * <p>
         * An optional nested statement that narrows the scope of the rate-based statement to matching web requests.
         * This can be any nestable statement, and you can nest statements at any level below this scope-down statement.
         * </p>
         * This is a convenience that creates an instance of the {@link Statement.Builder} avoiding the need to create
         * one manually via {@link Statement#builder()}.
         *
         * When the {@link Consumer} completes, {@link Statement.Builder#build()} is called immediately and its result
         * is passed to {@link #scopeDownStatement(Statement)}.
         * 
         * @param scopeDownStatement
         *        a consumer that will call methods on {@link Statement.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scopeDownStatement(Statement)
         */
        default Builder scopeDownStatement(Consumer<Statement.Builder> scopeDownStatement) {
            return scopeDownStatement(Statement.builder().applyMutation(scopeDownStatement).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Long limit;

        private String aggregateKeyType;

        private Statement scopeDownStatement;

        private BuilderImpl() {
        }

        private BuilderImpl(RateBasedStatement model) {
            limit(model.limit);
            aggregateKeyType(model.aggregateKeyType);
            scopeDownStatement(model.scopeDownStatement);
        }

        public final Long getLimit() {
            return limit;
        }

        @Override
        public final Builder limit(Long limit) {
            this.limit = limit;
            return this;
        }

        public final void setLimit(Long limit) {
            this.limit = limit;
        }

        public final String getAggregateKeyTypeAsString() {
            return aggregateKeyType;
        }

        @Override
        public final Builder aggregateKeyType(String aggregateKeyType) {
            this.aggregateKeyType = aggregateKeyType;
            return this;
        }

        @Override
        public final Builder aggregateKeyType(RateBasedStatementAggregateKeyType aggregateKeyType) {
            this.aggregateKeyType(aggregateKeyType == null ? null : aggregateKeyType.toString());
            return this;
        }

        public final void setAggregateKeyType(String aggregateKeyType) {
            this.aggregateKeyType = aggregateKeyType;
        }

        public final Statement.Builder getScopeDownStatement() {
            return scopeDownStatement != null ? scopeDownStatement.toBuilder() : null;
        }

        @Override
        public final Builder scopeDownStatement(Statement scopeDownStatement) {
            this.scopeDownStatement = scopeDownStatement;
            return this;
        }

        public final void setScopeDownStatement(Statement.BuilderImpl scopeDownStatement) {
            this.scopeDownStatement = scopeDownStatement != null ? scopeDownStatement.build() : null;
        }

        @Override
        public RateBasedStatement build() {
            return new RateBasedStatement(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
